# Copyright (C) 2013-2014, 2017-2018, 2024  Luke T. Shumaker <lukeshu@parabola.nu>
# Copyright (C) 2017  Isaac David <isacdaavid@isacdaavid.info>
#
# SPDX-License-Identifier: GPL-2.0-or-later
#
# This file is part of Parabola Libretools.
#
# Libretools is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# Libretools is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

load ../lib/common

set -o pipefail

@test "libreblacklist works with just pkgname" {
	libreblacklist normalize <<<skype | equals $'skype::::\n'
	libreblacklist get-pkg <<<skype | equals $'skype\n'
	libreblacklist get-rep <<<irreplaceable | equals $'\n'
	libreblacklist get-url <<<skype | equals $'\n'
	libreblacklist get-reason <<<skype | equals $'\n'
}

@test "libreblacklist works with everything set" {
	line='linux:conflict:parabola:id:[semifree] blobs and firmware'
	libreblacklist normalize <<<"$line" | equals "$line"$'\n'
	libreblacklist get-pkg <<<"$line" | equals $'linux\n'
	libreblacklist get-rep <<<"$line" | equals $'libre/linux-libre\n'
	libreblacklist get-url <<<"$line" | equals $'https://labs.parabola.nu/issues/id\n'
	libreblacklist get-reason <<<"$line" | equals $'[semifree] blobs and firmware\n'
}

@test "libreblacklist normalizes correctly" {
	libreblacklist normalize <<<'#comment' | equals ''
	libreblacklist normalize <<<pkg | equals $'pkg::::\n'
	libreblacklist normalize <<<pkg: | equals $'pkg::::\n'
	libreblacklist normalize <<<pkg:: | equals $'pkg::::\n'
	libreblacklist normalize <<<pkg:rep | equals $'pkg:rep:::\n'
	libreblacklist normalize <<<pkg:rep: | equals $'pkg:rep:::\n'
	libreblacklist normalize <<<pkg:rep:ref | equals $'pkg:rep:ref::\n'
	libreblacklist normalize <<<pkg:rep:ref: | equals $'pkg:rep:ref::\n'
	libreblacklist normalize <<<pkg:rep:ref:id | equals $'pkg:rep:ref:id:\n'
	libreblacklist normalize <<<pkg:rep:ref:id: | equals $'pkg:rep:ref:id:\n'
	libreblacklist normalize <<<pkg:rep:ref:id:reason | equals $'pkg:rep:ref:id:reason\n'
}

@test "libreblacklist works with colons in reason" {
	line='package:conflict:parabola:id:my:reason'
	libreblacklist normalize <<<"$line" | equals "$line"$'\n'
	libreblacklist get-pkg <<<"$line" | equals $'package\n'
	libreblacklist get-rep <<<"$line" | equals $'\n'
	libreblacklist get-url <<<"$line" | equals $'https://labs.parabola.nu/issues/id\n'
	libreblacklist get-reason <<<"$line" | equals $'my:reason\n'
}

@test "libreblacklist prints urls only for valid references" {
	libreblacklist get-url <<<package:::id: | equals $'\n'
	libreblacklist get-url <<<package::unknown:id: | equals $'\n'
}

@test "libreblacklist fails update with no blacklist or network" {
	cat >>"$XDG_CONFIG_HOME/libretools/libretools.conf" <<-eot
		BLACKLIST='phony://example.com'
	eot

	libreblacklist update >$tmpdir/stdout 2>$tmpdir/stderr || status=$?

	[[ $status != 0 ]]
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
}

@test "libreblacklist fails cat with no blacklist or network" {
	cat >>"$XDG_CONFIG_HOME/libretools/libretools.conf" <<-eot
		BLACKLIST='phony://example.com'
	eot

	libreblacklist cat >$tmpdir/stdout 2>$tmpdir/stderr || status=$?

	[[ $status != 0 ]]
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
}

@test "libreblacklist fails update when BLACKLIST is unset" {
	cat >>"$XDG_CONFIG_HOME/libretools/libretools.conf" <<-eot
		BLACKLIST=
	eot

	libreblacklist update >$tmpdir/stdout 2>$tmpdir/stderr || status=$?

	[[ $status != 0 ]]
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
}

@test "libreblacklist fails cat when syntax error in conf" {
	# there is a stray single quote in there
	cat >>"$XDG_CONFIG_HOME/libretools/libretools.conf" <<-eot
		BLACKLIST='https://git.parabola.nu/blacklist.git/plain/blacklist.txt
	eot

	libreblacklist cat >$tmpdir/stdout 2>$tmpdir/stderr || status=$?

	[[ $status != 0 ]]
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
}

# bats test_tags=network
@test "libreblacklist downloads the blacklist as needed" {
	libreblacklist cat >$tmpdir/stdout 2>$tmpdir/stderr

	assert_file_not_empty $tmpdir/stdout
}

# bats test_tags=network
@test "libreblacklist downloads the blacklist repeatedly" {
	libreblacklist update
	libreblacklist update
}

# bats test_tags=locale-es
@test "libreblacklist follows cli conventions and fails with no args" {
	assert_follows_cli_conventions libreblacklist
	assert_fails_with_errusage libreblacklist
}

@test "libreblacklist displays help when given -h cat" {
	LC_ALL=C libreblacklist -h cat >$tmpdir/stdout 2>$tmpdir/stderr

	[[ "$(sed 1q $tmpdir/stdout)" == 'Usage: libreblacklist cat' ]]
	assert_file_empty $tmpdir/stderr
}

@test "libreblacklist checks for replacements from the repos" {
	libreblacklist get-rep <<<'linux' >$tmpdir/stdout 2>$tmpdir/stderr
	[[ "$(cat $tmpdir/stdout)" == 'libre/linux-libre' ]]
	assert_file_empty $tmpdir/stderr
}
