# Copyright (C) 2013-2014, 2016-2018, 2024  Luke T. Shumaker <lukeshu@parabola.nu>
#
# SPDX-License-Identifier: GPL-2.0-or-later
#
# This file is part of Parabola Libretools.
#
# Libretools is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# Libretools is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

load ../lib/common

# bats test_tags=network,sudo
@test "librechroot creates repo for new chroots" {
	testsudo librechroot -l "$BATS_TEST_NAME" run test -r /repo/repo.db
}

# bats test_tags=network,sudo
@test "librechroot cleans the local repo correctly" {
	testsudo librechroot -l "$BATS_TEST_NAME" make
	testsudo librechroot -l "$BATS_TEST_NAME" clean-repo
	testsudo librechroot -l "$BATS_TEST_NAME" run test -r /repo/repo.db
	# TODO: inspect /repo/* more
}

# bats test_tags=network,sudo
@test "librechroot respects exit status if out isnt a tty" {
	set -o pipefail
	{ testsudo librechroot -l "$BATS_TEST_NAME" run bash -c 'exit 3' | cat; } || status=$?

	[[ $status == 3 ]]
}

# bats test_tags=network,sudo
@test "librechroot creates ca certificates" {
	testsudo librechroot -l "$BATS_TEST_NAME" run test -r /etc/ssl/certs/ca-certificates.crt
}

# bats test_tags=network,sudo
@test "librechroot disables networking when requested" {
	assert testsudo librechroot -l "$BATS_TEST_NAME" run curl https://repo.parabola.nu/ >/dev/null
	refute testsudo librechroot -l "$BATS_TEST_NAME" -N run curl https://repo.parabola.nu/ >/dev/null
}

# bats test_tags=network,sudo
@test "librechroot clean-pkgs obeys CHROOTEXTRAPKG" {
	refute testsudo librechroot -l "$BATS_TEST_NAME" run lsof
	echo "CHROOTEXTRAPKG=(lsof)" >>"$XDG_CONFIG_HOME"/libretools/chroot.conf
	testsudo librechroot -l "$BATS_TEST_NAME" install-name lsof
	testsudo librechroot -l "$BATS_TEST_NAME" clean-pkgs
	testsudo librechroot -l "$BATS_TEST_NAME" run lsof
	echo "CHROOTEXTRAPKG=()" >>"$XDG_CONFIG_HOME"/libretools/chroot.conf
	testsudo librechroot -l "$BATS_TEST_NAME" clean-pkgs
	refute testsudo librechroot -l "$BATS_TEST_NAME" run lsof
}

# bats test_tags=network,sudo
@test "librechroot clean-pkgs doesn't remove locally-updated parts of base-devel" {
	local copydir="$chrootdir/default/$BATS_TEST_NAME"

	# Get info on the upstream package.
	local oldpkgname=libre/pacman
	testsudo librechroot -l "$BATS_TEST_NAME" run sh -c "pacman -Sp --print-format='%v %D' ${oldpkgname} >/info.txt"
	local oldpkgver pkgdeps
	read -r oldpkgver pkgdeps <"$copydir/info.txt"
	read -r -a pkgdeps <<<"$pkgdeps"

	# Build our locally-updated version.
	local newpkgname="repo/${oldpkgname##*/}"
	local newpkgver="${oldpkgver}.1"
	mkdir "$tmpdir/workdir/abs"
	sed \
		-e "s/@PKGVER@/${newpkgver%%-*}/g" \
		-e "s/@PKGREL@/${newpkgver#*-}/g" \
		-e "s/@PKGDEPS@/${pkgdeps[*]@Q}/g" \
		<fixtures/librechroot/PKGBUILD-base-devel \
		>"$tmpdir/workdir/abs/PKGBUILD"
	pushd "$tmpdir/workdir/abs"
	testsudo libremakepkg -l "$BATS_TEST_NAME"
	popd
	testsudo librechroot -l "$BATS_TEST_NAME" run pacman -Sy

	# Verify that the package got added to [repo]...
	echo "${newpkgname##*/} $newpkgver" >"$tmpdir/exp.txt"
	testsudo librechroot -l "$BATS_TEST_NAME" run sh -c "pacman -Sp --print-format='%n %v' ${newpkgname} >/act.txt"
	diff -u "$tmpdir/exp.txt" "$copydir/act.txt"
	# ... but not installed.
	echo "${oldpkgname##*/} $oldpkgver" >"$tmpdir/exp.txt"
	testsudo librechroot -l "$BATS_TEST_NAME" run sh -c "pacman -Q ${oldpkgname##*/} >/act.txt"
	diff -u "$tmpdir/exp.txt" "$copydir/act.txt"

	# Run the code-under-test.
	testsudo librechroot -l "$BATS_TEST_NAME" clean-pkgs

	# Validate that clean-pkgs didn't change anything.
	#
	# In [repo]...
	echo "${oldpkgname##*/} $oldpkgver" >"$tmpdir/exp.txt"
	testsudo librechroot -l "$BATS_TEST_NAME" run sh -c "pacman -Q ${oldpkgname##*/} >/act.txt"
	diff -u "$tmpdir/exp.txt" "$copydir/act.txt"
	# ... but not installed.
	echo "${newpkgname##*/} $newpkgver" >"$tmpdir/exp.txt"
	testsudo librechroot -l "$BATS_TEST_NAME" run sh -c "pacman -Sp --print-format='%n %v' ${newpkgname} >/act.txt"
	diff -u "$tmpdir/exp.txt" "$copydir/act.txt"
}

# bats test_tags=network,sudo
@test "librechroot clean-pkgs obeys PKGBUILD depends" {
	# NB: We chown `/startdir` because chcleanup runs `makepkg
	# --printsrcinfo > .SRCINFO` as the user that owns
	# `/startdir/PKGBUILD`, and `makepkg` doesn't like to be run
	# as root.  Unfortunately, this test runs with a chroot that
	# doesn't have the `builduser` set up, and it'd be a pain to
	# set it up.
	local builduser_uid="${SUDO_UID:-$UID}"
	local builduser_gid="$(id -g "$builduser_uid")"
	testsudo librechroot -l "$BATS_TEST_NAME" run sh -c "
		echo 'builduser:x:${builduser_gid}:'                                            >>/etc/group
		echo 'builduser:x:${builduser_uid}:${builduser_gid}:builduser:/build:/bin/bash' >>/etc/passwd
		echo 'builduser:!!:$(($(date -u +%s) / 86400))::::::'                           >>/etc/shadow"

	# Pre-flight: Make sure `lsof` isn't in the base install
	refute testsudo librechroot -l "$BATS_TEST_NAME" run lsof
	# Make sure librechroot removes `lsof` when the PKGBUILD is not present.
	testsudo librechroot -l "$BATS_TEST_NAME" install-name lsof
	testsudo librechroot -l "$BATS_TEST_NAME" run lsof
	testsudo librechroot -l "$BATS_TEST_NAME" clean-pkgs
	refute testsudo librechroot -l "$BATS_TEST_NAME" run lsof
	# Make sure librechroot keeps `lsof` when the PKGBUILD is present.
	testsudo librechroot -l "$BATS_TEST_NAME" install-name lsof
	testsudo librechroot -l "$BATS_TEST_NAME" run sh -c 'printf "%s\n" "$1" >/startdir/PKGBUILD' -- "$(cat fixtures/librechroot/PKGBUILD-arches)"
	testsudo librechroot -l "$BATS_TEST_NAME" run chown -R builduser: /startdir
	testsudo librechroot -l "$BATS_TEST_NAME" clean-pkgs
	testsudo librechroot -l "$BATS_TEST_NAME" run lsof
}

# bats test_tags=network,sudo
@test "librechroot clean-pkgs can install libretools" {
	# NB: We chown `/startdir` because chcleanup runs `makepkg
	# --printsrcinfo > .SRCINFO` as the user that owns
	# `/startdir/PKGBUILD`, and `makepkg` doesn't like to be run
	# as root.
	local builduser_uid="${SUDO_UID:-$UID}"
	local builduser_gid="$(id -g "$builduser_uid")"
	testsudo librechroot -l "$BATS_TEST_NAME" run sh -c "
		echo 'builduser:x:${builduser_gid}:'                                            >>/etc/group
		echo 'builduser:x:${builduser_uid}:${builduser_gid}:builduser:/build:/bin/bash' >>/etc/passwd
		echo 'builduser:!!:$(($(date -u +%s) / 86400))::::::'                           >>/etc/shadow"

	refute testsudo librechroot -l "$BATS_TEST_NAME" run test -f /usr/bin/chcleanup
	testsudo librechroot -l "$BATS_TEST_NAME" run mkdir /startdir
	testsudo librechroot -l "$BATS_TEST_NAME" run sh -c 'printf "%s\n" "$1" >/startdir/PKGBUILD' -- "$(cat fixtures/librechroot/PKGBUILD-libretools)"
	testsudo librechroot -l "$BATS_TEST_NAME" run chown -R builduser: /startdir
	testsudo librechroot -l "$BATS_TEST_NAME" clean-pkgs
	testsudo librechroot -l "$BATS_TEST_NAME" run test -f /usr/bin/chcleanup
}

@test "librechroot displays help as normal user" {
	rm -rf "$XDG_CONFIG_HOME"
	assert_shows_usage librechroot help
}

@test "librechroot otherwise fails as normal user" {
	librechroot -l "$BATS_TEST_NAME" run true >$tmpdir/stdout 2>$tmpdir/stderr || status=$?

	[[ $status != 0 ]]
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
}

# bats test_tags=locale-es
@test "librechroot follows cli conventions and fails with 0 args" {
	assert_follows_cli_conventions librechroot
	assert_fails_with_errusage librechroot
	assert_fails_with_errusage librechroot unknown-command
	assert_fails_with_errusage librechroot --unknown-flag
	assert_fails_with_errusage librechroot -u
}

# bats test_tags=network,sudo
@test "librechroot obeys the n flag" {
	refute test -f "$chrootdir/$BATS_TEST_NAME/$BATS_TEST_NAME/$BATS_TEST_NAME.stamp"

	testsudo librechroot -n "$BATS_TEST_NAME" -l "$BATS_TEST_NAME" run touch /"$BATS_TEST_NAME.stamp"

	test -f "$chrootdir/$BATS_TEST_NAME/$BATS_TEST_NAME/$BATS_TEST_NAME.stamp"
}

# bats test_tags=sudo
@test "librechroot fails when syncing a copy with itself" {
	testsudo timeout 5 librechroot -l root sync || status=$?
	case $status in
		0 | 124 | 137) # success|timeout+TERM|timeout+KILL
			false ;;
		*)
			true
			;;
	esac
}

# bats test_tags=network,sudo
@test "librechroot deletes copies" {
	testsudo librechroot -l "$BATS_TEST_NAME" make
	test -d "$chrootdir/default/$BATS_TEST_NAME"
	testsudo librechroot -l "$BATS_TEST_NAME" delete
	refute test -e "$chrootdir/default/$BATS_TEST_NAME"
}

# bats test_tags=network,sudo,btrfs
@test "librechroot deletes subvolumes recursively" {
	[[ "$(stat -f -c %T "$chrootdir" 2>/dev/null || true)" == 'btrfs' ]]
	testsudo librechroot -l "$BATS_TEST_NAME" make
	testsudo librechroot -l "$BATS_TEST_NAME" install-name btrfs-progs
	test -d "$chrootdir/default/$BATS_TEST_NAME"
	refute test -e "$chrootdir/default/$BATS_TEST_NAME/var/subvolume"
	testsudo librechroot -l "$BATS_TEST_NAME" run btrfs subvolume create /var/subvolume
	test -d "$chrootdir/default/$BATS_TEST_NAME/var/subvolume"
	testsudo librechroot -l "$BATS_TEST_NAME" delete
	refute test -e "$chrootdir/default/$BATS_TEST_NAME"
}

# bats test_tags=network,sudo
@test "librechroot cleans up TMPDIR" {
	local dir="$tmpdir/tmp"
	mkdir -- "$dir"

	TMPDIR=$dir testsudo librechroot -l "$BATS_TEST_NAME" -A x86_64 make

	# Make sure $dir is now empty
	rmdir -- "$dir"
}
