# Copyright (C) 2013-2015, 2017-2018, 2024, 2026  Luke T. Shumaker <lukeshu@parabola.nu>
#
# SPDX-License-Identifier: GPL-2.0-or-later
#
# This file is part of Parabola Libretools.
#
# Libretools is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# Libretools is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

load ../lib/common

setup() {
	libretools_common_per_test_setup

	cat >>"$XDG_CONFIG_HOME/pacman/makepkg.conf" <<-eot
		DLAGENTS=('ftp::/usr/bin/curl -fC - --ftp-pasv --retry 3 --retry-delay 3 -o %o %u'
		          'http::/usr/bin/curl -fLC - --retry 3 --retry-delay 3 -o %o %u'
		          'https::/usr/bin/curl -fLC - --retry 3 --retry-delay 3 -o %o %u'
		          'rsync::/usr/bin/rsync --no-motd -z %u %o'
		          'scp::/usr/bin/scp -C %u %o')
		BUILDDIR=""
		source ${_librelib_conf_sh_sysconfdir@Q}/makepkg.conf.d/librefetch.conf
	eot

	install -Dm644 /dev/stdin "$XDG_CONFIG_HOME/libretools/librefetch.conf" <<-eot
		MIRRORS=("phony://example.com/dir/")
		DOWNLOADER=/usr/bin/false
	eot
}

# bats test_tags=locale-es
@test "librefetch follows cli conventions" {
	assert_follows_cli_conventions librefetch
	assert_shows_usage librefetch --help -C
	assert_shows_usage librefetch --help -D
	assert_shows_usage librefetch --help -g
	assert_shows_usage librefetch --help -S
	assert_shows_usage librefetch --help -M
	assert_shows_usage librefetch -C --help
	assert_shows_usage librefetch -D --help
	assert_shows_usage librefetch -g --help
	assert_shows_usage librefetch -S --help
	assert_shows_usage librefetch -M --help
}

@test "librefetch fails with bad flags" {
	local srcball=testpkg-1.0.tar.gz
	cp fixtures/librefetch/* "$tmpdir/"
	cd "$tmpdir"
	mv PKGBUILD{-mksource,}

	assert_fails_with_errusage librefetch --bogus-flag libre://"$srcball"
	refute test -e $tmpdir/workdir/srcdest/$srcball
}

# This test used to be called "librefetch cleans src-libre first", but let's
# be honest: it checks pretty much everything related to normal
# operation.
@test "librefetch runs with mksource" {
	local srcball=testpkg-1.0.tar.gz
	cp fixtures/librefetch/* "$tmpdir/"
	cd "$tmpdir"
	mv PKGBUILD{-mksource,}

	# Create garbage, to verifiy that it cleans src-libre first
	mkdir -p src-libre/foo
	touch src-libre/foo/file

	# Run librefetch
	makepkg -g >&log.txt

	# Verify that no temporary files were left around
	refute test -e librefetch.*

	# Verify that there were no warnings about missing backup=()
	# files
	refute grep -F etc/testpkg.conf log.txt

	# Verify:
	# - The srcball was created...
	# - ... and is in the correct directory
	# - The srcball does not contain the garbage created earlier
	# - The files in the srcball are in the correct order (if the
	#   order isn't ensured, then this would only sometimes fail,
	#   unfortunately).
	bsdtar tf "$tmpdir/workdir/srcdest/$srcball" >list-pkg.txt
	diff -u list.txt list-pkg.txt
	# Verify that the signature was created and matches
	gpg --quiet --verify "$tmpdir/workdir/srcdest/$srcball"{.sig,} 2>/dev/null

	assert_file_exists "$tmpdir/workdir/logdest/${srcball%.tar.gz}-mksource-prepare.log"
	assert_file_not_empty "$tmpdir/workdir/logdest/${srcball%.tar.gz}-mksource-build.log"
	assert_file_exists "$tmpdir/workdir/logdest/${srcball%.tar.gz}-mksource-check.log"
	assert_file_exists "$tmpdir/workdir/logdest/${srcball%.tar.gz}-mksource-package.log"
}

@test "librefetch runs with srcbuild" {
	local srcball=testpkg-1.0.tar.gz
	cp fixtures/librefetch/* "$tmpdir/"
	cd "$tmpdir"
	mv PKGBUILD{-srcbuild,}
	mv SRCBUILD{-srcbuild,}

	# Create garbage, to verifiy that it cleans src-libre first
	mkdir -p src-libre/foo
	touch src-libre/foo/file

	# Run librefetch
	makepkg -g

	# Verify that no temporary files were left around
	refute test -e librefetch.*

	# Verify:
	# - The srcball was created...
	# - ... and is in the correct directory
	# - The srcball does not contain the garbage created earlier
	# - The files in the srcball are in the correct order (if the
	#   order isn't ensured, then this would only sometimes fail,
	#   unfortunately).
	bsdtar tf "$tmpdir/workdir/srcdest/$srcball" >list-pkg.txt
	diff -u list.txt list-pkg.txt
	# Verify that the signature was created and matches
	gpg --quiet --verify "$tmpdir/workdir/srcdest/$srcball"{.sig,} 2>/dev/null
}

@test "librefetch recurses" {
	local srcball=testpkg-1.0.tar.gz
	cp fixtures/librefetch/* "$tmpdir/"
	cd "$tmpdir"
	mv PKGBUILD{-recurse,}

	makepkg -g
	bsdtar tf "$tmpdir/workdir/srcdest/$srcball" >list-pkg.txt
	diff -u list.txt list-pkg.txt
	gpg --quiet --verify "$tmpdir/workdir/srcdest/$srcball"{.sig,} 2>/dev/null
}

@test "librefetch doesnt recurse extra" {
	local srcball=testpkg-1.0.tar.gz
	cp fixtures/librefetch/* "$tmpdir/"
	cd "$tmpdir"
	mv PKGBUILD{-recurse,}

	: >"$tmpdir/workdir/srcdest/$srcball"
	makepkg -g
	assert_file_empty "$tmpdir/workdir/srcdest/$srcball"
	gpg --quiet --verify "$tmpdir/workdir/srcdest/$srcball"{.sig,} 2>/dev/null
}

@test "librefetch handles split packages" {
	cp fixtures/librefetch/* "$tmpdir/"
	cd "$tmpdir"
	mv PKGBUILD{-split,}

	makepkg -g
}

@test "librefetch uses the specified compression" {
	cp fixtures/librefetch/* "$tmpdir/"
	cd "$tmpdir"
	local srcball

	# Both gzip and bzip2 are in 'base', so this choice of
	# compression methods doesn't add anything to our list of
	# dependencies.

	srcball=testpkg-1.0.tar.gz
	mv PKGBUILD{-mksource,}
	makepkg -g
	gzip -d <"$tmpdir/workdir/srcdest/$srcball" >/dev/null

	srcball=testpkg-bz2-1.0.tar.bz2
	mv PKGBUILD{-bz2,}
	makepkg -g
	bzip2 -d <"$tmpdir/workdir/srcdest/$srcball" >/dev/null
}
