# Copyright (C) 2013-2015, 2017-2018, 2024  Luke T. Shumaker <lukeshu@parabola.nu>
#
# SPDX-License-Identifier: GPL-2.0-or-later
#
# This file is part of Parabola Libretools.
#
# Libretools is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# Libretools is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

load ../lib/common

shopt -s extglob

setup() {
	libretools_common_per_test_setup

	cat >>"$XDG_CONFIG_HOME/pacman/makepkg.conf" <<-eot
		unset PKGDEST SRCPKGDEST
	eot
}

teardown() {
	local file
	for file in "$tmpdir"/*.pid; do
		[[ -f $file ]] || continue
		xargs -a "$file" kill --
	done

	libretools_common_per_test_teardown
}

tail-and-fail() {
	status=$?
	tail "$1" | cat -v
	return $status
}

# bats test_tags=network,sudo
@test "libremakepkg builds a trivial package" {
	cp fixtures/libremakepkg/PKGBUILD-hello "$tmpdir/PKGBUILD"
	cd "$tmpdir"

	testsudo libremakepkg -l "$BATS_TEST_NAME"

	globfile libretools-hello-1.0-1-any.pkg.tar?(.!(sig|*.*))
}

# bats test_tags=network,sudo
@test "libremakepkg disables networking during prepare" {
	cp fixtures/libremakepkg/PKGBUILD-netprepare "$tmpdir/PKGBUILD"
	cd "$tmpdir"

	refute testsudo libremakepkg -l "$BATS_TEST_NAME"
	refute globfile libretools-netprepare-1.0-1-any.pkg.tar?(.!(sig|*.*))
	testsudo libremakepkg -l "$BATS_TEST_NAME" -N
	globfile libretools-netprepare-1.0-1-any.pkg.tar?(.!(sig|*.*))
}

# bats test_tags=network,sudo
@test "libremakepkg disables networking during build" {
	cp fixtures/libremakepkg/PKGBUILD-netbuild "$tmpdir/PKGBUILD"
	cd "$tmpdir"

	refute testsudo libremakepkg -l "$BATS_TEST_NAME"
	refute globfile libretools-netbuild-1.0-1-any.pkg.tar?(.!(sig|*.*))
	testsudo libremakepkg -l "$BATS_TEST_NAME" -N
	globfile libretools-netbuild-1.0-1-any.pkg.tar?(.!(sig|*.*))
}

# bats test_tags=network,sudo
@test "libremakepkg disables networking during package" {
	cp fixtures/libremakepkg/PKGBUILD-netpackage "$tmpdir/PKGBUILD"
	cd "$tmpdir"

	refute testsudo libremakepkg -l "$BATS_TEST_NAME"
	refute globfile libretools-netpackage-1.0-1-any.pkg.tar?(.!(sig|*.*))
	testsudo libremakepkg -l "$BATS_TEST_NAME" -N
	globfile libretools-netpackage-1.0-1-any.pkg.tar?(.!(sig|*.*))
}

# bats test_tags=network,sudo
@test "libremakepkg cleans the chroot before building" {
	# 1. First, we build testpkg1
	# 2. Then, we build testpkg2, which depends on testpkg1
	#    Therefore, testpkg1 will be installed after testpkg2 is built, we
	#    check for that.
	# 3. Then, we build hello, which depends on neither, so testpkg1 should
	#    be removed.

	# Also, do funny things with the output of libremakepkg to get a helpful
	# fail case.

	mkdir -p "$tmpdir"/{1,2,3}
	cp fixtures/libremakepkg/PKGBUILD-testpkg1 "$tmpdir/1/PKGBUILD"
	cp fixtures/libremakepkg/PKGBUILD-testpkg2 "$tmpdir/2/PKGBUILD"
	cp fixtures/libremakepkg/PKGBUILD-hello "$tmpdir/3/PKGBUILD"

	cd "$tmpdir/1"
	testsudo libremakepkg -l "$BATS_TEST_NAME" &>"$tmpdir/out" || tail-and-fail "$tmpdir/out"

	cd "$tmpdir/2"
	testsudo libremakepkg -l "$BATS_TEST_NAME" &>"$tmpdir/out" || tail-and-fail "$tmpdir/out"
	testsudo librechroot -l "$BATS_TEST_NAME" run libretools-testpkg1 'first time, pass'

	# This next line is actually a separate test, but it fits in
	# well with this test, and chroot tests are slow.
	# https://labs.parabola.nu/issues/435
	#
	# @test "libremakepkg doesnt cache local packages" {
	refute testsudo librechroot -l "$BATS_TEST_NAME" run bash -O extglob -c 'test -e /var/cache/pacman/pkg/libretools-testpkg1-1.0-1-any.pkg.tar?(.!(sig|*.*))'

	cd "$tmpdir/3"
	testsudo libremakepkg -l "$BATS_TEST_NAME" &>"$tmpdir/out" || tail-and-fail "$tmpdir/out"
	refute testsudo librechroot -l "$BATS_TEST_NAME" run libretools-testpkg1 'second time, fail'
}

# bats test_tags=network,sudo
@test "libremakepkg handles PKGDEST not existing" {
	cp fixtures/libremakepkg/PKGBUILD-hello "$tmpdir/PKGBUILD"
	cd "$tmpdir"

	testsudo env PKGDEST="$tmpdir/dest/pkgdest" libremakepkg -l "$BATS_TEST_NAME"

	globfile dest/pkgdest/libretools-hello-1.0-1-any.pkg.tar?(.!(sig|*.*))
}

@test "libremakepkg displays help as normal user" {
	rm -rf "$XDG_CONFIG_HOME"
	assert_shows_usage libremakepkg -h
}

@test "libremakepkg otherwise fails as normal user" {
	# I do this to give it a chance of passing
	cp fixtures/libremakepkg/PKGBUILD-hello "$tmpdir/PKGBUILD"
	cd "$tmpdir"

	libremakepkg >$tmpdir/stdout 2>$tmpdir/stderr || status=$?

	[[ $status != 0 ]]
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
}

# bats test_tags=locale-es
@test "libremakepkg follows cli conventions" {
	assert_follows_cli_conventions libremakepkg
}

# bats test_tags=network,sudo
@test "libremakepkg fails if a hook fails" {
	cp fixtures/libremakepkg/PKGBUILD-hello "$tmpdir/PKGBUILD"
	cd "$tmpdir"

	cat >>"$XDG_CONFIG_HOME/libretools/libretools.conf" <<-eot
		BLACKLIST='phony://example.com'
	eot

	testsudo libremakepkg -l "$BATS_TEST_NAME" >$tmpdir/stdout 2>$tmpdir/stderr || status=$?

	[[ $status != 0 ]]
	tail -n1 $tmpdir/stderr | grep -qF '==> ERROR: Failure(s) in check_pkgbuild: check_pkgbuild_nonfree'
}

# bats test_tags=network,sudo
@test "libremakepkg detects distcc files" {
	cp fixtures/libremakepkg/PKGBUILD-hello "$tmpdir/PKGBUILD"
	cd "$tmpdir"

	cat >>"$XDG_CONFIG_HOME/libretools/chroot.conf" <<-eot
		CHROOTEXTRAPKG+=(distcc-nozeroconf socat)
	eot
	testsudo librechroot -l "$BATS_TEST_NAME" install-name distcc-nozeroconf socat

	# first make sure that the engine works
	testsudo libremakepkg -l "$BATS_TEST_NAME"
	globfile libretools-hello-1.0-1-any.pkg.tar?(.!(sig|*.*))
	rm -f -- libretools-hello-1.0-1-any.pkg.tar?(.!(sig|*.*))
	# now throw a wrench in it
	testsudo librechroot -l "$BATS_TEST_NAME" run touch /bin/distcc-tool
	# and make sure that the engine broke
	testsudo libremakepkg -l "$BATS_TEST_NAME" || status=$?
	[[ $status != 0 ]]
	refute globfile libretools-hello-1.0-1-any.pkg.tar?(.!(sig|*.*))
}

# bats test_tags=network,sudo
@test "libremakepkg forwards distcc ports" {
	# The maximum AF_UNIX socket path is 108 bytes; so let's have
	# a chroot name that's guaranteed to be >110 characters, to
	# make sure we handle that.
	local chrootname=$BATS_TEST_NAME.aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa

	local distcc_port
	distcc_port=$(./lib/runserver "$tmpdir/distcc.pid" \
		echo 'Hello, I am a distcc server')

	cat >>"$XDG_CONFIG_HOME/libretools/chroot.conf" <<-eot
		CHROOTEXTRAPKG+=(distcc-nozeroconf socat)
	eot
	cat >>"$XDG_CONFIG_HOME/pacman/makepkg.conf" <<-eot
		DISTCC_HOSTS=127.0.0.1:${distcc_port@Q}
	eot

	cp fixtures/libremakepkg/PKGBUILD-distcc "$tmpdir/PKGBUILD"
	cd "$tmpdir"
	testsudo librechroot -l "$chrootname" install-name distcc-nozeroconf socat
	testsudo libremakepkg -l "$chrootname"
	globfile libretools-distcc-1.0-1-any.pkg.tar?(.!(sig|*.*))
}

# bats test_tags=network,sudo
@test "libremakepkg doesnt symlink outputs" {
	sed -i /^unset/d "$XDG_CONFIG_HOME/pacman/makepkg.conf"

	cp fixtures/libremakepkg/PKGBUILD-hello "$tmpdir/PKGBUILD"
	cd "$tmpdir"

	testsudo libremakepkg -l "$BATS_TEST_NAME"

	refute stat libretools-hello-1.0-1-any.pkg.tar?(.!(sig|*.*))
	refute stat libretools-hello-1.0-1-any.src.tar?(.!(sig|*.*))
	globfile "$tmpdir/workdir/pkgdest"/libretools-hello-1.0-1-any.pkg.tar?(.!(sig|*.*))
	globfile "$tmpdir/workdir/srcpkgdest"/libretools-hello-1.0-1-any.src.tar?(.!(sig|*.*))
}

# bats test_tags=network,sudo
@test "libremakepkg succeeds with good signatures" {
	cp fixtures/libremakepkg/PKGBUILD-signed "$tmpdir/PKGBUILD"
	cp fixtures/libremakepkg/hello.sh "$tmpdir/hello.sh"
	cd "$tmpdir"
	gpg --detach-sign --use-agent --no-armor hello.sh

	testsudo libremakepkg -l "$BATS_TEST_NAME"
}

# bats test_tags=network,sudo
@test "libremakepkg fails with bad signatures" {
	cp fixtures/libremakepkg/PKGBUILD-signed "$tmpdir/PKGBUILD"
	cp fixtures/libremakepkg/hello.sh "$tmpdir/hello.sh"
	cd "$tmpdir"
	gpg --detach-sign --use-agent --no-armor hello.sh
	echo 'echo pwned' >>hello.sh
	makepkg -g >>PKGBUILD

	refute testsudo libremakepkg -l "$BATS_TEST_NAME"
}

# bats test_tags=network,sudo
@test "libremakepkg does not run pkgver" {
	cp fixtures/libremakepkg/PKGBUILD-pkgver "$tmpdir/PKGBUILD"
	pushd "$tmpdir"

	testsudo libremakepkg -l "$BATS_TEST_NAME"

	globfile libretools-pkgver-1-1-any.pkg.tar?(.!(sig|*.*))
	refute globfile libretools-pkgver-2-1-any.pkg.tar?(.!(sig|*.*))
	popd
	diff -u fixtures/libremakepkg/PKGBUILD-pkgver "$tmpdir/PKGBUILD"
}

# bats test_tags=network,sudo
@test "libremakepkg has a flag to make startdir rw" {
	cp fixtures/libremakepkg/PKGBUILD-rwstartdir "$tmpdir/PKGBUILD"
	cd "$tmpdir"

	refute testsudo libremakepkg -l "$BATS_TEST_NAME"
	refute globfile libretools-rwstartdir-1.0-1-any.pkg.tar?(.!(sig|*.*))
	testsudo libremakepkg -l "$BATS_TEST_NAME" -W
	globfile libretools-rwstartdir-1.0-1-any.pkg.tar?(.!(sig|*.*))
}

# bats test_tags=network,sudo
@test "libremakepkg can re-use source-packages" {
	cp fixtures/libremakepkg/PKGBUILD-hello "$tmpdir/PKGBUILD"
	cd "$tmpdir"

	# I'm tempted to use file checksums here, but that would break
	# when we gain support for reproducible builds.  So just use
	# timestamps.

	testsudo libremakepkg -l "$BATS_TEST_NAME"
	globfile libretools-hello-1.0-1-any.src.tar?(.!(sig|*.*))
	globfile libretools-hello-1.0-1-any.pkg.tar?(.!(sig|*.*))
	a_stime=$(stat -c %Y -- libretools-hello-1.0-1-any.src.tar?(.!(sig|*.*)))
	a_ptime=$(stat -c %Y -- libretools-hello-1.0-1-any.pkg.tar?(.!(sig|*.*)))

	testsudo libremakepkg -l "$BATS_TEST_NAME" -S libretools-hello-1.0-1-any.src.tar?(.!(sig|*.*))
	globfile libretools-hello-1.0-1-any.src.tar?(.!(sig|*.*))
	globfile libretools-hello-1.0-1-any.pkg.tar?(.!(sig|*.*))
	b_stime=$(stat -c %Y -- libretools-hello-1.0-1-any.src.tar?(.!(sig|*.*)))
	b_ptime=$(stat -c %Y -- libretools-hello-1.0-1-any.pkg.tar?(.!(sig|*.*)))

	((a_stime == b_stime))
	((a_ptime < b_ptime))
	((a_ptime > a_stime))
	((b_ptime > a_stime))
}
