# Copyright (C) 2013-2014, 2017-2018, 2024  Luke T. Shumaker <lukeshu@parabola.nu>
# Copyright (C) 2024  Bill Auger <bill-auger@programmer.net>
#
# SPDX-License-Identifier: GPL-2.0-or-later
#
# This file is part of Parabola Libretools.
#
# Libretools is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# Libretools is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

load ../lib/common

## helpers ##

# Stub server-side `db-update` for SSH localhost to run
StubDbupdate() {
	cat <<-eot
		#!/bin/bash
		echo "==> Updating [fake-repo]"
		{
				printf '%s\n' "\$DBSCRIPTS_CONFIG"
				readlink -f -- "\$STAGING"
				find "\$STAGING" -printf '%P\n' | LC_COLLATE=C sort
		} >${tmpdir@Q}/log.txt
	eot
}

# Stub server-side `pbot-say` for SSH localhost to run
StubPbotsay() {
	cat <<-eot
		#!/bin/bash
		echo "\$*" >${tmpdir@Q}/pbot.txt
	eot
}

StageFiles() {
	local staging_dir="$1"

	mkdir -p "$staging_dir"/repo1 "$staging_dir"/repo2/sub
	touch \
		"$staging_dir"/repo1/file1 \
		"$staging_dir"/repo1/file2 \
		"$staging_dir/repo2/file with spaces" \
		"$staging_dir"/repo2/sub/subfolder
}

## setup/teardown ##

setup() {
	libretools_common_per_test_setup

	local ssh_port

	# Configure and start the SSH server
	install -Dm644 /dev/stdin "$tmpdir/etc/ssh/sshd_config" <<-eot
		AuthorizedKeysCommand /usr/bin/cat ${HOME}/.ssh/id_rsa.pub
		AuthorizedKeysCommandUser ${USER}
		PasswordAuthentication no
		AcceptEnv TMPDIR
		AcceptEnv _HOME GNUPGHOME XDG_CACHE_HOME XDG_CONFIG_HOME
		AcceptEnv _PATH LIBRETOOLS_LIBRARY_PATH _librelib_conf_sh_sysconfdir _librelib_conf_sh_pkgconfdir
		AcceptEnv GPGKEY
		ForceCommand HOME=\$_HOME; PATH=\$_PATH; eval "\$SSH_ORIGINAL_COMMAND"
	eot
	ssh-keygen -A -f "$tmpdir"
	ssh_port=$(./lib/runserver "$tmpdir/sshd.pid" \
		"$(type -p sshd)" -i \
		-f "$tmpdir/etc/ssh/sshd_config" \
		-h "$tmpdir/etc/ssh/ssh_host_ecdsa_key")

	# Configure the SSH client
	mkdir -p -- "$HOME/.ssh"
	ssh-keygen -N '' -f "$HOME/.ssh/id_rsa"
	ssh-keyscan -p "$ssh_port" 127.0.0.1 >"$HOME/.ssh/known_hosts"
	touch -- "$HOME/.ssh/config"

	# Configure libretools
	cat >>"$XDG_CONFIG_HOME/libretools/libretools.conf" <<-eot
		TIER0_HOST=127.0.0.1
		TIER0_PORT=$ssh_port
		TIER0_STAGING=${tmpdir@Q}/srv-staging/
		DBSCRIPTS_CONFIG=/etc/dbscripts/config.local.phony
		HOOKPRERELEASE=()

		_HOOKPOSTRELEASE=()
		for hookcmd in "\${HOOKPOSTRELEASE[@]}"; do
		  if [[ \$hookcmd == *pbot* ]]; then
		    _HOOKPOSTRELEASE+=("\$hookcmd")
		  fi
		done
		HOOKPOSTRELEASE=("\${_HOOKPOSTRELEASE[@]}")
		unset _HOOKPOSTRELEASE
	eot
}

teardown() {
	xargs -a "$tmpdir/sshd.pid" kill --

	libretools_common_per_test_teardown
}

cat-and-fail() {
	status=$?
	cat "$1"
	return $status
}

## tests ##

# bats test_tags=locale-es
@test "librerelease follows cli conventions" {
	assert_follows_cli_conventions librerelease
	assert_shows_usage librerelease --help -c
	assert_shows_usage librerelease --help -C
	assert_shows_usage librerelease --help -l
	assert_shows_usage librerelease -c --help
	assert_shows_usage librerelease -C --help
	assert_shows_usage librerelease -l --help
}

@test "librerelease lists all files" {
	StageFiles "$tmpdir"/workdir/staging

	LC_ALL=C librerelease -l &>"$tmpdir/list" || cat-and-fail "$tmpdir/list"

	cat >"$tmpdir/list-correct" <<-eot
		  -> repo1
		     file1
		     file2
		  -> repo2
		     file with spaces
		     sub/subfolder
	eot

	diff -u "$tmpdir/list-correct" "$tmpdir/list"
}

@test "librerelease fails if GPGKEY not set" {
	unset GPGKEY

	StageFiles "$tmpdir"/workdir/staging

	LC_ALL=C librerelease -l >"$tmpdir/stdout" 2>"$tmpdir/stderr" || status=$?

	[[ $status != 0 ]]
	assert_file_empty "$tmpdir/stdout"
	grep GPGKEY "$tmpdir/stderr"
}

@test "librerelease fails if TIER0_HOST not set" {
	sed -i 's|TIER0_HOST=.*|TIER0_HOST=|' "$XDG_CONFIG_HOME/libretools/libretools.conf"

	StageFiles "$tmpdir"/workdir/staging

	LC_ALL=C librerelease -l >"$tmpdir"/stdout 2>"$tmpdir"/stderr || status=$?

	[[ $status != 0 ]]
	assert_file_empty "$tmpdir"/stdout
	grep TIER0_HOST "$tmpdir"/stderr
}

@test "librerelease fails if GPGKEY and TIER0_HOST are not set" {
	unset GPGKEY
	sed -i 's|TIER0_HOST=.*|TIER0_HOST=|' "$XDG_CONFIG_HOME/libretools/libretools.conf"

	StageFiles "$tmpdir"/workdir/staging

	LC_ALL=C librerelease -l >"$tmpdir"/stdout 2>"$tmpdir"/stderr || status=$?

	[[ $status != 0 ]]
	assert_file_empty "$tmpdir"/stdout
	grep GPGKEY "$tmpdir"/stderr
	grep TIER0_HOST "$tmpdir"/stderr
}

@test "librerelease fails if DBSCRIPTS_CONFIG is not set" {
	cat >>"$XDG_CONFIG_HOME/libretools/libretools.conf" <<-eot
		DBSCRIPTS_CONFIG=''
	eot

	StageFiles "$tmpdir"/workdir/staging

	LC_ALL=C librerelease -l >"$tmpdir/stdout" 2>"$tmpdir/stderr" || status=$?

	[[ $status != 0 ]]
	assert_file_empty "$tmpdir/stdout"
	grep DBSCRIPTS_CONFIG "$tmpdir/stderr"
}

@test "librerelease cleans local staging" {
	local staging_dir="$tmpdir"/workdir/staging
	StageFiles "$staging_dir"

	cat >"$tmpdir"/expected-stdout <<-eot
		removed 'repo1/file1'
		removed 'repo1/file2'
		removed 'repo2/file with spaces'
		removed 'repo2/sub/subfolder'
	eot

	LC_ALL=C librerelease -c >"$tmpdir"/stdout 2>"$tmpdir"/stderr || status=$?

	[[ $status == 0 ]]
	grep "==> Removing files from local staging directory" "$tmpdir"/stderr
	diff -u "$tmpdir"/expected-stdout "$tmpdir"/stdout
	((!$(command ls -A1 "$staging_dir" | wc -l)))
}

@test "librerelease cleans remote staging" {
	local staging_dir="$tmpdir"/srv-staging
	StageFiles "$staging_dir"

	cat >"$tmpdir"/expected-stdout <<-eot
		removed '$staging_dir//repo1/file1'
		removed '$staging_dir//repo1/file2'
		removed directory '$staging_dir//repo1'
		removed '$staging_dir//repo2/file with spaces'
		removed '$staging_dir//repo2/sub/subfolder'
		removed directory '$staging_dir//repo2/sub'
		removed directory '$staging_dir//repo2'
	eot

	LC_ALL=C librerelease -C >"$tmpdir"/stdout 2>"$tmpdir"/stderr || status=$?

	[[ $status == 0 ]]
	grep "==> Removing files from remote staging directory" "$tmpdir"/stderr
	# `rm -rfv` order depends on the underlying filesystem and so
	# we need to sort this for it to be reproducible.
	LC_ALL=C sort <"$tmpdir"/expected-stdout >"$tmpdir"/expected-stdout.sorted
	LC_ALL=C sort <"$tmpdir"/stdout >"$tmpdir"/stdout.sorted
	diff -u "$tmpdir"/expected-stdout.sorted "$tmpdir"/stdout.sorted
	((!$(command ls -A1 "$staging_dir" | wc -l)))
}

@test "librerelease publishes packages successfully" {
	# Add server-side stub
	install -Dm755 /dev/stdin "$tmpdir/bin/db-update" < <(StubDbupdate) # writes log.txt
	PATH=$tmpdir/bin:$PATH

	# Log which directories the hooks are run in.
	cat >>"$XDG_CONFIG_HOME/libretools/libretools.conf" <<-eot
		HOOKPRERELEASE+=('pwd >${tmpdir@Q}/prerelease.txt')
		HOOKPOSTRELEASE+=('pwd >${tmpdir@Q}/postrelease.txt')
	eot

	StageFiles "$tmpdir"/workdir/staging

	# Run
	librerelease

	# Make sure everything went OK
	pwd >"$tmpdir/pwd.txt"
	cat >"$tmpdir/log-correct.txt" <<-eot
		/etc/dbscripts/config.local.phony
		$(readlink -f -- "$tmpdir/srv-staging")

		repo1
		repo1/file1
		repo1/file1.sig
		repo1/file2
		repo1/file2.sig
		repo2
		repo2/file with spaces
		repo2/file with spaces.sig
		repo2/sub
		repo2/sub/subfolder
		repo2/sub/subfolder.sig
	eot
	diff -u "$tmpdir/log-correct.txt" "$tmpdir/log.txt"
	diff -u "$tmpdir/pwd.txt" "$tmpdir/prerelease.txt"
	diff -u "$tmpdir/pwd.txt" "$tmpdir/postrelease.txt"
}

@test "librerelease notifies pbot" {
	# Add server-side stubs
	install -Dm755 /dev/stdin "$tmpdir"/bin/db-update < <(StubDbupdate) # writes log.txt
	install -Dm755 /dev/stdin "$tmpdir"/bin/pbot-say < <(StubPbotsay)   # writes pbot.txt
	PATH=$tmpdir/bin:$PATH

	StageFiles "$tmpdir"/workdir/staging

	LC_ALL=C librerelease 2>"$tmpdir"/stderr

	grep -E "  -> Notifying pbot:" "$tmpdir"/stderr
	#grep "     $USER just published:" "$tmpdir"/pbot.txt
	grep "$USER just published:" "$tmpdir"/pbot.txt
}

@test "librerelease logs-in as TIER0_LOGIN" {
	PATH=$tmpdir/bin:$PATH

	echo "TIER0_LOGIN=tier0-user" >>"$XDG_CONFIG_HOME"/libretools/libretools.conf
	LC_ALL=C librerelease -C >"$tmpdir"/stdout 2>"$tmpdir"/stderr || status=$?

	[[ $status != 0 ]]
	assert_file_empty "$tmpdir"/stdout
	grep "tier0-user@127.0.0.1: Permission denied" "$tmpdir"/stderr
}
