# Copyright (C) 2013-2014, 2017-2018, 2024  Luke T. Shumaker <lukeshu@parabola.nu>
#
# SPDX-License-Identifier: GPL-2.0-or-later
#
# This file is part of Parabola Libretools.
#
# Libretools is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# Libretools is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

load ../lib/common

setup() {
	libretools_common_per_test_setup

	# Disable networking for blacklist.txt, and install a cached version instead
	cat >>"$XDG_CONFIG_HOME/libretools/libretools.conf" <<-eot
		BLACKLIST='phony://example.com'
	eot
	install -Dm644 /dev/stdin "$XDG_CACHE_HOME/libretools/blacklist.txt" <<-eot
		linux:linux-libre:nonfree blobs and firmwares
		rubygemsx:rubygemsx:repackaged
		skype
	eot
}

# Make sure these match pkgbuild-check-nonfree
declare -rig _E_OK=0
declare -rig _E_ERROR=1
declare -rig _E_LIC_UNKNOWN=4
declare -rig _E_LIC_NOGPL=8
declare -rig _E_LIC_NONFREE=16
declare -rig _E_DEP_NONFREE=32
declare -rig _E_PKG_NONFREE=64

# Make sure these match pkgbuild-summarize-nonfree (libremessages)
declare -rig EXIT_SUCCESS=0
declare -rig EXIT_FAILURE=1

# basic operation ##############################################################

# bats test_tags=locale-es
@test "pkgbuild-check-nonfree follows cli conventions" {
	assert_follows_cli_conventions pkgbuild-check-nonfree
	assert_fails_with_errusage pkgbuild-check-nonfree -x
}

@test "pkgbuild-check-nonfree fails when there is no blacklist" {
	rm $XDG_CACHE_HOME/libretools/blacklist.txt

	pkgbuild-check-nonfree fixtures/pkgbuild-check-nonfree/PKGBUILD.free >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $_E_ERROR ]]
}

# bats test_tags=locale-es
@test "pkgbuild-summarize-nonfree follows cli conventions" {
	assert_follows_cli_conventions pkgbuild-summarize-nonfree
}

# PKGBUILD codes ###############################################################

# basic _E_OK
@test "pkgbuild-check-nonfree succeeds for free depends" {
	pkgbuild-check-nonfree fixtures/pkgbuild-check-nonfree/PKGBUILD.free >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $_E_OK ]]
}

# _E_DEP_LIC_UNKNOWN
@test "pkgbuild-check-nonfree warns about unknown licenses" {
	pkgbuild-check-nonfree fixtures/pkgbuild-check-nonfree/PKGBUILD.lic-unknown >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $_E_LIC_UNKNOWN ]]

	local pcn_stat=$status
	status=0

	pkgbuild-summarize-nonfree $pcn_stat >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $EXIT_SUCCESS ]]
}

# _E_DEP_LIC_NOGPL
@test "pkgbuild-check-nonfree warns about GPL-incompatible licenses" {
	pkgbuild-check-nonfree fixtures/pkgbuild-check-nonfree/PKGBUILD.lic-nogpl >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $_E_LIC_NOGPL ]]

	local pcn_stat=$status
	status=0

	pkgbuild-summarize-nonfree $pcn_stat >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $EXIT_SUCCESS ]]
}

# _E_DEP_LIC_NONFREE
@test "pkgbuild-check-nonfree fails on nonfree licenses" {
	pkgbuild-check-nonfree fixtures/pkgbuild-check-nonfree/PKGBUILD.lic-nonfree >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $_E_LIC_NONFREE ]]

	local pcn_stat=$status
	status=0

	pkgbuild-summarize-nonfree $pcn_stat >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $EXIT_FAILURE ]]
}

# _E_DEP_NONFREE
@test "pkgbuild-check-nonfree succeeds for repackaged depend" {
	pkgbuild-check-nonfree fixtures/pkgbuild-check-nonfree/PKGBUILD.dep-repackaged >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $_E_OK ]]
}
@test "pkgbuild-check-nonfree succeeds for nonfree depend with replacement" {
	pkgbuild-check-nonfree fixtures/pkgbuild-check-nonfree/PKGBUILD.dep-nonfree-replacement >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $_E_OK ]]
}
@test "pkgbuild-check-nonfree fails for nonfree depend" {
	pkgbuild-check-nonfree fixtures/pkgbuild-check-nonfree/PKGBUILD.dep-nonfree >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $_E_DEP_NONFREE ]]

	local pcn_stat=$status
	status=0

	pkgbuild-summarize-nonfree $pcn_stat >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $EXIT_FAILURE ]]
}

# _E_PKG_NONFREE
@test "pkgbuild-check-nonfree succeeds for repackaged package" {
	pkgbuild-check-nonfree fixtures/pkgbuild-check-nonfree/PKGBUILD.pkg-repackaged >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $_E_OK ]]
}
@test "pkgbuild-check-nonfree fails for package with replacement" {
	pkgbuild-check-nonfree fixtures/pkgbuild-check-nonfree/PKGBUILD.pkg-nonfree-replacement >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $_E_PKG_NONFREE ]]

	local pcn_stat=$status
	status=0

	pkgbuild-summarize-nonfree $pcn_stat >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $EXIT_FAILURE ]]
}
@test "pkgbuild-check-nonfree fails for nonfree package" {
	pkgbuild-check-nonfree fixtures/pkgbuild-check-nonfree/PKGBUILD.pkg-nonfree >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $_E_PKG_NONFREE ]]

	local pcn_stat=$status
	status=0

	pkgbuild-summarize-nonfree $pcn_stat >$tmpdir/stdout 2>$tmpdir/stderr || status=$?
	assert_file_empty $tmpdir/stdout
	assert_file_not_empty $tmpdir/stderr
	[[ $status == $EXIT_FAILURE ]]
}
