#!/hint/bash
# Copyright (C) 2013-2015, 2017-2018, 2024  Luke T. Shumaker <lukeshu@parabola.nu>
# Copyright (C) 2017  Isaac David <isacdaavid@isacdaavid.info>
#
# SPDX-License-Identifier: GPL-2.0-or-later
#
# This file is part of Parabola Libretools.
#
# Libretools is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# Libretools is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

if [[ -z $LIBRETOOLS_LIBRARY_PATH || -z $_librelib_conf_sh_sysconfdir || -z $_librelib_conf_sh_pkgconfdir ]]; then
	libremessages error 'Must be run with ./testenv'
	exit 1
fi

# per-test setup/teardown ############################################

libretools_common_per_test_setup() {
	tmpdir="$(mktemp -d --tmpdir test.XXXXXXXXXXXX)"
	chmod 755 "$tmpdir"

	status=0

	# Clear the list of makepkg variables
	unset PKGDEST SRCDEST SRCPKGDEST LOGDEST BUILDDIR PKGEXT SRCEXT GPGKEY PACKAGER CARCH

	# Set up a test HOME
	export HOME="$tmpdir/home"
	export GNUPGHOME="$HOME/.gnupg"
	export XDG_CACHE_HOME="$HOME/.cache"
	export XDG_CONFIG_HOME="$HOME/.config"

	# Create a GPGKEY
	mkdir -p -- "$GNUPGHOME"
	chmod 700 -- "$GNUPGHOME"
	gpg --quiet --no-tty --batch --gen-key <<-eot
		Key-Type: DSA
		Key-Length: 1024
		Subkey-Type: ECDSA
		Subkey-Curve: nistp256

		Key-Usage: sign
		Name-Real: Bob Tester
		Name-Email: tester@localhost

		Expire-Date: 0
		%no-protection
		%commit
	eot
	export GPGKEY="$(gpg --quiet --list-secret-keys --with-colons | awk -F: '/^sec:/{print substr($5,9)}')"

	# Configure libretools
	export chrootdir="${chrootdir:-$TMPDIR/chroots}"

	install -Dm644 /dev/stdin "$XDG_CONFIG_HOME"/libretools/libretools.conf <<-eot
		WORKDIR=${tmpdir@Q}/workdir
		BLACKLIST=https://git.parabola.nu/blacklist.git/plain/blacklist.txt
	eot
	install -Dm644 /dev/stdin "$XDG_CONFIG_HOME"/libretools/chroot.conf <<-eot
		CHROOTDIR=${chrootdir@Q}
		CHROOT=default
		CHROOTEXTRAPKG=()
	eot
	install -Dm644 /dev/stdin "$XDG_CONFIG_HOME"/pacman/makepkg.conf <<-eot
		PKGDEST=${tmpdir@Q}/workdir/pkgdest
		SRCDEST=${tmpdir@Q}/workdir/srcdest
		SRCPKGDEST=${tmpdir@Q}/workdir/srcpkgdest
		LOGDEST=${tmpdir@Q}/workdir/logdest
		PACKAGER='Bob Tester <tester@localhost>'
	eot
	mkdir -p -- "$tmpdir"/workdir/{pkg,src,srcpkg,log}dest
}

libretools_common_per_test_teardown() {
	gpg-connect-agent KILLAGENT /bye || true
	if [[ -n ${tmpdir:-} ]]; then
		if libremessages in_array 'sudo' "${BATS_TEST_TAGS[@]}"; then
			sudo rm -rf -- "$tmpdir"
		else
			rm -rf -- "$tmpdir"
		fi
	fi
	# Clean up non-root chroots.  On btrfs it wouldn't be terrible
	# to wait until the end because CoW means that they don't take
	# up much space, but on other filesystems, we don't want to
	# require such a large disk.
	if [[ -n ${chrootdir:-} ]]; then
		local dir
		for dir in "$chrootdir"/*/*/; do
			if [[ -d $dir && $dir != */root/ ]]; then
				if [[ "$(stat -f -c %T "$dir")" == btrfs ]]; then
					sudo find "$dir" -depth -inum 256 -exec \
						btrfs subvolume delete {} \; &>/dev/null
				fi
				sudo rm -rf -- "$dir"
			fi
		done
	fi
}

setup() {
	libretools_common_per_test_setup
}

teardown() {
	libretools_common_per_test_teardown
}

# Utility functions for use in test definitions ######################

bats_require_minimum_version 1.7.0
bats_load_library bats-support
bats_load_library bats-assert
bats_load_library bats-file

# Plain command substitution would
#
#  1. remove trailing whitespace, despite being significant when
#     testing for newline-terminated lines.
#
#  2. make it hard to detect failure from the command being
#     substituted (`equals` allows us to detect this with `set -o
#     pipefail`).
equals() {
	local stdin
	IFS= read -rd '' stdin || :
	[[ $1 == "$stdin" ]]
}

globfile() {
	[[ -f $1 ]]
}

assert_stderr_line() {
	local lines=("${stderr_lines[@]}")
	assert_line "$@"
}

assert_shows_usage() {
	echo "Assert shows usage: ${@@Q}"
	LC_ALL=C run --keep-empty-lines --separate-stderr "$@"
	assert_success
	assert_equal "$stderr" ''

	echo "Checking 'Usage:' line(s)..."
	assert_regex "${lines[0]}" "^Usage: .*${1}"
	declare -p lines

	local or_re="^   or: .*${1}"
	local have_desc=false
	local line
	for line in "${lines[@]:1}"; do
		if ! $have_desc; then
			if [[ $line =~ $or_re ]]; then
				continue
			fi
			echo "Checking one-line description..."
			assert_regex "$line" '^[A-Z]([^.]|\.\S)+\.$'
			have_desc=true
		else
			echo "Checking blank..."
			assert_equal "$line" ''
			break
		fi
	done

	assert_regex "$output" $'\n  -h, --help '
}

assert_fails_with_errusage() {
	echo "Assert fails with errusage: ${@@Q}"
	LC_ALL=C run --keep-empty-lines --separate-stderr "$@"
	assert_failure 2
	assert_equal "$output" ''
	assert [ ${#stderr_lines[@]} -gt 1 ]
	assert_regex "${stderr_lines[-1]}" "^Try .*${1} --help' for more information\.\$"
}

assert_follows_cli_conventions() {
	local progname=$1

	assert_shows_usage "$progname" -h
	assert_shows_usage "$progname" --help

	echo "Assert usage is translated: ${@@Q}"
	local lang
	if ! lang=$(localedef --list-archive | grep -m1 ^es_); then
		echo 'This test requires that you have an "es_*" locale enabled in /etc/locale.gen'
		false
	fi
	LC_ALL=$lang run --keep-empty-lines --separate-stderr "$progname" -h
	assert_success
	assert_equal "$stderr" ''
	assert_regex "${lines[0]}" "^Uso: .*${1}"

	assert_fails_with_errusage "$progname" --frob
	assert_stderr_line "$progname: unrecognized option '--frob'"
	assert_fails_with_errusage "$progname" --help --frob
	assert_stderr_line "$progname: unrecognized option '--frob'"
	assert_fails_with_errusage "$progname" --frob --help
	assert_stderr_line "$progname: unrecognized option '--frob'"
}
