package Lire::Firewall::WatchguardDlfConverter;

use strict;

use Lire::DlfConverter;
use Lire::Firewall qw/firewall_number2names/;
use Lire::Syslog;
use Carp;

use base qw/Lire::DlfConverter/;

sub new {
    my $proto = shift;
    bless {}, (ref $proto || $proto);
}

sub name { 'watchguard' }
sub title { 'Watchguard firewall log' }
sub description { '<para>Watchguard firewall log</para>' }
sub schemas { qw/firewall/ }

sub handle_log_lines { 1 }

sub init_dlf_converter {
    my ($self, $process) = @_;

    $self->{'parser'} = new Lire::Syslog;
}

sub process_log_line {
    my ($self, $process, $line) = @_;

    #
    # skip invalid log entries
    #
    return $process->ignore_log_line($line)
        unless $line =~ /firewalld/;

    eval {
        my $log = $self->{'parser'}->parse($line);
        local $_ = $log->{'content'};
        #
        # skip invalid log entries
        #
        return $process->ignore_log_line($line)
            unless defined $log->{'process'} && $log->{'process'} eq 'firewalld';

        my @v = split ' ', $log->{'content'};
        return $process->error($line,
                'invalid firewalld record at line '
                .$process->line_count
                .': should contain more than 9 fields')
            if @v < 9;

        #
        # assign log values to hash
        #
        my %dlf = (
            'time' => $log->{'timestamp'},
            'count' => 1,
            'length' => $v[3],
            'protocol' => $v[4],
            'from_ip' => $v[7],
            'to_ip' => $v[8],
            'from_port' => $v[9],
            'to_port' => $v[10],
        );

        if($v[0] eq 'deny')  {
            $dlf{'action'} = 'denied'
        } elsif($v[0] eq 'allow') {
            $dlf{'action'} = 'permitted'
        } else {
            return $process->ignore_log_line($line)
        }

        $log->{'content'} =~ / \((.+)\)/;
        $dlf{'rule'} = $1;

        #
        # convert numbers to names and create dlf-record
        #
        firewall_number2names(\%dlf);
        $process->write_dlf('firewall', \%dlf);
    };

    if($@) {
        $process->error($line, $@);
    }
}

sub finish_conversion {
    delete $_[0]->{'parser'};
}

1; # nag nag.

__END__

=pod

=head1 NAME

Lire::Firewall::WatchguardDlfConverter - convert Watchguard logs to firewall DLF

=head1 DESCRIPTION

B<Lire::Firewall::WatchguardDlfConverter> converts Watchguard logs into
firewall DLF format.
Input for this converter is the standard Watchguard syslog log file.

=head1 AUTHOR

Wessel Dankers <wsl@logreport.org>

Initial code by Joost Bekkers <joost@jodocus.org>, now maintained by the
LogReport team.  Based upon the cisco_ios2dlf.in script.

=head1 SEE ALSO

The WatchGuard website at http://www.watchguard.com/ for some information on
the WatchGuard Firebox System.  Unfortunately, only very little information is
freely available.

=head1 VERSION

$Id: WatchguardDlfConverter.pm,v 1.10 2006/07/23 13:16:35 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2001-2003 Stichting LogReport Foundation LogReport@LogReport.org

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html.

=cut

# Local Variables:
# mode: cperl
# End:
