#!/usr/bin/env python
"""Script for checking the ABINIT automatic tests."""

import sys
import os

from os.path import join as pj, abspath as absp, basename
from pprint import pprint

from optparse import OptionParser
from warnings import warn

#try:
#    import tests
#except ImportError:
# Add the directory [...]/abinit/tests to $PYTHONPATH
pack_dir, tail = os.path.split(absp(__file__))
pack_dir, tail = os.path.split(pack_dir)
sys.path.insert(0,pack_dir)
import tests

from tests import abitests
abenv = tests.abenv

__version__ = "0.1"
__author__  = "Matteo Giantomassi"

###############################################################################
### Helper functions and tools
###############################################################################

def find_unknown_keywords(suite):
    "Check whether keywords have been documented"
    unknowns = {}
    for test in suite:
        for key in test.keywords:
            if not key: continue
            if key not in tests.KNOWN_KEYWORDS:
                if not unknowns.has_key(key):
                    unknowns[key] = [test.full_id]
                else:
                    unknowns[key].append(test.full_id)
    return unknowns

def path2str(path):
    head, fname = os.path.split(path)
    head, x = os.path.split(head)
    x, dir = os.path.split(head)
    return "["+dir+"]["+fname+"]"

def exclude_path(p):
    p = basename(p)
    if ( p.startswith(".") or p.endswith("~") ): return True
    return False

def find_stale_or_lost_refs(full_database):
    """
    Check whether all reference files located in the Refs directories
    are tested namely that if they compare in the files_to_test field.
    """
    # Build the list of files that are tested.
    retcode = 0
    for suite_name in abitests.suite_names:
        # List all the files in suite_name/Refs (ignore hidden files).
        ref_dir = abenv.apath_of("tests", suite_name, "Refs")

        if not os.path.exists(ref_dir):
            print "%s does not exist" % ref_dir
            continue

        listdir = [f for f in os.listdir(ref_dir) if not exclude_path(f)]
        ref_fnames = [pj(ref_dir, f) for f in listdir] # use absolute path.

        # Mapping ref_fname --> number of tests using it.
        ref2test = dict().fromkeys(ref_fnames, 0)

        # Extract the sub-suite.
        suite = full_database[suite_name]

        for test in suite:
            files_to_test = [pj(ref_dir, f.name) for f in test.files_to_test]

            for o in files_to_test:
                # FIXME due to out --> stdout replacement
                if o.endswith(".stdout"): o = o[:-7] + ".out" 
                if o not in ref2test:
                    err_msg = "files_to_test %s does not appear in Refs!" % o
                    warn(err_msg)
                    #raise ValueError(err_msg)
                else:
                    ref2test[o] += 1

        # At this point ref2test should contain only ones.
        for ref_fname, ntimes in ref2test.items():
            if ntimes != 1:
                err_msg = "Ref file %s is tested %s time(s)" % (path2str(ref_fname), ntimes)
                print(err_msg)
                retcode += 1

    return retcode

def find_stale_or_lost_inputs(full_database):
    """
    Check whether all reference files located in the Refs directories
    are tested namely that if they compare in the files_to_test field.
    """
    # Build the list of files that are tested.
    retcode = 0
    for suite_name in abitests.suite_names:
        #
        # List all the files in suite_name/Input (exclude hidden files or vim backup files).
        inp_dir = abenv.apath_of("tests", suite_name, "Input")

        listdir =  [ f for f in os.listdir(inp_dir) if not exclude_path(f)]
        inp_fnames = [pj(inp_dir, f) for f in listdir] # use absolute path....

        # Mapping inp_fname --> number of tests using it.
        inp2test = dict().fromkeys(inp_fnames,0)

        # Extract the sub-suite.
        suite = full_database[suite_name]

        for test in suite:
            for ius in test.inputs_used:
                if ius not in inp2test:
                    raise ValueError("Input [%s] [%s] does not appear in Input2keys!" % (suite_name, ius))
                else:
                    inp2test[ius] += 1

        def remove_file(fname):
            # XG130810 : When the report.in files in abirules/Input/report.in  buildsys/Input/report.in 
            # will have been suppressed, one might replace the next line by the simpler :    
            # return fname.endswith(".files")
            return fname.endswith(".files") or basename(fname) in ["report.in",]

        for fname, ntimes in inp2test.items():
            if remove_file(fname):
                ntest = inp2test.pop(fname)
                assert ntest == 0

        # At this point inp2test should be >= 1.
        for fname, ntimes in inp2test.items():
            #if ntimes != 1:
            if ntimes == 0:
                retcode += 1
                err_msg = "Input file %s is used %s time(s)" % (path2str(fname), ntimes)
                print(err_msg)

    return retcode

def check_options_in_test(test):
    recommended_opts = ["keywords", "description", "authors", "max_nprocs"]
    d = {}
    for ropt in recommended_opts:
        try:
            value = getattr(test, ropt)
            if not value: d[ropt] = "EMPTY"
        except AttributeError:
            d[ropt] = "MISSING"
    return d

def first_second_name(string):
    idx = string.rfind(".")
    if idx == -1:
        first, second = "", string
    else:
        first, second = string[:idx], string[idx+1:]
    return first.strip(), second.strip()

def check_authors(suite):
    second_names  = []
    for test in suite:
        for string in test.authors:
            f, s = first_second_name(string)
            if not f and s and s != "Unknown":
                print "author(s) first name is missing in file %s, string = %s " %(test.full_id, s)
            second_names.append(s)
        #print test.id, first_second_name(test.authors[0])[1]
    second_names = set(second_names)
    return second_names

###############################################################################

def main():
    usage = "usage: %prog [suite_name] [options] [-h|--help] for help)"
    version="%prog "+ str(__version__)
    parser = OptionParser(usage=usage, version=version)

    parser.add_option("-v", "--verbose", dest="verbose", action="store_true", default=False,
                      help="verbose mode")

    (options, args) = parser.parse_args()

    # Regenerate the full database.
    full_database = abitests.build_database(with_disabled=True)

    #res_table = full_database.init_result_table()
    #pprint(res_table)
    #sys.exit(1)

    print 100*"="
    print "stale or lost refs"
    find_stale_or_lost_refs(full_database)
    #full_database.find_stale_or_lost_refs()
    #sys.exit(0)

    print 100*"="
    print "stale or lost inputs"
    find_stale_or_lost_inputs(full_database)
    #full_database.find_stale_or_lost_inputs()

    print 100*"="
    print "wrong keywords"
    for (suite_name, suite) in full_database.items():
        unknowns = find_unknown_keywords(suite)
        if unknowns:
            print "The following keys are not documented: \n\t%s" % str([k for k in unknowns.keys()])
            print "Mapping keywords --> input file"
            pprint(unknowns)

    #sys.exit(1)

    # Check authors.
    #second_names = set()
    #for (suite_name, suite) in full_database.items():
    #  second_names = second_names.union( check_authors(suite) )
    #pprint(second_names)
    #sys.exit(1)

    # Text presence of important options.
    #for (suite_name, suite) in full_database.items():
    #    for test in suite:
    #        wrong_options = check_options_in_test(test)
    #        for (opt, stat) in wrong_options.items():
    #            print "%s: option %s is %s" % (test.full_id, opt, stat)

    sys.exit(0)

###############################################################################

if __name__ == "__main__":
    main()
