#!/bin/bash

set -e -u

iso_name=hyperbola-milky-way
iso_version=v0.2.5
iso_label="HYPER_${iso_version//./}"
export galaxy_version="Milky Way"
install_dir=hyperbola
work_dir=work
out_dir=out
data_dir=/usr/share/hyperiso/data

arch=$(uname -m)
verbose=""
pacman_conf=${work_dir}/pacman.conf
script_path=$(readlink -f ${0%/*})

_usage ()
{
    echo "usage ${0} [options]"
    echo
    echo " General options:"
    echo "    -N <iso_name>      Set an iso filename (prefix)"
    echo "                        Default: ${iso_name}"
    echo "    -V <iso_version>   Set an iso version (in filename)"
    echo "                        Default: ${iso_version}"
    echo "    -L <iso_label>     Set an iso label (disk label)"
    echo "                        Default: ${iso_label}"
    echo "    -D <install_dir>   Set an install_dir (directory inside iso)"
    echo "                        Default: ${install_dir}"
    echo "    -w <work_dir>      Set the working directory"
    echo "                        Default: ${work_dir}"
    echo "    -o <out_dir>       Set the output directory"
    echo "                        Default: ${out_dir}"
    echo "    -v                 Enable verbose output"
    echo "    -h                 This help message"
    exit ${1}
}

# Helper function to run make_*() only one time per architecture.
run_once() {
    if [[ ! -e ${work_dir}/build.${1}_${arch} ]]; then
        $1
        touch ${work_dir}/build.${1}_${arch}
    fi
}

# Setup custom pacman.conf with current cache directories.
make_pacman_conf() {
    local _cache_dirs
    _cache_dirs=($(pacman -v 2>&1 | grep '^Cache Dirs:' | sed 's/Cache Dirs:\s*//g'))
    sed -r "s|^#?\\s*CacheDir.+|CacheDir = $(echo -n ${_cache_dirs[@]})|g" ${script_path}/pacman.conf > ${pacman_conf}
}

# Base installation, plus needed packages (root-image)
make_basefs() {
    setarch ${arch} mkhyperiso ${verbose} -w "${work_dir}/${arch}" -C "${pacman_conf}" -D "${install_dir}" init
    setarch ${arch} mkhyperiso ${verbose} -w "${work_dir}/${arch}" -C "${pacman_conf}" -D "${install_dir}" -p "memtest86+ mkinitcpio-nfs-utils nbd" install
}

# Additional packages (root-image)
make_packages() {
    setarch ${arch} mkhyperiso ${verbose} -w "${work_dir}/${arch}" -C "${pacman_conf}" -D "${install_dir}" -p "$(grep -h -v ^# ${script_path}/packages.{both,${arch}})" install
}

# Copy mkinitcpio hyperiso hooks and build initramfs (root-image)
make_setup_mkinitcpio() {
    local _hook
    for _hook in hyperiso hyperiso_shutdown hyperiso_pxe_common hyperiso_pxe_nbd hyperiso_pxe_http hyperiso_pxe_nfs hyperiso_loop_mnt; do
        cp /usr/lib/initcpio/hooks/${_hook} ${work_dir}/${arch}/root-image/usr/lib/initcpio/hooks
        cp /usr/lib/initcpio/install/${_hook} ${work_dir}/${arch}/root-image/usr/lib/initcpio/install
    done
    cp /usr/lib/initcpio/install/hyperiso_kms ${work_dir}/${arch}/root-image/usr/lib/initcpio/install
    cp /usr/lib/initcpio/hyperiso_shutdown ${work_dir}/${arch}/root-image/usr/lib/initcpio
    cp ${script_path}/mkinitcpio.conf ${work_dir}/${arch}/root-image/etc/mkinitcpio-hyperiso.conf
    setarch ${arch} mkhyperiso ${verbose} -w "${work_dir}/${arch}" -C "${pacman_conf}" -D "${install_dir}" -r 'mkinitcpio -c /etc/mkinitcpio-hyperiso.conf -k /boot/vmlinuz-linux-libre-lts -g /boot/hyperiso.img' run
}

# Customize installation (root-image)
make_customize_root_image() {
    cp -dRf --preserve=mode,timestamps,links ${script_path}/root-image ${work_dir}/${arch}

    iso_version="${iso_version}" setarch ${arch} mkhyperiso ${verbose} -w "${work_dir}/${arch}" -C "${pacman_conf}" -D "${install_dir}" -r '/root/customize_root_image.sh' run
    rm ${work_dir}/${arch}/root-image/root/customize_root_image.sh
}

# Prepare kernel/initramfs ${install_dir}/boot/
make_boot() {
    mkdir -p ${work_dir}/iso/${install_dir}/boot/${arch}
    cp ${work_dir}/${arch}/root-image/boot/hyperiso.img ${work_dir}/iso/${install_dir}/boot/${arch}/hyperiso.img
    cp ${work_dir}/${arch}/root-image/boot/vmlinuz-linux-libre-lts ${work_dir}/iso/${install_dir}/boot/${arch}/vmlinuz
}

# Add other aditional/extra files to ${install_dir}/boot/
make_boot_extra() {
    cp ${work_dir}/${arch}/root-image/boot/memtest86+/memtest.bin ${work_dir}/iso/${install_dir}/boot/memtest
    cp ${work_dir}/${arch}/root-image/usr/share/licenses/common/GPL2/license.txt ${work_dir}/iso/${install_dir}/boot/memtest.COPYING
}

# Prepare /${install_dir}/boot/syslinux
make_syslinux() {
    mkdir -p ${work_dir}/iso/${install_dir}/boot/syslinux
    for _cfg in ${script_path}/syslinux/*.cfg; do
        sed "s|%HYPERISO_LABEL%|${iso_label}|g;
             s|%INSTALL_DIR%|${install_dir}|g" ${_cfg} > ${work_dir}/iso/${install_dir}/boot/syslinux/${_cfg##*/}
    done
    cp ${data_dir}/splash.png ${work_dir}/iso/${install_dir}/boot/syslinux
    cp ${work_dir}/${arch}/root-image/usr/lib/syslinux/bios/*.c32 ${work_dir}/iso/${install_dir}/boot/syslinux
    cp ${work_dir}/${arch}/root-image/usr/lib/syslinux/bios/lpxelinux.0 ${work_dir}/iso/${install_dir}/boot/syslinux
    cp ${work_dir}/${arch}/root-image/usr/lib/syslinux/bios/memdisk ${work_dir}/iso/${install_dir}/boot/syslinux
    mkdir -p ${work_dir}/iso/${install_dir}/boot/syslinux/hdt
    gzip -c -9 ${work_dir}/${arch}/root-image/usr/share/hwdata/pci.ids > ${work_dir}/iso/${install_dir}/boot/syslinux/hdt/pciids.gz
    gzip -c -9 ${work_dir}/${arch}/root-image/usr/lib/modules/*-gnu-*/modules.alias > ${work_dir}/iso/${install_dir}/boot/syslinux/hdt/modalias.gz
}

# Prepare /isolinux
make_isolinux() {
    mkdir -p ${work_dir}/iso/isolinux
    sed "s|%INSTALL_DIR%|${install_dir}|g" ${script_path}/isolinux/isolinux.cfg > ${work_dir}/iso/isolinux/isolinux.cfg
    cp ${work_dir}/${arch}/root-image/usr/lib/syslinux/bios/isolinux.bin ${work_dir}/iso/isolinux/
    cp ${work_dir}/${arch}/root-image/usr/lib/syslinux/bios/isohdpfx.bin ${work_dir}/iso/isolinux/
    cp ${work_dir}/${arch}/root-image/usr/lib/syslinux/bios/ldlinux.c32 ${work_dir}/iso/isolinux/
}

# Prepare efiboot.img::/EFI for "El Torito" EFI boot mode
make_efiboot() {
    mkdir -p ${work_dir}/iso/EFI/hyperiso
    truncate -s 38M ${work_dir}/iso/EFI/hyperiso/efiboot.img
    mkfs.vfat -n HYPER_EFI ${work_dir}/iso/EFI/hyperiso/efiboot.img

    mkdir -p ${work_dir}/efiboot
    modprobe loop
    mount ${work_dir}/iso/EFI/hyperiso/efiboot.img ${work_dir}/efiboot

    mkdir -p ${work_dir}/efiboot/EFI/hyperiso
    cp ${work_dir}/iso/${install_dir}/boot/x86_64/vmlinuz ${work_dir}/efiboot/EFI/hyperiso/vmlinuz.efi
    cp ${work_dir}/iso/${install_dir}/boot/x86_64/hyperiso.img ${work_dir}/efiboot/EFI/hyperiso/hyperiso.img

    mkdir -p ${work_dir}/efiboot/EFI/boot
    for _cfg in ${script_path}/efiboot/syslinux/*.cfg; do
        sed "s|%HYPERISO_LABEL%|${iso_label}|g;
             s|%INSTALL_DIR%|${install_dir}|g" ${_cfg} > ${work_dir}/efiboot/EFI/boot/${_cfg##*/}
    done
    cp ${data_dir}/splash.png ${work_dir}/efiboot/EFI/boot
    cp ${work_dir}/x86_64/root-image/usr/lib/syslinux/efi64/* ${work_dir}/efiboot/EFI/boot
    cp ${work_dir}/x86_64/root-image/usr/lib/syslinux/efi64/syslinux.efi ${work_dir}/efiboot/EFI/boot/bootx64.efi

    umount ${work_dir}/efiboot
}

# Copy aitab
make_aitab() {
    mkdir -p ${work_dir}/iso/${install_dir}
    cp ${script_path}/aitab ${work_dir}/iso/${install_dir}/aitab
}

# Build all filesystem images specified in aitab (.fs.sfs .sfs)
make_prepare() {
    cp -dRfl --preserve=mode,timestamps,links ${work_dir}/${arch}/root-image ${work_dir}
    setarch ${arch} mkhyperiso ${verbose} -w "${work_dir}" -D "${install_dir}" pkglist
    setarch ${arch} mkhyperiso ${verbose} -w "${work_dir}" -D "${install_dir}" prepare
    rm -rf ${work_dir}/root-image
    # rm -rf ${work_dir}/${arch}/root-image (if low space, this helps)
}

# Build ISO
make_iso() {
    mkhyperiso ${verbose} -w "${work_dir}" -D "${install_dir}" checksum
    mkhyperiso ${verbose} -w "${work_dir}" -D "${install_dir}" -L "${iso_label}" -o "${out_dir}" iso "${iso_name}-${iso_version}-dual.iso"
}

if [[ ${EUID} -ne 0 ]]; then
    echo "This script must be run as root."
    _usage 1
fi

if [[ ${arch} != x86_64 ]]; then
    echo "This script needs to be run on x86_64"
    _usage 1
fi

while getopts 'N:V:L:D:w:o:vh' arg; do
    case "${arg}" in
        N) iso_name="${OPTARG}" ;;
        V) iso_version="${OPTARG}" ;;
        L) iso_label="${OPTARG}" ;;
        D) install_dir="${OPTARG}" ;;
        w) work_dir="${OPTARG}" ;;
        o) out_dir="${OPTARG}" ;;
        v) verbose="-v" ;;
        h) _usage 0 ;;
        *)
           echo "Invalid argument '${arg}'"
           _usage 1
           ;;
    esac
done

mkdir -p ${work_dir}

run_once make_pacman_conf

# Do all stuff for each root-image
for arch in i686 x86_64; do
    run_once make_basefs
    run_once make_packages
    run_once make_setup_mkinitcpio
    run_once make_customize_root_image
done

for arch in i686 x86_64; do
    run_once make_boot
done

# Do all stuff for "iso"
run_once make_boot_extra
run_once make_syslinux
run_once make_isolinux
run_once make_efiboot

run_once make_aitab

for arch in i686 x86_64; do
    run_once make_prepare
done

run_once make_iso
