#!/usr/bin/env bash
# This may be included with or without `set -euE`

# Copyright (C) 2013-2014, 2016-2017 Luke Shumaker <lukeshu@sbcglobal.net>
#
# License: GNU GPLv2+
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# make sure XDG_CACHE_HOME is set
. "$(librelib conf)"

# Usage: blacklist-normalize <$file
# Normalizes the syntax of the blacklist on stdin.
blacklist-normalize() {
	sed -e '/^#/d' -e 's/^[^:]*$/&::/' -e 's/^[^:]*:[^:]*$/&:/'
}

# Usage: blacklist-cat
# Prints the blacklist.
# Uses the cache, but downloads it if it doesn't exist.  Also normalizes the blacklist for easier parsing.
blacklist-cat() {
	local file="$XDG_CACHE_HOME/libretools/blacklist.txt"
	if ! [[ -e $file ]]; then
		# exit on failure, whether set -e or not
		blacklist-update || return
	fi
	blacklist-normalize < "$file"
}

# Usage: blacklist-update
# Updates (or creates) the cached copy of the blacklist.
blacklist-update() (
	. "$(librelib messages)"
	load_files libretools || return 1
	check_vars libretools BLACKLIST || return 1

	local remote_blacklist="$BLACKLIST"
	local local_blacklist="$XDG_CACHE_HOME/libretools/blacklist.txt"

	_l stat_busy "Downloading blacklist of proprietary software packages"

	mkdir -p "${local_blacklist%/*}"
	if wget -N -q -O "${local_blacklist}.part" "$remote_blacklist" 2>/dev/null; then
		stat_done
		mv -f "${local_blacklist}.part" "$local_blacklist"
	else
		stat_done
		rm "${local_blacklist}.part"
		if [[ -e "$local_blacklist" ]]; then
			_l warning "Using local copy of blacklist"
		else
			_l error "Download failed, exiting"
			return 1
		fi

	fi
)

# Usage: blacklist-cat | blacklist-lookup $pkgname
# Filters to obtain the line for $pkgname from the blacklist on stdin.
# Exits successfully whether a line is found or not.
blacklist-lookup() {
	local pkg=$1
	# we accept that $pkg contains no regex-nes
	blacklist-normalize | grep "^$pkg:" || true
}

# Usage: blacklist-cat | blacklist-get-pkg
# Prints only the package name field of the blacklist line(s) on stdin.
blacklist-get-pkg() {
	blacklist-normalize | cut -d: -f1
}

# Usage: blacklist-cat | blacklist-get-rep
# Prints only the replacement package field of the blacklist line(s) on stdin.
blacklist-get-rep() {
	blacklist-normalize | cut -d: -f2
}

# Usage: blacklist-cat | blacklist-get-reason
# Prints only the reason field of the blacklist line(s) on stdin.
blacklist-get-reason() {
	blacklist-normalize | cut -d: -f3-
}
