#!/hint/bash -euE
# License: GNU GPLv2
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

. "$(librelib common.sh)"
. "$(librelib archroot.sh)"

shopt -s nullglob

init_variables() {
	default_makepkg_args=(-s --noconfirm -L --holdver)
	makepkg_args=("${default_makepkg_args[@]}")
	repack=false
	update_first=false
	clean_first=false
	run_namcap=false
	temp_chroot=false
	chrootdir=
	passeddir=
	makepkg_user=
	declare -ga install_pkgs
	declare -gi ret=0

	bindmounts_ro=()
	bindmounts_rw=()

	copy=$USER
	[[ -n ${DOAS_USER:-} ]] && copy=$DOAS_USER
	[[ -z "$copy" || $copy = root ]] && copy=copy
	src_owner=${DOAS_USER:-$USER}
}

usage() {
	echo "Usage: ${0##*/} [options] -r <chrootdir> [--] [makepkg args]"
	echo ' Run this script in a PKGBUILD dir to build a package inside a'
	echo ' clean chroot. Arguments passed to this script after the'
	echo ' end-of-options marker (--) will be passed to makepkg.'
	echo ''
	echo ' The chroot dir consists of the following directories:'
	echo ' <chrootdir>/{root, copy} but only "root" is required'
	echo ' by default. The working copy will be created as needed'
	echo ''
	echo 'The chroot "root" directory must be created via the following'
	echo 'command:'
	echo '    mkarchroot <chrootdir>/root base-devel'
	echo ''
	echo 'This script reads {SRC,SRCPKG,PKG,LOG}DEST, MAKEFLAGS and PACKAGER'
	echo 'from makepkg.conf(5), if those variables are not part of the'
	echo 'environment.'
	echo ''
	echo "Default makepkg args: ${default_makepkg_args[*]}"
	echo ''
	echo 'Flags:'
	echo '-h         This help'
	echo '-c         Clean the chroot before building'
	echo '-d <dir>   Bind directory into build chroot as read-write'
	echo '-D <dir>   Bind directory into build chroot as read-only'
	echo '-u         Update the working copy of the chroot before building'
	echo '           This is useful for rebuilds without dirtying the pristine'
	echo '           chroot'
	echo '-r <dir>   The chroot dir to use'
	echo '-I <pkg>   Install a package into the working copy of the chroot'
	echo '-l <copy>  The directory to use as the working copy of the chroot'
	echo '           Useful for maintaining multiple copies'
	echo "           Default: $copy"
	echo '-n         Run namcap on the package'
	echo '-T         Build in a temporary directory'
	echo '-U         Run makepkg as a specified user'
	exit 1
}

# {{{ functions
# Usage: load_vars $makepkg_conf
# Globals:
#  - SRCDEST
#  - SRCPKGDEST
#  - PKGDEST
#  - LOGDEST
#  - MAKEFLAGS
#  - PACKAGER
load_vars() {
	local makepkg_conf="$1" var

	[[ -f $makepkg_conf ]] || return 1

	for var in {SRC,SRCPKG,PKG,LOG}DEST MAKEFLAGS PACKAGER; do
		[[ -z ${!var:-} ]] && eval "$(grep "^${var}=" "$makepkg_conf")"
	done

	return 0
}

# Usage: sync_chroot $rootdir $copydir [$copy]
sync_chroot() {
	local rootdir=$1
	local copydir=$2
	local copy=${3:-$2}

	if [[ "$rootdir" -ef "$copydir" ]]; then
		error 'Cannot sync copy with itself: %s' "$copydir"
		return 1
	fi

	# Get a read lock on the root chroot to make
	# sure we don't clone a half-updated chroot
	slock 8 "$rootdir.lock" \
		"Locking clean chroot [%s]" "$rootdir"

	stat_busy "Synchronizing chroot copy [%s] -> [%s]" "$rootdir" "$copy"
	if is_subvolume "$rootdir" && is_same_fs "$rootdir" "$(dirname -- "$copydir")" && ! mountpoint -q "$copydir"; then
		if is_subvolume "$copydir"; then
			subvolume_delete_recursive "$copydir" ||
				die "Unable to delete subvolume %s" "$copydir"
		else
			# avoid change of filesystem in case of an umount failure
			rm --recursive --force --one-file-system "$copydir" ||
				die "Unable to delete %s" "$copydir"
		fi
		btrfs subvolume snapshot "$rootdir" "$copydir" >/dev/null ||
			die "Unable to create subvolume %s" "$copydir"
	else
		mkdir -p "$copydir"
		rsync -a --delete -q -W -x "$rootdir/" "$copydir"
	fi
	stat_done

	# Drop the read lock again
	lock_close 8

	# Update mtime
	touch "$copydir"
}

# Usage: delete_chroot $copydir [$copy]
delete_chroot() {
	local copydir=$1
	local copy=${1:-$2}

	stat_busy "Removing chroot copy [%s]" "$copy"
	if is_subvolume "$copydir" && ! mountpoint -q "$copydir"; then
		subvolume_delete_recursive "$copydir" ||
			die "Unable to delete subvolume %s" "$copydir"
	else
		# avoid change of filesystem in case of an umount failure
		rm --recursive --force --one-file-system "$copydir" ||
			die "Unable to delete %s" "$copydir"
	fi

	# remove lock file
	rm -f "$copydir.lock"
	stat_done
}

# Usage: install_packages $copydir $pkgs...
install_packages() {
	local copydir=$1
	local install_pkgs=("${@:2}")

	local -a pkgnames
	local ret

	pkgnames=("${install_pkgs[@]##*/}")

	cp -- "${install_pkgs[@]}" "$copydir/root/"
	arch-nspawn "$copydir" "${bindmounts_ro[@]}" "${bindmounts_rw[@]}" \
		pacman -U --noconfirm -- "${pkgnames[@]/#//root/}"
	ret=$?
	rm -- "${pkgnames[@]/#/$copydir/root/}"

	return $ret
}

# Usage: prepare_chroot $copydir $HOME $repack $run_namcap
# Globals:
#  - MAKEFLAGS
#  - PACKAGER
prepare_chroot() {
	local copydir=$1
	local USER_HOME=$2
	local repack=$3
	local run_namcap=$4

	$repack || rm -rf "$copydir/build"

	local builduser_uid builduser_gid
	builduser_uid="$(id -u ${DOAS_USER:-$USER})"
	builduser_gid="$(id -g "$builduser_uid")"
	local install="install -o $builduser_uid -g $builduser_gid"
	local x

	# We can't use useradd without chrooting, otherwise it invokes PAM modules
	# which we might not be able to load (i.e. when building i686 packages on
	# an x86_64 host).
	sed -e '\/adm:/ s/,builduser//g' -i "$copydir"/etc/{group,gshadow}
	sed -e '/^builduser:/d' -i "$copydir"/etc/{passwd,group,gshadow,shadow}
	printf >>"$copydir/etc/passwd"  'builduser:x:%d:%d:builduser:/build:/bin/bash\n' "$builduser_uid" "$builduser_gid"
	printf >>"$copydir/etc/shadow"  'builduser:x:14871::::::\n'
	printf >>"$copydir/etc/group"   'builduser:x:%d:\n' "$builduser_gid"
	printf >>"$copydir/etc/gshadow" 'builduser:x::\n'
	sed -e '\/adm:/ s/root,daemon/root,daemon,builduser/' -i "$copydir"/etc/{group,gshadow}

	$install -d "$copydir"/{build,build/.gnupg,startdir,{pkg,srcpkg,src,log}dest}

	for x in .gnupg/pubring.{kbx,gpg}; do
		[[ -r $USER_HOME/$x ]] || continue
		$install -m 644 "$USER_HOME/$x" "$copydir/build/$x"
	done

	sed -e '/^MAKEFLAGS=/d' -e '/^PACKAGER=/d' -i "$copydir/etc/makepkg.conf"
	for x in BUILDDIR=/build PKGDEST=/pkgdest SRCPKGDEST=/srcpkgdest SRCDEST=/srcdest LOGDEST=/logdest \
		"MAKEFLAGS='${MAKEFLAGS:-}'" "PACKAGER='${PACKAGER:-}'"
	do
		grep -q "^$x" "$copydir/etc/makepkg.conf" && continue
		echo "$x" >>"$copydir/etc/makepkg.conf"
	done

	if [ $(doas grep -qs 'permit nopass builduser as root cmd /usr/bin/pacman' "$copydir/etc/doas.conf"; echo $?) != 0 ]; then
		cat > "$copydir/etc/doas.conf" <<EOF
permit nopass :wheel
permit nopass builduser as root cmd /usr/bin/pacman
EOF
	fi

	if ! grep -q '^\[repo\]' "$copydir/etc/pacman.conf"; then
		local line
		line=$(grep -n '^\[' "$copydir/etc/pacman.conf" |grep -Fv ':[options]'|sed 's/:.*//;1q')
		local ins='[repo]
SigLevel = Optional TrustAll
Server = file:///repo
'
		sed -i "${line}i${ins//$'\n'/\\n}" "$copydir/etc/pacman.conf"
	fi
	# Avoid having to use `pacman -Sy` to update [repo], as
	# networking might be disabled inside of the chroot.
	cp "$copydir/repo/repo.db" "$copydir/var/lib/pacman/sync/repo.db"

	# This is a little gross, but this way the script is recreated every time in the
	# working copy
	{
		printf '#!/bin/bash\n'
		declare -f _chrootprepare
		printf '_chrootprepare "$@"\n'
	} > "$copydir/chrootprepare"
	chmod +x "$copydir/chrootprepare"
	{
		printf '#!/bin/bash\n'
		declare -f _chrootbuild
		printf '_chrootbuild "$@" || exit\n'

		if $run_namcap; then
			declare -f _chrootnamcap
			printf '_chrootnamcap || exit\n'
		fi
	} >"$copydir/chrootbuild"
	chmod +x "$copydir/chrootbuild"
}

# These functions aren't run in makechrootpkg,
# so no global variables
_chrootprepare() {
	# shellcheck source=/dev/null
	. /etc/profile
	if [ $(doas grep -qs 'permit nopass setenv { HOME="/build" PATH="/usr/local/bin:/usr/bin:/bin:/usr/local/games:/usr/games:/usr/local/sbin:/usr/sbin:/sbin:/usr/bin/site_perl:/usr/bin/vendor_perl:/usr/bin/core_perl" _="/bin/bash" } builduser' /etc/doas.conf; echo $?) != 0 ]; then
		echo 'permit nopass setenv { HOME="/build" PATH="/usr/local/bin:/usr/bin:/bin:/usr/local/games:/usr/games:/usr/local/sbin:/usr/sbin:/sbin:/usr/bin/site_perl:/usr/bin/vendor_perl:/usr/bin/core_perl" _="/bin/bash" } builduser' >> "$copydir/etc/doas.conf"
	fi
	doas -u builduser bash -c 'cd $HOME; cd /startdir; DOAS_USER='$DOAS_USER' makepkg "$@" --nobuild' -bash "$@"
	sed -i '\|permit nopass setenv { HOME="/build" PATH="/usr/local/bin:/usr/bin:/bin:/usr/local/games:/usr/games:/usr/local/sbin:/usr/sbin:/sbin:/usr/bin/site_perl:/usr/bin/vendor_perl:/usr/bin/core_perl" _="/bin/bash" } builduser|d' "$copydir/etc/doas.conf"
}

_chrootbuild() {
	# shellcheck source=/dev/null
	. /etc/profile
	if [ $(doas grep -qs 'permit nopass setenv { HOME="/build" PATH="/usr/local/bin:/usr/bin:/bin:/usr/local/games:/usr/games:/usr/local/sbin:/usr/sbin:/sbin" _="/bin/bash" } builduser' /etc/doas.conf; echo $?) != 0 ]; then
		echo 'permit nopass setenv { HOME="/build" PATH="/usr/local/bin:/usr/bin:/bin:/usr/local/games:/usr/games:/usr/local/sbin:/usr/sbin:/sbin" _="/bin/bash" } builduser' >> "$copydir/etc/doas.conf"
	fi
	doas -u builduser bash -c 'cd $HOME; cd /startdir; DOAS_USER='$DOAS_USER' makepkg "${@:2}" --allsource' -bash ".src.tar.gz" "$@" || return
	sed -i '\|permit nopass setenv { HOME="/build" PATH="/usr/local/bin:/usr/bin:/bin:/usr/local/games:/usr/games:/usr/local/sbin:/usr/sbin:/sbin:" _="/bin/bash" } builduser|d' "$copydir/etc/doas.conf"
	if [ $(doas grep -qs 'permit nopass setenv { HOME="/build" PATH="/usr/local/bin:/usr/bin:/bin:/usr/local/games:/usr/games:/usr/local/sbin:/usr/sbin:/sbin" _="/bin/bash" } builduser' /etc/doas.conf; echo $?) != 0 ]; then
		echo 'permit nopass setenv { HOME="/build" PATH="/usr/local/bin:/usr/bin:/bin:/usr/local/games:/usr/games:/usr/local/sbin:/usr/sbin:/sbin" _="/bin/bash" } builduser' >> "$copydir/etc/doas.conf"
	fi
	doas -u builduser bash -c 'cd $HOME; cd /startdir; DOAS_USER='$DOAS_USER' makepkg "$@" --noextract --noprepare' -bash "$@"
	sed -i '\|permit nopass setenv { HOME="/build" PATH="/usr/local/bin:/usr/bin:/bin:/usr/local/games:/usr/games:/usr/local/sbin:/usr/sbin:/sbin" _="/bin/bash" } builduser|d' "$copydir/etc/doas.conf"
}

_chrootnamcap() {
	pacman -S --needed --noconfirm namcap
	for pkgfile in /startdir/PKGBUILD /pkgdest/*; do
		echo "Checking ${pkgfile##*/}"
		doas -u builduser namcap "$pkgfile" 2>&1 | tee "/logdest/${pkgfile##*/}-namcap.log"
	done
}

# Usage: download_sources $copydir $makepkg_user
# Globals:
#  - SRCDEST
#  - USER
download_sources() {
	local copydir=$1
	local makepkg_user=$2

	local builddir
	builddir="$(mktemp -d)"
	chmod 1777 "$builddir"

	# Ensure sources are downloaded
	if [[ "$(id -u "$makepkg_user")" != 0 ]]; then
		doas -u "$makepkg_user" env SRCDEST="$SRCDEST" BUILDDIR="$builddir" \
			makepkg --config="$copydir/etc/makepkg.conf" --verifysource -o ||
			die "Could not download sources."
	else
		error "Running makepkg as root is not allowed."
		exit 1
	fi

	# Clean up garbage from verifysource
	rm -rf "$builddir"
}

# Usage: move_products $copydir $owner
# Globals:
#  - PKGDEST
#  - LOGDEST
move_products() {
	local copydir=$1
	local src_owner=$2

	local pkgfile
	for pkgfile in "$copydir"/pkgdest/*; do
		chown "$src_owner" "$pkgfile"
		mv "$pkgfile" "$PKGDEST"

		# Fix broken symlink because of temporary chroot PKGDEST /pkgdest
		if [[ "$PWD" != "$PKGDEST" && -L "$PWD/${pkgfile##*/}" ]]; then
			ln -sf "$PKGDEST/${pkgfile##*/}"
		fi
	done

	local l
	for l in "$copydir"/logdest/*; do
		[[ $l == */logpipe.* ]] && continue
		chown "$src_owner" "$l"
		mv "$l" "$LOGDEST"
	done

	for s in "$copydir"/srcpkgdest/*; do
		chown "$src_owner" "$s"
		mv "$s" "$SRCPKGDEST"
	done
}
# }}}

main() {
	init_variables

	while getopts 'hcur:I:l:nTD:d:U:' arg; do
		case "$arg" in
			c) clean_first=true ;;
			D) bindmounts_ro+=("--bind-ro=$OPTARG") ;;
			d) bindmounts_rw+=("--bind=$OPTARG") ;;
			u) update_first=true ;;
			r) passeddir="$OPTARG" ;;
			I) install_pkgs+=("$OPTARG") ;;
			l) copy="$OPTARG" ;;
			n) run_namcap=true; makepkg_args+=(-i) ;;
			T) temp_chroot=true; copy+="-$$" ;;
			U) makepkg_user="$OPTARG" ;;
			h|*) usage ;;
		esac
	done

	[[ ! -f PKGBUILD && -z "${install_pkgs[*]}" ]] && die 'This must be run in a directory containing a PKGBUILD.'
	[[ -n $makepkg_user && -z $(id -u "$makepkg_user") ]] && die 'Invalid makepkg user.'
	makepkg_user=${makepkg_user:-${DOAS_USER:-$USER}}

	check_root

	# Canonicalize chrootdir, getting rid of trailing /
	chrootdir=$(readlink -e "$passeddir")
	[[ ! -d $chrootdir ]] && die "No chroot dir defined, or invalid path '%s'" "$passeddir"
	[[ ! -d $chrootdir/root ]] && die "Missing chroot dir root directory. Try using: mkarchroot %s/root base-devel" "$chrootdir"

	if [[ ${copy:0:1} = / ]]; then
		copydir=$copy
	else
		copydir="$chrootdir/$copy"
	fi

	# Pass all arguments after -- right to makepkg
	makepkg_args+=("${@:$OPTIND}")

	# See if -R was passed to makepkg
	for arg in "${@:OPTIND}"; do
		case ${arg%%=*} in
			-*R*|--repackage)
				repack=true
				break 2
				;;
		esac
	done

	if [[ -n $DOAS_USER ]]; then
		eval "USER_HOME=~$DOAS_USER"
	else
		USER_HOME=$HOME
	fi

	umask 0022

	load_vars "${XDG_CONFIG_HOME:-$USER_HOME/.config}/pacman/makepkg.conf" || load_vars "$USER_HOME/.makepkg.conf"
	load_vars /etc/makepkg.conf

	# Use PKGBUILD directory if these don't exist
	[[ -d $PKGDEST ]]    || PKGDEST=$PWD
	[[ -d $SRCDEST ]]    || SRCDEST=$PWD
	[[ -d $SRCPKGDEST ]] || SRCPKGDEST=$PWD
	[[ -d $LOGDEST ]]    || LOGDEST=$PWD

	# Lock the chroot we want to use. We'll keep this lock until we exit.
	lock 9 "$copydir.lock" "Locking chroot copy [%s]" "$copy"

	if [[ ! -d $copydir ]] || $clean_first; then
		sync_chroot "$chrootdir/root" "$copydir" "$copy"
	fi

	$update_first && arch-nspawn "$copydir" \
			"${bindmounts_ro[@]}" "${bindmounts_rw[@]}" \
			pacman -Syu --noconfirm

	if [[ -n ${install_pkgs[*]:-} ]]; then
		install_packages "$copydir" "${install_pkgs[@]}"
		ret=$?
		# If there is no PKGBUILD we have done
		[[ -f PKGBUILD ]] || return $ret
	fi

	download_sources "$copydir" "$makepkg_user"

	prepare_chroot "$copydir" "$USER_HOME" "$repack"

	if arch-nspawn "$copydir" \
		--bind="$PWD:/startdir" \
		--bind="$SRCDEST:/srcdest" \
		"${bindmounts_ro[@]}" "${bindmounts_rw[@]}" \
		/chrootprepare "${makepkg_args[@]}" &&
	   arch-nspawn "$copydir" \
		--bind="$PWD:/startdir" \
		--bind="$SRCDEST:/srcdest" \
		"${bindmounts_ro[@]}" "${bindmounts_rw[@]}" \
		/chrootbuild "${makepkg_args[@]}"
	then
		move_products "$copydir" "$src_owner"
	else
		(( ret += 1 ))
	fi

	$temp_chroot && delete_chroot "$copydir" "$copy"

	if (( ret != 0 )); then
		if $temp_chroot; then
			die "Build failed"
		else
			die "Build failed, check %s/build" "$copydir"
		fi
	else
		true
	fi
}

