"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = __importDefault(require("path"));
const assert_1 = __importDefault(require("assert"));
const typescript_1 = __importDefault(require("typescript"));
const vscode_languageserver_1 = require("vscode-languageserver");
const documents_1 = require("../../../../src/lib/documents");
const SignatureHelpProvider_1 = require("../../../../src/plugins/typescript/features/SignatureHelpProvider");
const LSAndTSDocResolver_1 = require("../../../../src/plugins/typescript/LSAndTSDocResolver");
const utils_1 = require("../../../../src/utils");
const ls_config_1 = require("../../../../src/ls-config");
const testDir = path_1.default.join(__dirname, '..');
describe('SignatureHelpProvider', () => {
    function setup() {
        const docManager = new documents_1.DocumentManager((textDocument) => new documents_1.Document(textDocument.uri, textDocument.text));
        const filePath = path_1.default.join(testDir, 'testfiles', 'signature-help', 'signature-help.svelte');
        const lsAndTsDocResolver = new LSAndTSDocResolver_1.LSAndTSDocResolver(docManager, [(0, utils_1.pathToUrl)(testDir)], new ls_config_1.LSConfigManager());
        const provider = new SignatureHelpProvider_1.SignatureHelpProviderImpl(lsAndTsDocResolver);
        const document = docManager.openDocument({
            uri: (0, utils_1.pathToUrl)(filePath),
            text: typescript_1.default.sys.readFile(filePath)
        });
        return { provider, document };
    }
    it('provide signature help with formatted documentation', async () => {
        const { provider, document } = setup();
        const result = await provider.getSignatureHelp(document, vscode_languageserver_1.Position.create(3, 8), undefined);
        assert_1.default.deepStrictEqual(result, {
            signatures: [
                {
                    label: 'foo(): boolean',
                    documentation: { value: 'bars\n\n*@author* — John', kind: vscode_languageserver_1.MarkupKind.Markdown },
                    parameters: []
                }
            ],
            activeParameter: 0,
            activeSignature: 0
        });
    });
    it('provide signature help with function signatures', async () => {
        const { provider, document } = setup();
        const result = await provider.getSignatureHelp(document, vscode_languageserver_1.Position.create(4, 12), undefined);
        assert_1.default.deepStrictEqual(result, {
            signatures: [
                {
                    label: 'abc(a: number, b: number): string',
                    documentation: undefined,
                    parameters: [
                        {
                            label: [4, 13]
                        },
                        {
                            label: [15, 24]
                        }
                    ]
                },
                {
                    label: 'abc(a: number, b: string): string',
                    documentation: undefined,
                    parameters: [
                        {
                            label: [4, 13]
                        },
                        {
                            label: [15, 24],
                            documentation: 'formatted number'
                        }
                    ]
                }
            ],
            activeParameter: 1,
            activeSignature: 1
        });
    });
    it('filter out svelte2tsx signature', async () => {
        const { provider, document } = setup();
        const result = await provider.getSignatureHelp(document, vscode_languageserver_1.Position.create(18, 18), undefined);
        assert_1.default.equal(result, null);
    });
    it('provide signature help with formatted documentation', async () => {
        const { provider, document } = setup();
        const cancellationTokenSource = new vscode_languageserver_1.CancellationTokenSource();
        const signatureHelpPromise = provider.getSignatureHelp(document, vscode_languageserver_1.Position.create(3, 8), undefined, cancellationTokenSource.token);
        cancellationTokenSource.cancel();
        assert_1.default.deepStrictEqual(await signatureHelpPromise, null);
    });
});
//# sourceMappingURL=SignatureHelpProvider.test.js.map