from __future__ import unicode_literals
from __future__ import absolute_import, division, print_function

import logging
import os
import re

import papis.config
import click
import papis.document
import papis.importer
import papis.utils

logger = logging.getLogger("bibtex")

bibtex_types = [
  "article", "book", "booklet", "conference", "inbook", "incollection",
  "inproceedings", "manual", "mastersthesis", "misc", "phdthesis",
  "proceedings", "techreport", "unpublished"
] + papis.config.getlist('extra-bibtex-types')

bibtex_type_converter = {
  "conferencePaper": "inproceedings",
  "journalArticle": "article",
  "journal": "article"
}

bibtex_keys = [
    "addendum", "address", "afterword", "annotator", "annote", "author",
    "bookauthor", "booksubtitle", "booktitle", "booktitleaddon", "chapter",
    "commentator", "crossref", "date", "doi", "edition", "editor", "editora",
    "editorb", "editorc", "eid", "eprint", "eprintclass", "eprinttype",
    "eventdate", "eventtitle", "eventtitleaddon", "foreword", "holder",
    "howpublished", "institution", "introduction", "isbn", "isrn",
    "issn", "issue", "issuesubtitle", "issuetitle", "journal",
    "journalsubtitle", "journaltitle", "key", "language", "location",
    "mainsubtitle", "maintitle", "maintitleaddon", "month", "note",
    "number", "organization", "origlanguage", "pages", "pagetotal", "part",
    "publisher", "pubstate", "school", "series", "subtitle", "title",
    "translator", "type", "titleaddon", "url", "urldate", "venue", "version",
    "volume", "volumes", "year",
] + papis.config.getlist('extra-bibtex-keys')

bibtex_key_converter = {
    "abstractNote": "abstract",
    "university": "school",
    "conferenceName": "eventtitle",
    "place": "location",
    "publicationTitle": "journal",
    "proceedingsTitle": "booktitle"
}


class Importer(papis.importer.Importer):

    """Importer that parses a bibtex files"""

    def __init__(self, **kwargs):
        papis.importer.Importer.__init__(self, name='bibtex', **kwargs)

    @classmethod
    def match(cls, uri):
        if (not os.path.exists(uri) or os.path.isdir(uri) or
                papis.utils.get_document_extension(uri) == 'pdf'):
            return None
        importer = Importer(uri=uri)
        importer.fetch()
        return importer if importer.ctx else None

    @papis.importer.cache
    def fetch(self):
        self.logger.info("Reading input file = %s" % self.uri)
        try:
            bib_data = bibtex_to_dict(self.uri)
            if len(bib_data) > 1:
                self.logger.warning(
                    'Your bibtex file contains more than one entry,'
                    ' I will be taking the first entry')
            if bib_data:
                self.ctx.data = bib_data[0]
        except Exception as e:
            self.logger.debug(e)


@click.command('bibtex')
@click.pass_context
@click.argument('bibfile', type=click.Path(exists=True))
@click.help_option('--help', '-h')
def explorer(ctx, bibfile):
    """
    Import documents from a bibtex file

    Examples of its usage are

    papis explore bibtex lib.bib pick

    """
    logger = logging.getLogger('explore:bibtex')
    logger.info('Reading in bibtex file {}'.format(bibfile))
    docs = [
        papis.document.from_data(d)
        for d in bibtex_to_dict(bibfile)
    ]
    ctx.obj['documents'] += docs
    logger.info('{} documents found'.format(len(docs)))


def bibtexparser_entry_to_papis(entry):
    """Convert keys of a bib entry in bibtexparser format to papis compatible
    format.

    :param entry: Dictionary with keys of bibtexparser format.
    :type  entry: dict
    :returns: Dictionary with keys of papis format.

    """
    from bibtexparser.customization import splitname
    def to_author_list(authors):
        author_list = []
        for author in re.split(r"\s+and\s+", authors):
            parts = splitname(author)
            given = " ".join(parts["first"])
            family = " ".join(parts["von"] + parts["last"] + parts["jr"])

            author_list.append(dict(family=family, given=given))

        return author_list

    from collections import OrderedDict
    key_conversion = OrderedDict({
        "ID": {"key": "ref"},
        "ENTRYTYPE": {"key": "type"},
        "link": {"key": "url"},
        "author": {"key": "author_list", "action": to_author_list},
    })

    result = papis.document.keyconversion_to_data(key_conversion, entry,
            keep_unknown_keys=True)

    return result


def bibtex_to_dict(bibtex):
    """
    Convert bibtex file to dict

    .. code:: python

        { type: "article ...", "ref": "example1960etAl", author:" ..."}

    :param bibtex: Bibtex file path or bibtex information in string format.
    :type  bibtex: str
    :returns: Dictionary with bibtex information with keys that bibtex
        formally recognizes.
    :rtype:  list
    """
    from bibtexparser.bparser import BibTexParser

    parser = BibTexParser(common_strings=True,
            ignore_nonstandard_types=False,
            homogenize_fields=False,
            interpolate_strings=True)

    # bibtexparser has too many debug messages to be useful
    logging.getLogger("bibtexparser.bparser").setLevel(logging.WARNING)
    global logger
    if os.path.exists(bibtex):
        with open(bibtex) as fd:
            logger.debug("Reading in file %s" % bibtex)
            text = fd.read()
    else:
        text = bibtex
    entries = parser.parse(text, partial=True).entries
    # Clean entries
    return [bibtexparser_entry_to_papis(entry) for entry in entries]


def unicode_to_latex(text):
    """
    unicode_to_latex - what it says

    Provides unicode_to_latex(u) and unicode_to_latex_string(u).

    unicode_to_latex returns ASCII bytes that can be fed to LaTeX to reproduce
    the Unicode string 'u' as closely as possible.

    unicode_to_latex_string returns a Unicode string rather than bytes. That
    is,

       unicode_to_latex(u) = unicode_to_latex_string(u).encode('ascii').

    """
    # Adapted from
    # https://github.com/pkgw/bibtools/master/bibtools/unicode_to_latex.py
    # Thank you pkgw!
    #
    # Based on https://gist.github.com/798549 (owned by github user piquadrat),
    # but modified to make a table usable with unicode.translate(). I had to
    # comment out a few things and also tweak various of the conversions to
    # make the LaTeX more natural. There were also some surprising missing
    # conversions (e.g. u2010 -> -).

    # The original table comes from
    # http://www.w3.org/Math/characters/unicode.xml, converted via this XSL
    # script: https://gist.github.com/798546 . Based on my experience so far,
    # the source table is far from perfect.

    # from six import text_type

    unicode_to_latex_table_base = {
        u"\u0023": r"\#",
        u"\u0024": r"\textdollar{}",
        u"\u0025": r"\%",
        u"\u0026": r"\&",
        u"\u0027": r"'",
        u"\u002A": r"\ast{}",
        u"\u005C": r"\textbackslash{}",
        u"\u005E": r"\^{}",
        u"\u005F": r"\_",
        u"\u0060": r"\textasciigrave{}",
        u"\u007B": r"\lbrace{}",
        u"\u007C": r"\vert{}",
        u"\u007D": r"\rbrace{}",
        u"\u007E": r"\textasciitilde{}",
        u"\u00A0": r"~",
        u"\u00A1": r"\textexclamdown{}",
        u"\u00A2": r"\textcent{}",
        u"\u00A3": r"\textsterling{}",
        u"\u00A4": r"\textcurrency{}",
        u"\u00A5": r"\textyen{}",
        u"\u00A6": r"\textbrokenbar{}",
        u"\u00A7": r"\textsection{}",
        u"\u00A8": r"\textasciidieresis{}",
        u"\u00A9": r"\textcopyright{}",
        u"\u00AA": r"\textordfeminine{}",
        u"\u00AB": r"\guillemotleft{}",
        u"\u00AC": r"\lnot{}",
        u"\u00AD": r"\-",
        u"\u00AE": r"\textregistered{}",
        u"\u00AF": r"\textasciimacron{}",
        u"\u00B0": r"\textdegree{}",
        u"\u00B1": r"\pm{}",
        u"\u00B2": r"\ensuremath{^2}",
        u"\u00B3": r"\ensuremath{^3}",
        u"\u00B4": r"\textasciiacute{}",
        u"\u00B5": r"\mathrm{\mu}",
        u"\u00B6": r"\textparagraph{}",
        u"\u00B7": r"\cdot{}",
        u"\u00B8": r"\c{}",
        u"\u00B9": r"\ensuremath{^1}",
        u"\u00BA": r"\textordmasculine{}",
        u"\u00BB": r"\guillemotright{}",
        u"\u00BC": r"\textonequarter{}",
        u"\u00BD": r"\textonehalf{}",
        u"\u00BE": r"\textthreequarters{}",
        u"\u00BF": r"\textquestiondown{}",
        u"\u00C0": r"\`{A}",
        u"\u00C1": r"\'{A}",
        u"\u00C2": r"\^{A}",
        u"\u00C3": r"\~{A}",
        # note single-quoted string
        u"\u00C4": r'\"{A}',
        u"\u00C5": r"\AA{}",
        u"\u00C6": r"\AE{}",
        u"\u00C7": r"\c{C}",
        u"\u00C8": r"\`{E}",
        u"\u00C9": r"\'{E}",
        u"\u00CA": r"\^{E}",
        # note single-quoted string
        u"\u00CB": r'\"{E}',
        u"\u00CC": r"\`{I}",
        u"\u00CD": r"\'{I}",
        u"\u00CE": r"\^{I}",
        # note single-quoted string
        u"\u00CF": r'\"{I}',
        u"\u00D0": r"\DH{}",
        u"\u00D1": r"\~{N}",
        u"\u00D2": r"\`{O}",
        u"\u00D3": r"\'{O}",
        u"\u00D4": r"\^{O}",
        u"\u00D5": r"\~{O}",
        # note single-quoted string
        u"\u00D6": r'\"{O}',
        u"\u00D7": r"\texttimes{}",
        u"\u00D8": r"\O{}",
        u"\u00D9": r"\`{U}",
        u"\u00DA": r"\'{U}",
        u"\u00DB": r"\^{U}",
        # note single-quoted string
        u"\u00DC": r'\"{U}',
        u"\u00DD": r"\'{Y}",
        u"\u00DE": r"\TH{}",
        u"\u00DF": r"\ss{}",
        u"\u00E0": r"\`{a}",
        u"\u00E1": r"\'{a}",
        u"\u00E2": r"\^{a}",
        u"\u00E3": r"\~{a}",
        # note single-quoted string
        u"\u00E4": r'\"{a}',
        u"\u00E5": r"\aa{}",
        u"\u00E6": r"\ae{}",
        u"\u00E7": r"\c{c}",
        u"\u00E8": r"\`{e}",
        u"\u00E9": r"\'{e}",
        u"\u00EA": r"\^{e}",
        # note single-quoted string
        u"\u00EB": r'\"{e}',
        u"\u00EC": r"\`{\i}",
        u"\u00ED": r"\'{\i}",
        u"\u00EE": r"\^{\i}",
        # note single-quoted string
        u"\u00EF": r'\"{\i}',
        u"\u00F0": r"\dh{}",
        u"\u00F1": r"\~{n}",
        u"\u00F2": r"\`{o}",
        u"\u00F3": r"\'{o}",
        u"\u00F4": r"\^{o}",
        u"\u00F5": r"\~{o}",
        # note single-quoted string
        u"\u00F6": r'\"{o}',
        u"\u00F7": r"\div{}",
        u"\u00F8": r"\o{}",
        u"\u00F9": r"\`{u}",
        u"\u00FA": r"\'{u}",
        u"\u00FB": r"\^{u}",
        # note single-quoted string
        u"\u00FC": r'\"{u}',
        u"\u00FD": r"\'{y}",
        u"\u00FE": r"\th{}",
        # note single-quoted string
        u"\u00FF": r'\"{y}',
        u"\u0100": r"\={A}",
        u"\u0101": r"\={a}",
        # note non-raw string (\u confuses Python)
        u"\u0102": "\\u{A}",
        # note non-raw string (\u confuses Python)
        u"\u0103": "\\u{a}",
        u"\u0104": r"\k{A}",
        u"\u0105": r"\k{a}",
        u"\u0106": r"\'{C}",
        u"\u0107": r"\'{c}",
        u"\u0108": r"\^{C}",
        u"\u0109": r"\^{c}",
        u"\u010A": r"\.{C}",
        u"\u010B": r"\.{c}",
        u"\u010C": r"\v{C}",
        u"\u010D": r"\v{c}",
        u"\u010E": r"\v{D}",
        u"\u010F": r"\v{d}",
        u"\u0110": r"\DJ{}",
        u"\u0111": r"\dj{}",
        u"\u0112": r"\={E}",
        u"\u0113": r"\={e}",
        # note non-raw string (\u confuses Python)
        u"\u0114": "\\u{E}",
        # note non-raw string (\u confuses Python)
        u"\u0115": "\\u{e}",
        u"\u0116": r"\.{E}",
        u"\u0117": r"\.{e}",
        u"\u0118": r"\k{E}",
        u"\u0119": r"\k{e}",
        u"\u011A": r"\v{E}",
        u"\u011B": r"\v{e}",
        u"\u011C": r"\^{G}",
        u"\u011D": r"\^{g}",
        # note non-raw string (\u confuses Python)
        u"\u011E": "\\u{G}",
        # note non-raw string (\u confuses Python)
        u"\u011F": "\\u{g}",
        u"\u0120": r"\.{G}",
        u"\u0121": r"\.{g}",
        u"\u0122": r"\c{G}",
        u"\u0123": r"\c{g}",
        u"\u0124": r"\^{H}",
        u"\u0125": r"\^{h}",
        u"\u0126": r"{\fontencoding{LELA}\selectfont\char40}",
        u"\u0127": r"\Elzxh{}",
        u"\u0128": r"\~{I}",
        u"\u0129": r"\~{\i}",
        u"\u012A": r"\={I}",
        u"\u012B": r"\={\i}",
        # note non-raw string (\u confuses Python)
        u"\u012C": "\\u{I}",
        # note non-raw string (\u confuses Python)
        u"\u012D": "\\u{\\i}",
        u"\u012E": r"\k{I}",
        u"\u012F": r"\k{i}",
        u"\u0130": r"\.{I}",
        u"\u0131": r"\i{}",
        u"\u0132": r"IJ",
        u"\u0133": r"ij",
        u"\u0134": r"\^{J}",
        u"\u0135": r"\^{\j}",
        u"\u0136": r"\c{K}",
        u"\u0137": r"\c{k}",
        u"\u0138": r"{\fontencoding{LELA}\selectfont\char91}",
        u"\u0139": r"\'{L}",
        u"\u013A": r"\'{l}",
        u"\u013B": r"\c{L}",
        u"\u013C": r"\c{l}",
        u"\u013D": r"\v{L}",
        u"\u013E": r"\v{l}",
        u"\u013F": r"{\fontencoding{LELA}\selectfont\char201}",
        u"\u0140": r"{\fontencoding{LELA}\selectfont\char202}",
        u"\u0141": r"\L{}",
        u"\u0142": r"\l{}",
        u"\u0143": r"\'{N}",
        u"\u0144": r"\'{n}",
        u"\u0145": r"\c{N}",
        u"\u0146": r"\c{n}",
        u"\u0147": r"\v{N}",
        u"\u0148": r"\v{n}",
        u"\u0149": r"'n",
        u"\u014A": r"\NG{}",
        u"\u014B": r"\ng{}",
        u"\u014C": r"\={O}",
        u"\u014D": r"\={o}",
        # note non-raw string (\u confuses Python)
        u"\u014E": "\\u{O}",
        # note non-raw string (\u confuses Python)
        u"\u014F": "\\u{o}",
        u"\u0150": r"\H{O}",
        u"\u0151": r"\H{o}",
        u"\u0152": r"\OE{}",
        u"\u0153": r"\oe{}",
        u"\u0154": r"\'{R}",
        u"\u0155": r"\'{r}",
        u"\u0156": r"\c{R}",
        u"\u0157": r"\c{r}",
        u"\u0158": r"\v{R}",
        u"\u0159": r"\v{r}",
        u"\u015A": r"\'{S}",
        u"\u015B": r"\'{s}",
        u"\u015C": r"\^{S}",
        u"\u015D": r"\^{s}",
        u"\u015E": r"\c{S}",
        u"\u015F": r"\c{s}",
        u"\u0160": r"\v{S}",
        u"\u0161": r"\v{s}",
        u"\u0162": r"\c{T}",
        u"\u0163": r"\c{t}",
        u"\u0164": r"\v{T}",
        u"\u0165": r"\v{t}",
        u"\u0166": r"{\fontencoding{LELA}\selectfont\char47}",
        u"\u0167": r"{\fontencoding{LELA}\selectfont\char63}",
        u"\u0168": r"\~{U}",
        u"\u0169": r"\~{u}",
        u"\u016A": r"\={U}",
        u"\u016B": r"\={u}",
        # note non-raw string (\u confuses Python)
        u"\u016C": "\\u{U}",
        # note non-raw string (\u confuses Python)
        u"\u016D": "\\u{u}",
        u"\u016E": r"\r{U}",
        u"\u016F": r"\r{u}",
        u"\u0170": r"\H{U}",
        u"\u0171": r"\H{u}",
        u"\u0172": r"\k{U}",
        u"\u0173": r"\k{u}",
        u"\u0174": r"\^{W}",
        u"\u0175": r"\^{w}",
        u"\u0176": r"\^{Y}",
        u"\u0177": r"\^{y}",
        # note single-quoted string
        u"\u0178": r'\"{Y}',
        u"\u0179": r"\'{Z}",
        u"\u017A": r"\'{z}",
        u"\u017B": r"\.{Z}",
        u"\u017C": r"\.{z}",
        u"\u017D": r"\v{Z}",
        u"\u017E": r"\v{z}",
        u"\u0195": r"\texthvlig{}",
        u"\u019E": r"\textnrleg{}",
        u"\u01AA": r"\eth{}",
        u"\u01BA": r"{\fontencoding{LELA}\selectfont\char195}",
        u"\u01C2": r"\textdoublepipe{}",
        u"\u01F5": r"\'{g}",
        u"\u0250": r"\Elztrna{}",
        u"\u0252": r"\Elztrnsa{}",
        u"\u0254": r"\Elzopeno{}",
        u"\u0256": r"\Elzrtld{}",
        u"\u0258": r"{\fontencoding{LEIP}\selectfont\char61}",
        u"\u0259": r"\Elzschwa{}",
        u"\u025B": r"\varepsilon{}",
        u"\u0263": r"\Elzpgamma{}",
        u"\u0264": r"\Elzpbgam{}",
        u"\u0265": r"\Elztrnh{}",
        u"\u026C": r"\Elzbtdl{}",
        u"\u026D": r"\Elzrtll{}",
        u"\u026F": r"\Elztrnm{}",
        u"\u0270": r"\Elztrnmlr{}",
        u"\u0271": r"\Elzltlmr{}",
        u"\u0272": r"\Elzltln{}",
        u"\u0273": r"\Elzrtln{}",
        u"\u0277": r"\Elzclomeg{}",
        u"\u0278": r"\textphi{}",
        u"\u0279": r"\Elztrnr{}",
        u"\u027A": r"\Elztrnrl{}",
        u"\u027B": r"\Elzrttrnr{}",
        u"\u027C": r"\Elzrl{}",
        u"\u027D": r"\Elzrtlr{}",
        u"\u027E": r"\Elzfhr{}",
        u"\u027F": r"{\fontencoding{LEIP}\selectfont\char202}",
        u"\u0282": r"\Elzrtls{}",
        u"\u0283": r"\Elzesh{}",
        u"\u0287": r"\Elztrnt{}",
        u"\u0288": r"\Elzrtlt{}",
        u"\u028A": r"\Elzpupsil{}",
        u"\u028B": r"\Elzpscrv{}",
        u"\u028C": r"\Elzinvv{}",
        u"\u028D": r"\Elzinvw{}",
        u"\u028E": r"\Elztrny{}",
        u"\u0290": r"\Elzrtlz{}",
        u"\u0292": r"\Elzyogh{}",
        u"\u0294": r"\Elzglst{}",
        u"\u0295": r"\Elzreglst{}",
        u"\u0296": r"\Elzinglst{}",
        u"\u029E": r"\textturnk{}",
        u"\u02A4": r"\Elzdyogh{}",
        u"\u02A7": r"\Elztesh{}",
        u"\u02C7": r"\textasciicaron{}",
        u"\u02C8": r"\Elzverts{}",
        u"\u02CC": r"\Elzverti{}",
        u"\u02D0": r"\Elzlmrk{}",
        u"\u02D1": r"\Elzhlmrk{}",
        u"\u02D2": r"\Elzsbrhr{}",
        u"\u02D3": r"\Elzsblhr{}",
        u"\u02D4": r"\Elzrais{}",
        u"\u02D5": r"\Elzlow{}",
        u"\u02D8": r"\textasciibreve{}",
        u"\u02D9": r"\textperiodcentered{}",
        u"\u02DA": r"\r{}",
        u"\u02DB": r"\k{}",
        u"\u02DC": r"\texttildelow{}",
        u"\u02DD": r"\H{}",
        u"\u02E5": r"\tone{55}",
        u"\u02E6": r"\tone{44}",
        u"\u02E7": r"\tone{33}",
        u"\u02E8": r"\tone{22}",
        u"\u02E9": r"\tone{11}",
        u"\u0300": r"\`",
        u"\u0301": r"\'",
        u"\u0302": r"\^",
        u"\u0303": r"\~",
        u"\u0304": r"\=",
        # note non-raw string (\u confuses Python)
        u"\u0306": "\\u",
        u"\u0307": r"\.",
        # note single-quoted string
        u"\u0308": r'\"',
        u"\u030A": r"\r",
        u"\u030B": r"\H",
        u"\u030C": r"\v",
        u"\u030F": r"\cyrchar\C",
        u"\u0311": r"{\fontencoding{LECO}\selectfont\char177}",
        u"\u0318": r"{\fontencoding{LECO}\selectfont\char184}",
        u"\u0319": r"{\fontencoding{LECO}\selectfont\char185}",
        u"\u0321": r"\Elzpalh{}",
        u"\u0322": r"\Elzrh{}",
        u"\u0327": r"\c",
        u"\u0328": r"\k",
        u"\u032A": r"\Elzsbbrg{}",
        u"\u032B": r"{\fontencoding{LECO}\selectfont\char203}",
        u"\u032F": r"{\fontencoding{LECO}\selectfont\char207}",
        u"\u0335": r"\Elzxl{}",
        u"\u0336": r"\Elzbar{}",
        u"\u0337": r"{\fontencoding{LECO}\selectfont\char215}",
        u"\u0338": r"{\fontencoding{LECO}\selectfont\char216}",
        u"\u033A": r"{\fontencoding{LECO}\selectfont\char218}",
        u"\u033B": r"{\fontencoding{LECO}\selectfont\char219}",
        u"\u033C": r"{\fontencoding{LECO}\selectfont\char220}",
        u"\u033D": r"{\fontencoding{LECO}\selectfont\char221}",
        u"\u0361": r"{\fontencoding{LECO}\selectfont\char225}",
        u"\u0386": r"\'{A}",
        u"\u0388": r"\'{E}",
        u"\u0389": r"\'{H}",
        u"\u038A": r"\'{}{I}",
        u"\u038C": r"\'{}O",
        u"\u038E": r"\mathrm{'Y}",
        u"\u038F": r"\mathrm{'\Omega}",
        u"\u0390": r"\acute{\ddot{\iota}}",
        u"\u0391": r"\ensuremath{\Alpha}",
        u"\u0392": r"\ensuremath{\Beta}",
        u"\u0393": r"\ensuremath{\Gamma}",
        u"\u0394": r"\ensuremath{\Delta}",
        u"\u0395": r"\ensuremath{\Epsilon}",
        u"\u0396": r"\ensuremath{\Zeta}",
        u"\u0397": r"\ensuremath{\Eta}",
        u"\u0398": r"\ensuremath{\Theta}",
        u"\u0399": r"\ensuremath{\Iota}",
        u"\u039A": r"\ensuremath{\Kappa}",
        u"\u039B": r"\ensuremath{\Lambda}",
        u"\u039E": r"\ensuremath{\Xi}",
        u"\u03A0": r"\ensuremath{\Pi}",
        u"\u03A1": r"\ensuremath{\Rho}",
        u"\u03A3": r"\ensuremath{\Sigma}",
        u"\u03A4": r"\ensuremath{\Tau}",
        # note non-raw string (\u confuses Python)
        u"\u03A5": "\\ensuremath{\\Upsilon}",
        u"\u03A6": r"\ensuremath{\Phi}",
        u"\u03A7": r"\ensuremath{\Chi}",
        u"\u03A8": r"\ensuremath{\Psi}",
        u"\u03A9": r"\ensuremath{\Omega}",
        u"\u03AA": r"\mathrm{\ddot{I}}",
        u"\u03AB": r"\mathrm{\ddot{Y}}",
        u"\u03AC": r"\'{$\alpha$}",
        u"\u03AD": r"\acute{\epsilon}",
        u"\u03AE": r"\acute{\eta}",
        u"\u03AF": r"\acute{\iota}",
        # note non-raw string (\u confuses Python)
        u"\u03B0": "\\acute{\\ddot{\\upsilon}}",
        u"\u03B1": r"\ensuremath{\alpha}",
        u"\u03B2": r"\ensuremath{\beta}",
        u"\u03B3": r"\ensuremath{\gamma}",
        u"\u03B4": r"\ensuremath{\delta}",
        u"\u03B5": r"\ensuremath{\epsilon}",
        u"\u03B6": r"\ensuremath{\zeta}",
        u"\u03B7": r"\ensuremath{\eta}",
        u"\u03B8": r"\texttheta{}",
        u"\u03B9": r"\ensuremath{\iota}",
        u"\u03BA": r"\ensuremath{\kappa}",
        u"\u03BB": r"\ensuremath{\lambda}",
        u"\u03BC": r"\ensuremath{\mu}",
        u"\u03BD": r"\ensuremath{\nu}",
        u"\u03BE": r"\ensuremath{\xi}",
        u"\u03C0": r"\ensuremath{\pi}",
        u"\u03C1": r"\ensuremath{\rho}",
        u"\u03C2": r"\ensuremath{\varsigma}",
        u"\u03C3": r"\ensuremath{\sigma}",
        u"\u03C4": r"\ensuremath{\tau}",
        # note non-raw string (\u confuses Python)
        u"\u03C5": "\\ensuremath{\\upsilon}",
        u"\u03C6": r"\ensuremath{\varphi}",
        u"\u03C7": r"\ensuremath{\chi}",
        u"\u03C8": r"\ensuremath{\psi}",
        u"\u03C9": r"\ensuremath{\omega}",
        u"\u03CA": r"\ensuremath{\ddot{\iota}}",
        # note non-raw string (\u confuses Python)
        u"\u03CB": "\\ensuremath{\\ddot{\\upsilon}}",
        u"\u03CC": r"\'{o}",
        # note non-raw string (\u confuses Python)
        u"\u03CD": "\\acute{\\upsilon}",
        u"\u03CE": r"\acute{\omega}",
        u"\u03D0": r"\Pisymbol{ppi022}{87}",
        u"\u03D1": r"\textvartheta{}",
        # note non-raw string (\u confuses Python)
        u"\u03D2": "\\ensuremath{\\Upsilon}",
        u"\u03D5": r"\ensuremath{\phi}",
        u"\u03D6": r"\ensuremath{\varpi}",
        u"\u03DA": r"\Stigma{}",
        u"\u03DC": r"\Digamma{}",
        u"\u03DD": r"\digamma{}",
        u"\u03DE": r"\Koppa{}",
        u"\u03E0": r"\Sampi{}",
        u"\u03F0": r"\varkappa{}",
        u"\u03F1": r"\varrho{}",
        u"\u03F4": r"\textTheta{}",
        u"\u03F6": r"\backepsilon{}",
        u"\u0401": r"\cyrchar\CYRYO{}",
        u"\u0402": r"\cyrchar\CYRDJE{}",
        u"\u0403": r"\cyrchar{\'\CYRG}",
        u"\u0404": r"\cyrchar\CYRIE{}",
        u"\u0405": r"\cyrchar\CYRDZE{}",
        u"\u0406": r"\cyrchar\CYRII{}",
        u"\u0407": r"\cyrchar\CYRYI{}",
        u"\u0408": r"\cyrchar\CYRJE{}",
        u"\u0409": r"\cyrchar\CYRLJE{}",
        u"\u040A": r"\cyrchar\CYRNJE{}",
        u"\u040B": r"\cyrchar\CYRTSHE{}",
        u"\u040C": r"\cyrchar{\'\CYRK}",
        u"\u040E": r"\cyrchar\CYRUSHRT{}",
        u"\u040F": r"\cyrchar\CYRDZHE{}",
        u"\u0410": r"\cyrchar\CYRA{}",
        u"\u0411": r"\cyrchar\CYRB{}",
        u"\u0412": r"\cyrchar\CYRV{}",
        u"\u0413": r"\cyrchar\CYRG{}",
        u"\u0414": r"\cyrchar\CYRD{}",
        u"\u0415": r"\cyrchar\CYRE{}",
        u"\u0416": r"\cyrchar\CYRZH{}",
        u"\u0417": r"\cyrchar\CYRZ{}",
        u"\u0418": r"\cyrchar\CYRI{}",
        u"\u0419": r"\cyrchar\CYRISHRT{}",
        u"\u041A": r"\cyrchar\CYRK{}",
        u"\u041B": r"\cyrchar\CYRL{}",
        u"\u041C": r"\cyrchar\CYRM{}",
        u"\u041D": r"\cyrchar\CYRN{}",
        u"\u041E": r"\cyrchar\CYRO{}",
        u"\u041F": r"\cyrchar\CYRP{}",
        u"\u0420": r"\cyrchar\CYRR{}",
        u"\u0421": r"\cyrchar\CYRS{}",
        u"\u0422": r"\cyrchar\CYRT{}",
        u"\u0423": r"\cyrchar\CYRU{}",
        u"\u0424": r"\cyrchar\CYRF{}",
        u"\u0425": r"\cyrchar\CYRH{}",
        u"\u0426": r"\cyrchar\CYRC{}",
        u"\u0427": r"\cyrchar\CYRCH{}",
        u"\u0428": r"\cyrchar\CYRSH{}",
        u"\u0429": r"\cyrchar\CYRSHCH{}",
        u"\u042A": r"\cyrchar\CYRHRDSN{}",
        u"\u042B": r"\cyrchar\CYRERY{}",
        u"\u042C": r"\cyrchar\CYRSFTSN{}",
        u"\u042D": r"\cyrchar\CYREREV{}",
        u"\u042E": r"\cyrchar\CYRYU{}",
        u"\u042F": r"\cyrchar\CYRYA{}",
        u"\u0430": r"\cyrchar\cyra{}",
        u"\u0431": r"\cyrchar\cyrb{}",
        u"\u0432": r"\cyrchar\cyrv{}",
        u"\u0433": r"\cyrchar\cyrg{}",
        u"\u0434": r"\cyrchar\cyrd{}",
        u"\u0435": r"\cyrchar\cyre{}",
        u"\u0436": r"\cyrchar\cyrzh{}",
        u"\u0437": r"\cyrchar\cyrz{}",
        u"\u0438": r"\cyrchar\cyri{}",
        u"\u0439": r"\cyrchar\cyrishrt{}",
        u"\u043A": r"\cyrchar\cyrk{}",
        u"\u043B": r"\cyrchar\cyrl{}",
        u"\u043C": r"\cyrchar\cyrm{}",
        u"\u043D": r"\cyrchar\cyrn{}",
        u"\u043E": r"\cyrchar\cyro{}",
        u"\u043F": r"\cyrchar\cyrp{}",
        u"\u0440": r"\cyrchar\cyrr{}",
        u"\u0441": r"\cyrchar\cyrs{}",
        u"\u0442": r"\cyrchar\cyrt{}",
        u"\u0443": r"\cyrchar\cyru{}",
        u"\u0444": r"\cyrchar\cyrf{}",
        u"\u0445": r"\cyrchar\cyrh{}",
        u"\u0446": r"\cyrchar\cyrc{}",
        u"\u0447": r"\cyrchar\cyrch{}",
        u"\u0448": r"\cyrchar\cyrsh{}",
        u"\u0449": r"\cyrchar\cyrshch{}",
        u"\u044A": r"\cyrchar\cyrhrdsn{}",
        u"\u044B": r"\cyrchar\cyrery{}",
        u"\u044C": r"\cyrchar\cyrsftsn{}",
        u"\u044D": r"\cyrchar\cyrerev{}",
        u"\u044E": r"\cyrchar\cyryu{}",
        u"\u044F": r"\cyrchar\cyrya{}",
        u"\u0451": r"\cyrchar\cyryo{}",
        u"\u0452": r"\cyrchar\cyrdje{}",
        u"\u0453": r"\cyrchar{\'\cyrg}",
        u"\u0454": r"\cyrchar\cyrie{}",
        u"\u0455": r"\cyrchar\cyrdze{}",
        u"\u0456": r"\cyrchar\cyrii{}",
        u"\u0457": r"\cyrchar\cyryi{}",
        u"\u0458": r"\cyrchar\cyrje{}",
        u"\u0459": r"\cyrchar\cyrlje{}",
        u"\u045A": r"\cyrchar\cyrnje{}",
        u"\u045B": r"\cyrchar\cyrtshe{}",
        u"\u045C": r"\cyrchar{\'\cyrk}",
        u"\u045E": r"\cyrchar\cyrushrt{}",
        u"\u045F": r"\cyrchar\cyrdzhe{}",
        u"\u0460": r"\cyrchar\CYROMEGA{}",
        u"\u0461": r"\cyrchar\cyromega{}",
        u"\u0462": r"\cyrchar\CYRYAT{}",
        u"\u0464": r"\cyrchar\CYRIOTE{}",
        u"\u0465": r"\cyrchar\cyriote{}",
        u"\u0466": r"\cyrchar\CYRLYUS{}",
        u"\u0467": r"\cyrchar\cyrlyus{}",
        u"\u0468": r"\cyrchar\CYRIOTLYUS{}",
        u"\u0469": r"\cyrchar\cyriotlyus{}",
        u"\u046A": r"\cyrchar\CYRBYUS{}",
        u"\u046C": r"\cyrchar\CYRIOTBYUS{}",
        u"\u046D": r"\cyrchar\cyriotbyus{}",
        u"\u046E": r"\cyrchar\CYRKSI{}",
        u"\u046F": r"\cyrchar\cyrksi{}",
        u"\u0470": r"\cyrchar\CYRPSI{}",
        u"\u0471": r"\cyrchar\cyrpsi{}",
        u"\u0472": r"\cyrchar\CYRFITA{}",
        u"\u0474": r"\cyrchar\CYRIZH{}",
        u"\u0478": r"\cyrchar\CYRUK{}",
        u"\u0479": r"\cyrchar\cyruk{}",
        u"\u047A": r"\cyrchar\CYROMEGARND{}",
        u"\u047B": r"\cyrchar\cyromegarnd{}",
        u"\u047C": r"\cyrchar\CYROMEGATITLO{}",
        u"\u047D": r"\cyrchar\cyromegatitlo{}",
        u"\u047E": r"\cyrchar\CYROT{}",
        u"\u047F": r"\cyrchar\cyrot{}",
        u"\u0480": r"\cyrchar\CYRKOPPA{}",
        u"\u0481": r"\cyrchar\cyrkoppa{}",
        u"\u0482": r"\cyrchar\cyrthousands{}",
        u"\u0488": r"\cyrchar\cyrhundredthousands{}",
        u"\u0489": r"\cyrchar\cyrmillions{}",
        u"\u048C": r"\cyrchar\CYRSEMISFTSN{}",
        u"\u048D": r"\cyrchar\cyrsemisftsn{}",
        u"\u048E": r"\cyrchar\CYRRTICK{}",
        u"\u048F": r"\cyrchar\cyrrtick{}",
        u"\u0490": r"\cyrchar\CYRGUP{}",
        u"\u0491": r"\cyrchar\cyrgup{}",
        u"\u0492": r"\cyrchar\CYRGHCRS{}",
        u"\u0493": r"\cyrchar\cyrghcrs{}",
        u"\u0494": r"\cyrchar\CYRGHK{}",
        u"\u0495": r"\cyrchar\cyrghk{}",
        u"\u0496": r"\cyrchar\CYRZHDSC{}",
        u"\u0497": r"\cyrchar\cyrzhdsc{}",
        u"\u0498": r"\cyrchar\CYRZDSC{}",
        u"\u0499": r"\cyrchar\cyrzdsc{}",
        u"\u049A": r"\cyrchar\CYRKDSC{}",
        u"\u049B": r"\cyrchar\cyrkdsc{}",
        u"\u049C": r"\cyrchar\CYRKVCRS{}",
        u"\u049D": r"\cyrchar\cyrkvcrs{}",
        u"\u049E": r"\cyrchar\CYRKHCRS{}",
        u"\u049F": r"\cyrchar\cyrkhcrs{}",
        u"\u04A0": r"\cyrchar\CYRKBEAK{}",
        u"\u04A1": r"\cyrchar\cyrkbeak{}",
        u"\u04A2": r"\cyrchar\CYRNDSC{}",
        u"\u04A3": r"\cyrchar\cyrndsc{}",
        u"\u04A4": r"\cyrchar\CYRNG{}",
        u"\u04A5": r"\cyrchar\cyrng{}",
        u"\u04A6": r"\cyrchar\CYRPHK{}",
        u"\u04A7": r"\cyrchar\cyrphk{}",
        u"\u04A8": r"\cyrchar\CYRABHHA{}",
        u"\u04A9": r"\cyrchar\cyrabhha{}",
        u"\u04AA": r"\cyrchar\CYRSDSC{}",
        u"\u04AB": r"\cyrchar\cyrsdsc{}",
        u"\u04AC": r"\cyrchar\CYRTDSC{}",
        u"\u04AD": r"\cyrchar\cyrtdsc{}",
        u"\u04AE": r"\cyrchar\CYRY{}",
        u"\u04AF": r"\cyrchar\cyry{}",
        u"\u04B0": r"\cyrchar\CYRYHCRS{}",
        u"\u04B1": r"\cyrchar\cyryhcrs{}",
        u"\u04B2": r"\cyrchar\CYRHDSC{}",
        u"\u04B3": r"\cyrchar\cyrhdsc{}",
        u"\u04B4": r"\cyrchar\CYRTETSE{}",
        u"\u04B5": r"\cyrchar\cyrtetse{}",
        u"\u04B6": r"\cyrchar\CYRCHRDSC{}",
        u"\u04B7": r"\cyrchar\cyrchrdsc{}",
        u"\u04B8": r"\cyrchar\CYRCHVCRS{}",
        u"\u04B9": r"\cyrchar\cyrchvcrs{}",
        u"\u04BA": r"\cyrchar\CYRSHHA{}",
        u"\u04BB": r"\cyrchar\cyrshha{}",
        u"\u04BC": r"\cyrchar\CYRABHCH{}",
        u"\u04BD": r"\cyrchar\cyrabhch{}",
        u"\u04BE": r"\cyrchar\CYRABHCHDSC{}",
        u"\u04BF": r"\cyrchar\cyrabhchdsc{}",
        u"\u04C0": r"\cyrchar\CYRpalochka{}",
        u"\u04C3": r"\cyrchar\CYRKHK{}",
        u"\u04C4": r"\cyrchar\cyrkhk{}",
        u"\u04C7": r"\cyrchar\CYRNHK{}",
        u"\u04C8": r"\cyrchar\cyrnhk{}",
        u"\u04CB": r"\cyrchar\CYRCHLDSC{}",
        u"\u04CC": r"\cyrchar\cyrchldsc{}",
        u"\u04D4": r"\cyrchar\CYRAE{}",
        u"\u04D5": r"\cyrchar\cyrae{}",
        u"\u04D8": r"\cyrchar\CYRSCHWA{}",
        u"\u04D9": r"\cyrchar\cyrschwa{}",
        u"\u04E0": r"\cyrchar\CYRABHDZE{}",
        u"\u04E1": r"\cyrchar\cyrabhdze{}",
        u"\u04E8": r"\cyrchar\CYROTLD{}",
        u"\u04E9": r"\cyrchar\cyrotld{}",
        u"\u2002": r"\hspace{0.6em}",
        u"\u2003": r"\hspace{1em}",
        u"\u2004": r"\hspace{0.33em}",
        u"\u2005": r"\hspace{0.25em}",
        u"\u2006": r"\hspace{0.166em}",
        u"\u2007": r"\hphantom{0}",
        u"\u2008": r"\hphantom{,}",
        u"\u2009": r"\hspace{0.167em}",
        u"\u200A": r"\mkern1mu{}",
        u"\u2010": r"-",
        u"\u2013": r"\textendash{}",
        u"\u2014": r"\textemdash{}",
        u"\u2015": r"\rule{1em}{1pt}",
        u"\u2016": r"\Vert{}",
        u"\u2018": r"`",
        u"\u2019": r"'",
        u"\u201B": r"\Elzreapos{}",
        u"\u201C": r"\textquotedblleft{}",
        u"\u201D": r"\textquotedblright{}",
        u"\u201E": r",,",
        u"\u2020": r"\textdagger{}",
        u"\u2021": r"\textdaggerdbl{}",
        u"\u2022": r"\textbullet{}",
        u"\u2025": r"..",
        u"\u2026": r"\ldots{}",
        u"\u2030": r"\textperthousand{}",
        u"\u2031": r"\textpertenthousand{}",
        u"\u2032": r"\ensuremath{'}",
        u"\u2033": r"\ensuremath{''}",
        u"\u2034": r"\ensuremath{'''}",
        u"\u2035": r"\backprime{}",
        u"\u2039": r"\guilsinglleft{}",
        u"\u203A": r"\guilsinglright{}",
        u"\u2057": r"''''",
        u"\u205F": r"\mkern4mu{}",
        u"\u2060": r"\nolinebreak{}",
        u"\u20A7": r"\ensuremath{\Elzpes}",
        u"\u20AC": r"\mbox{\texteuro}{}",
        u"\u20DB": r"\dddot{}",
        u"\u20DC": r"\ddddot{}",
        u"\u2102": r"\mathbb{C}",
        u"\u210A": r"\mathscr{g}",
        u"\u210B": r"\mathscr{H}",
        u"\u210C": r"\mathfrak{H}",
        u"\u210D": r"\mathbb{H}",
        u"\u210F": r"\hslash{}",
        u"\u2110": r"\mathscr{I}",
        u"\u2111": r"\mathfrak{I}",
        u"\u2112": r"\mathscr{L}",
        u"\u2113": r"\mathscr{l}",
        u"\u2115": r"\mathbb{N}",
        u"\u2116": r"\cyrchar\textnumero{}",
        u"\u2118": r"\wp{}",
        u"\u2119": r"\mathbb{P}",
        u"\u211A": r"\mathbb{Q}",
        u"\u211B": r"\mathscr{R}",
        u"\u211C": r"\mathfrak{R}",
        u"\u211D": r"\mathbb{R}",
        u"\u211E": r"\Elzxrat{}",
        u"\u2122": r"\texttrademark{}",
        u"\u2124": r"\mathbb{Z}",
        u"\u2126": r"\Omega{}",
        u"\u2127": r"\mho{}",
        u"\u2128": r"\mathfrak{Z}",
        u"\u2129": r"\ElsevierGlyph{2129}",
        u"\u212B": r"\AA{}",
        u"\u212C": r"\mathscr{B}",
        u"\u212D": r"\mathfrak{C}",
        u"\u212F": r"\mathscr{e}",
        u"\u2130": r"\mathscr{E}",
        u"\u2131": r"\mathscr{F}",
        u"\u2133": r"\mathscr{M}",
        u"\u2134": r"\mathscr{o}",
        u"\u2135": r"\aleph{}",
        u"\u2136": r"\beth{}",
        u"\u2137": r"\gimel{}",
        u"\u2138": r"\daleth{}",
        u"\u2153": r"\textfrac{1}{3}",
        u"\u2154": r"\textfrac{2}{3}",
        u"\u2155": r"\textfrac{1}{5}",
        u"\u2156": r"\textfrac{2}{5}",
        u"\u2157": r"\textfrac{3}{5}",
        u"\u2158": r"\textfrac{4}{5}",
        u"\u2159": r"\textfrac{1}{6}",
        u"\u215A": r"\textfrac{5}{6}",
        u"\u215B": r"\textfrac{1}{8}",
        u"\u215C": r"\textfrac{3}{8}",
        u"\u215D": r"\textfrac{5}{8}",
        u"\u215E": r"\textfrac{7}{8}",
        u"\u2190": r"\leftarrow{}",
        # note non-raw string (\u confuses Python)
        u"\u2191": "\\uparrow{}",
        u"\u2192": r"\rightarrow{}",
        u"\u2193": r"\downarrow{}",
        u"\u2194": r"\leftrightarrow{}",
        # note non-raw string (\u confuses Python)
        u"\u2195": "\\updownarrow{}",
        u"\u2196": r"\nwarrow{}",
        u"\u2197": r"\nearrow{}",
        u"\u2198": r"\searrow{}",
        u"\u2199": r"\swarrow{}",
        u"\u219A": r"\nleftarrow{}",
        u"\u219B": r"\nrightarrow{}",
        u"\u219C": r"\arrowwaveright{}",
        u"\u219D": r"\arrowwaveright{}",
        u"\u219E": r"\twoheadleftarrow{}",
        u"\u21A0": r"\twoheadrightarrow{}",
        u"\u21A2": r"\leftarrowtail{}",
        u"\u21A3": r"\rightarrowtail{}",
        u"\u21A6": r"\mapsto{}",
        u"\u21A9": r"\hookleftarrow{}",
        u"\u21AA": r"\hookrightarrow{}",
        u"\u21AB": r"\looparrowleft{}",
        u"\u21AC": r"\looparrowright{}",
        u"\u21AD": r"\leftrightsquigarrow{}",
        u"\u21AE": r"\nleftrightarrow{}",
        u"\u21B0": r"\Lsh{}",
        u"\u21B1": r"\Rsh{}",
        u"\u21B3": r"\ElsevierGlyph{21B3}",
        u"\u21B6": r"\curvearrowleft{}",
        u"\u21B7": r"\curvearrowright{}",
        u"\u21BA": r"\circlearrowleft{}",
        u"\u21BB": r"\circlearrowright{}",
        u"\u21BC": r"\leftharpoonup{}",
        u"\u21BD": r"\leftharpoondown{}",
        # note non-raw string (\u confuses Python)
        u"\u21BE": "\\upharpoonright{}",
        # note non-raw string (\u confuses Python)
        u"\u21BF": "\\upharpoonleft{}",
        u"\u21C0": r"\rightharpoonup{}",
        u"\u21C1": r"\rightharpoondown{}",
        u"\u21C2": r"\downharpoonright{}",
        u"\u21C3": r"\downharpoonleft{}",
        u"\u21C4": r"\rightleftarrows{}",
        u"\u21C5": r"\dblarrowupdown{}",
        u"\u21C6": r"\leftrightarrows{}",
        u"\u21C7": r"\leftleftarrows{}",
        # note non-raw string (\u confuses Python)
        u"\u21C8": "\\upuparrows{}",
        u"\u21C9": r"\rightrightarrows{}",
        u"\u21CA": r"\downdownarrows{}",
        u"\u21CB": r"\leftrightharpoons{}",
        u"\u21CC": r"\rightleftharpoons{}",
        u"\u21CD": r"\nLeftarrow{}",
        u"\u21CE": r"\nLeftrightarrow{}",
        u"\u21CF": r"\nRightarrow{}",
        u"\u21D0": r"\Leftarrow{}",
        # note non-raw string (\u confuses Python)
        u"\u21D1": "\\uparrow{}",
        u"\u21D2": r"\Rightarrow{}",
        u"\u21D3": r"\Downarrow{}",
        u"\u21D4": r"\Leftrightarrow{}",
        # note non-raw string (\u confuses Python)
        u"\u21D5": "\\updownarrow{}",
        u"\u21DA": r"\Lleftarrow{}",
        u"\u21DB": r"\Rrightarrow{}",
        u"\u21DD": r"\rightsquigarrow{}",
        u"\u21F5": r"\DownArrowUpArrow{}",
        u"\u2200": r"\forall{}",
        u"\u2201": r"\complement{}",
        u"\u2202": r"\partial{}",
        u"\u2203": r"\exists{}",
        u"\u2204": r"\nexists{}",
        u"\u2205": r"\varnothing{}",
        u"\u2207": r"\nabla{}",
        u"\u2208": r"\in{}",
        u"\u2209": r"\not\in{}",
        u"\u220B": r"\ni{}",
        u"\u220C": r"\not\ni{}",
        u"\u220F": r"\prod{}",
        u"\u2210": r"\coprod{}",
        u"\u2211": r"\sum{}",
        u"\u2213": r"\mp{}",
        u"\u2214": r"\dotplus{}",
        u"\u2216": r"\setminus{}",
        u"\u2217": r"{_\ast}",
        u"\u2218": r"\circ{}",
        u"\u2219": r"\bullet{}",
        u"\u221A": r"\surd{}",
        u"\u221D": r"\propto{}",
        u"\u221E": r"\infty{}",
        u"\u221F": r"\rightangle{}",
        u"\u2220": r"\angle{}",
        u"\u2221": r"\measuredangle{}",
        u"\u2222": r"\sphericalangle{}",
        u"\u2223": r"\mid{}",
        u"\u2224": r"\nmid{}",
        u"\u2225": r"\parallel{}",
        u"\u2226": r"\nparallel{}",
        u"\u2227": r"\wedge{}",
        u"\u2228": r"\vee{}",
        u"\u2229": r"\cap{}",
        u"\u222A": r"\cup{}",
        u"\u222B": r"\int{}",
        u"\u222C": r"\int\!\int{}",
        u"\u222D": r"\int\!\int\!\int{}",
        u"\u222E": r"\oint{}",
        u"\u222F": r"\surfintegral{}",
        u"\u2230": r"\volintegral{}",
        u"\u2231": r"\clwintegral{}",
        u"\u2232": r"\ElsevierGlyph{2232}",
        u"\u2233": r"\ElsevierGlyph{2233}",
        u"\u2234": r"\therefore{}",
        u"\u2235": r"\because{}",
        u"\u2237": r"\Colon{}",
        u"\u2238": r"\ElsevierGlyph{2238}",
        u"\u223A": r"\mathbin{{:}\!\!{-}\!\!{:}}",
        u"\u223B": r"\homothetic{}",
        u"\u223C": r"\sim{}",
        u"\u223D": r"\backsim{}",
        u"\u223E": r"\lazysinv{}",
        u"\u2240": r"\wr{}",
        u"\u2241": r"\not\sim{}",
        u"\u2242": r"\ElsevierGlyph{2242}",
        u"\u2243": r"\simeq{}",
        u"\u2244": r"\not\simeq{}",
        u"\u2245": r"\cong{}",
        u"\u2246": r"\approxnotequal{}",
        u"\u2247": r"\not\cong{}",
        u"\u2248": r"\approx{}",
        u"\u2249": r"\not\approx{}",
        u"\u224A": r"\approxeq{}",
        u"\u224B": r"\tildetrpl{}",
        u"\u224C": r"\allequal{}",
        u"\u224D": r"\asymp{}",
        u"\u224E": r"\Bumpeq{}",
        u"\u224F": r"\bumpeq{}",
        u"\u2250": r"\doteq{}",
        u"\u2251": r"\doteqdot{}",
        u"\u2252": r"\fallingdotseq{}",
        u"\u2253": r"\risingdotseq{}",
        u"\u2254": r":=",
        u"\u2255": r"=:",
        u"\u2256": r"\eqcirc{}",
        u"\u2257": r"\circeq{}",
        u"\u2259": r"\estimates{}",
        u"\u225A": r"\ElsevierGlyph{225A}",
        u"\u225B": r"\starequal{}",
        u"\u225C": r"\triangleq{}",
        u"\u225F": r"\ElsevierGlyph{225F}",
        u"\u2260": r"\not =",
        u"\u2261": r"\equiv{}",
        u"\u2262": r"\not\equiv{}",
        u"\u2264": r"\leq{}",
        u"\u2265": r"\geq{}",
        u"\u2266": r"\leqq{}",
        u"\u2267": r"\geqq{}",
        u"\u2268": r"\lneqq{}",
        u"\u2269": r"\gneqq{}",
        u"\u226A": r"\ll{}",
        u"\u226B": r"\gg{}",
        u"\u226C": r"\between{}",
        u"\u226D": r"\not\kern-0.3em\times{}",
        u"\u226E": r"\not&lt;",
        u"\u226F": r"\not&gt;",
        u"\u2270": r"\not\leq{}",
        u"\u2271": r"\not\geq{}",
        u"\u2272": r"\lessequivlnt{}",
        u"\u2273": r"\greaterequivlnt{}",
        u"\u2274": r"\ElsevierGlyph{2274}",
        u"\u2275": r"\ElsevierGlyph{2275}",
        u"\u2276": r"\lessgtr{}",
        u"\u2277": r"\gtrless{}",
        u"\u2278": r"\notlessgreater{}",
        u"\u2279": r"\notgreaterless{}",
        u"\u227A": r"\prec{}",
        u"\u227B": r"\succ{}",
        u"\u227C": r"\preccurlyeq{}",
        u"\u227D": r"\succcurlyeq{}",
        u"\u227E": r"\precapprox{}",
        u"\u227F": r"\succapprox{}",
        u"\u2280": r"\not\prec{}",
        u"\u2281": r"\not\succ{}",
        u"\u2282": r"\subset{}",
        u"\u2283": r"\supset{}",
        u"\u2284": r"\not\subset{}",
        u"\u2285": r"\not\supset{}",
        u"\u2286": r"\subseteq{}",
        u"\u2287": r"\supseteq{}",
        u"\u2288": r"\not\subseteq{}",
        u"\u2289": r"\not\supseteq{}",
        u"\u228A": r"\subsetneq{}",
        u"\u228B": r"\supsetneq{}",
        # note non-raw string (\u confuses Python)
        u"\u228E": "\\uplus{}",
        u"\u228F": r"\sqsubset{}",
        u"\u2290": r"\sqsupset{}",
        u"\u2291": r"\sqsubseteq{}",
        u"\u2292": r"\sqsupseteq{}",
        u"\u2293": r"\sqcap{}",
        u"\u2294": r"\sqcup{}",
        u"\u2295": r"\oplus{}",
        u"\u2296": r"\ominus{}",
        u"\u2297": r"\otimes{}",
        u"\u2298": r"\oslash{}",
        u"\u2299": r"\odot{}",
        u"\u229A": r"\circledcirc{}",
        u"\u229B": r"\circledast{}",
        u"\u229D": r"\circleddash{}",
        u"\u229E": r"\boxplus{}",
        u"\u229F": r"\boxminus{}",
        u"\u22A0": r"\boxtimes{}",
        u"\u22A1": r"\boxdot{}",
        u"\u22A2": r"\vdash{}",
        u"\u22A3": r"\dashv{}",
        u"\u22A4": r"\top{}",
        u"\u22A5": r"\perp{}",
        u"\u22A7": r"\truestate{}",
        u"\u22A8": r"\forcesextra{}",
        u"\u22A9": r"\Vdash{}",
        u"\u22AA": r"\Vvdash{}",
        u"\u22AB": r"\VDash{}",
        u"\u22AC": r"\nvdash{}",
        u"\u22AD": r"\nvDash{}",
        u"\u22AE": r"\nVdash{}",
        u"\u22AF": r"\nVDash{}",
        u"\u22B2": r"\vartriangleleft{}",
        u"\u22B3": r"\vartriangleright{}",
        u"\u22B4": r"\trianglelefteq{}",
        u"\u22B5": r"\trianglerighteq{}",
        u"\u22B6": r"\original{}",
        u"\u22B7": r"\image{}",
        u"\u22B8": r"\multimap{}",
        u"\u22B9": r"\hermitconjmatrix{}",
        u"\u22BA": r"\intercal{}",
        u"\u22BB": r"\veebar{}",
        u"\u22BE": r"\rightanglearc{}",
        u"\u22C0": r"\ElsevierGlyph{22C0}",
        u"\u22C1": r"\ElsevierGlyph{22C1}",
        u"\u22C2": r"\bigcap{}",
        u"\u22C3": r"\bigcup{}",
        u"\u22C4": r"\diamond{}",
        u"\u22C5": r"\cdot{}",
        u"\u22C6": r"\star{}",
        u"\u22C7": r"\divideontimes{}",
        u"\u22C8": r"\bowtie{}",
        u"\u22C9": r"\ltimes{}",
        u"\u22CA": r"\rtimes{}",
        u"\u22CB": r"\leftthreetimes{}",
        u"\u22CC": r"\rightthreetimes{}",
        u"\u22CD": r"\backsimeq{}",
        u"\u22CE": r"\curlyvee{}",
        u"\u22CF": r"\curlywedge{}",
        u"\u22D0": r"\Subset{}",
        u"\u22D1": r"\Supset{}",
        u"\u22D2": r"\Cap{}",
        u"\u22D3": r"\Cup{}",
        u"\u22D4": r"\pitchfork{}",
        u"\u22D6": r"\lessdot{}",
        u"\u22D7": r"\gtrdot{}",
        u"\u22D8": r"\verymuchless{}",
        u"\u22D9": r"\verymuchgreater{}",
        u"\u22DA": r"\lesseqgtr{}",
        u"\u22DB": r"\gtreqless{}",
        u"\u22DE": r"\curlyeqprec{}",
        u"\u22DF": r"\curlyeqsucc{}",
        u"\u22E2": r"\not\sqsubseteq{}",
        u"\u22E3": r"\not\sqsupseteq{}",
        u"\u22E5": r"\Elzsqspne{}",
        u"\u22E6": r"\lnsim{}",
        u"\u22E7": r"\gnsim{}",
        u"\u22E8": r"\precedesnotsimilar{}",
        u"\u22E9": r"\succnsim{}",
        u"\u22EA": r"\ntriangleleft{}",
        u"\u22EB": r"\ntriangleright{}",
        u"\u22EC": r"\ntrianglelefteq{}",
        u"\u22ED": r"\ntrianglerighteq{}",
        u"\u22EE": r"\vdots{}",
        u"\u22EF": r"\cdots{}",
        # note non-raw string (\u confuses Python)
        u"\u22F0": "\\upslopeellipsis{}",
        u"\u22F1": r"\downslopeellipsis{}",
        u"\u2305": r"\barwedge{}",
        u"\u2306": r"\perspcorrespond{}",
        u"\u2308": r"\lceil{}",
        u"\u2309": r"\rceil{}",
        u"\u230A": r"\lfloor{}",
        u"\u230B": r"\rfloor{}",
        u"\u2315": r"\recorder{}",
        # note single-quoted string
        u"\u2316": r'\mathchar"2208',
        # note non-raw string (\u confuses Python)
        u"\u231C": "\\ulcorner{}",
        # note non-raw string (\u confuses Python)
        u"\u231D": "\\urcorner{}",
        u"\u231E": r"\llcorner{}",
        u"\u231F": r"\lrcorner{}",
        u"\u2322": r"\frown{}",
        u"\u2323": r"\smile{}",
        u"\u2329": r"\langle{}",
        u"\u232A": r"\rangle{}",
        u"\u233D": r"\ElsevierGlyph{E838}",
        u"\u23A3": r"\Elzdlcorn{}",
        u"\u23B0": r"\lmoustache{}",
        u"\u23B1": r"\rmoustache{}",
        u"\u2423": r"\textvisiblespace{}",
        u"\u2460": r"\ding{172}",
        u"\u2461": r"\ding{173}",
        u"\u2462": r"\ding{174}",
        u"\u2463": r"\ding{175}",
        u"\u2464": r"\ding{176}",
        u"\u2465": r"\ding{177}",
        u"\u2466": r"\ding{178}",
        u"\u2467": r"\ding{179}",
        u"\u2468": r"\ding{180}",
        u"\u2469": r"\ding{181}",
        u"\u24C8": r"\circledS{}",
        u"\u2506": r"\Elzdshfnc{}",
        u"\u2519": r"\Elzsqfnw{}",
        u"\u2571": r"\diagup{}",
        u"\u25A0": r"\ding{110}",
        u"\u25A1": r"\square{}",
        u"\u25AA": r"\blacksquare{}",
        u"\u25AD": r"\fbox{~~}",
        u"\u25AF": r"\Elzvrecto{}",
        u"\u25B1": r"\ElsevierGlyph{E381}",
        u"\u25B2": r"\ding{115}",
        u"\u25B3": r"\bigtriangleup{}",
        u"\u25B4": r"\blacktriangle{}",
        u"\u25B5": r"\vartriangle{}",
        u"\u25B8": r"\blacktriangleright{}",
        u"\u25B9": r"\triangleright{}",
        u"\u25BC": r"\ding{116}",
        u"\u25BD": r"\bigtriangledown{}",
        u"\u25BE": r"\blacktriangledown{}",
        u"\u25BF": r"\triangledown{}",
        u"\u25C2": r"\blacktriangleleft{}",
        u"\u25C3": r"\triangleleft{}",
        u"\u25C6": r"\ding{117}",
        u"\u25CA": r"\lozenge{}",
        u"\u25CB": r"\bigcirc{}",
        u"\u25CF": r"\ding{108}",
        u"\u25D0": r"\Elzcirfl{}",
        u"\u25D1": r"\Elzcirfr{}",
        u"\u25D2": r"\Elzcirfb{}",
        u"\u25D7": r"\ding{119}",
        u"\u25D8": r"\Elzrvbull{}",
        u"\u25E7": r"\Elzsqfl{}",
        u"\u25E8": r"\Elzsqfr{}",
        u"\u25EA": r"\Elzsqfse{}",
        u"\u25EF": r"\bigcirc{}",
        u"\u2605": r"\ding{72}",
        u"\u2606": r"\ding{73}",
        u"\u260E": r"\ding{37}",
        u"\u261B": r"\ding{42}",
        u"\u261E": r"\ding{43}",
        u"\u263E": r"\rightmoon{}",
        u"\u263F": r"\mercury{}",
        u"\u2640": r"\venus{}",
        u"\u2642": r"\male{}",
        u"\u2643": r"\jupiter{}",
        u"\u2644": r"\saturn{}",
        # note non-raw string (\u confuses Python)
        u"\u2645": "\\uranus{}",
        u"\u2646": r"\neptune{}",
        u"\u2647": r"\pluto{}",
        u"\u2648": r"\aries{}",
        u"\u2649": r"\taurus{}",
        u"\u264A": r"\gemini{}",
        u"\u264B": r"\cancer{}",
        u"\u264C": r"\leo{}",
        u"\u264D": r"\virgo{}",
        u"\u264E": r"\libra{}",
        u"\u264F": r"\scorpio{}",
        u"\u2650": r"\sagittarius{}",
        u"\u2651": r"\capricornus{}",
        u"\u2652": r"\aquarius{}",
        u"\u2653": r"\pisces{}",
        u"\u2660": r"\ding{171}",
        u"\u2662": r"\diamond{}",
        u"\u2663": r"\ding{168}",
        u"\u2665": r"\ding{170}",
        u"\u2666": r"\ding{169}",
        u"\u2669": r"\quarternote{}",
        u"\u266A": r"\eighthnote{}",
        u"\u266D": r"\flat{}",
        u"\u266E": r"\natural{}",
        u"\u266F": r"\sharp{}",
        u"\u2701": r"\ding{33}",
        u"\u2702": r"\ding{34}",
        u"\u2703": r"\ding{35}",
        u"\u2704": r"\ding{36}",
        u"\u2706": r"\ding{38}",
        u"\u2707": r"\ding{39}",
        u"\u2708": r"\ding{40}",
        u"\u2709": r"\ding{41}",
        u"\u270C": r"\ding{44}",
        u"\u270D": r"\ding{45}",
        u"\u270E": r"\ding{46}",
        u"\u270F": r"\ding{47}",
        u"\u2710": r"\ding{48}",
        u"\u2711": r"\ding{49}",
        u"\u2712": r"\ding{50}",
        u"\u2713": r"\ding{51}",
        u"\u2714": r"\ding{52}",
        u"\u2715": r"\ding{53}",
        u"\u2716": r"\ding{54}",
        u"\u2717": r"\ding{55}",
        u"\u2718": r"\ding{56}",
        u"\u2719": r"\ding{57}",
        u"\u271A": r"\ding{58}",
        u"\u271B": r"\ding{59}",
        u"\u271C": r"\ding{60}",
        u"\u271D": r"\ding{61}",
        u"\u271E": r"\ding{62}",
        u"\u271F": r"\ding{63}",
        u"\u2720": r"\ding{64}",
        u"\u2721": r"\ding{65}",
        u"\u2722": r"\ding{66}",
        u"\u2723": r"\ding{67}",
        u"\u2724": r"\ding{68}",
        u"\u2725": r"\ding{69}",
        u"\u2726": r"\ding{70}",
        u"\u2727": r"\ding{71}",
        u"\u2729": r"\ding{73}",
        u"\u272A": r"\ding{74}",
        u"\u272B": r"\ding{75}",
        u"\u272C": r"\ding{76}",
        u"\u272D": r"\ding{77}",
        u"\u272E": r"\ding{78}",
        u"\u272F": r"\ding{79}",
        u"\u2730": r"\ding{80}",
        u"\u2731": r"\ding{81}",
        u"\u2732": r"\ding{82}",
        u"\u2733": r"\ding{83}",
        u"\u2734": r"\ding{84}",
        u"\u2735": r"\ding{85}",
        u"\u2736": r"\ding{86}",
        u"\u2737": r"\ding{87}",
        u"\u2738": r"\ding{88}",
        u"\u2739": r"\ding{89}",
        u"\u273A": r"\ding{90}",
        u"\u273B": r"\ding{91}",
        u"\u273C": r"\ding{92}",
        u"\u273D": r"\ding{93}",
        u"\u273E": r"\ding{94}",
        u"\u273F": r"\ding{95}",
        u"\u2740": r"\ding{96}",
        u"\u2741": r"\ding{97}",
        u"\u2742": r"\ding{98}",
        u"\u2743": r"\ding{99}",
        u"\u2744": r"\ding{100}",
        u"\u2745": r"\ding{101}",
        u"\u2746": r"\ding{102}",
        u"\u2747": r"\ding{103}",
        u"\u2748": r"\ding{104}",
        u"\u2749": r"\ding{105}",
        u"\u274A": r"\ding{106}",
        u"\u274B": r"\ding{107}",
        u"\u274D": r"\ding{109}",
        u"\u274F": r"\ding{111}",
        u"\u2750": r"\ding{112}",
        u"\u2751": r"\ding{113}",
        u"\u2752": r"\ding{114}",
        u"\u2756": r"\ding{118}",
        u"\u2758": r"\ding{120}",
        u"\u2759": r"\ding{121}",
        u"\u275A": r"\ding{122}",
        u"\u275B": r"\ding{123}",
        u"\u275C": r"\ding{124}",
        u"\u275D": r"\ding{125}",
        u"\u275E": r"\ding{126}",
        u"\u2761": r"\ding{161}",
        u"\u2762": r"\ding{162}",
        u"\u2763": r"\ding{163}",
        u"\u2764": r"\ding{164}",
        u"\u2765": r"\ding{165}",
        u"\u2766": r"\ding{166}",
        u"\u2767": r"\ding{167}",
        u"\u2776": r"\ding{182}",
        u"\u2777": r"\ding{183}",
        u"\u2778": r"\ding{184}",
        u"\u2779": r"\ding{185}",
        u"\u277A": r"\ding{186}",
        u"\u277B": r"\ding{187}",
        u"\u277C": r"\ding{188}",
        u"\u277D": r"\ding{189}",
        u"\u277E": r"\ding{190}",
        u"\u277F": r"\ding{191}",
        u"\u2780": r"\ding{192}",
        u"\u2781": r"\ding{193}",
        u"\u2782": r"\ding{194}",
        u"\u2783": r"\ding{195}",
        u"\u2784": r"\ding{196}",
        u"\u2785": r"\ding{197}",
        u"\u2786": r"\ding{198}",
        u"\u2787": r"\ding{199}",
        u"\u2788": r"\ding{200}",
        u"\u2789": r"\ding{201}",
        u"\u278A": r"\ding{202}",
        u"\u278B": r"\ding{203}",
        u"\u278C": r"\ding{204}",
        u"\u278D": r"\ding{205}",
        u"\u278E": r"\ding{206}",
        u"\u278F": r"\ding{207}",
        u"\u2790": r"\ding{208}",
        u"\u2791": r"\ding{209}",
        u"\u2792": r"\ding{210}",
        u"\u2793": r"\ding{211}",
        u"\u2794": r"\ding{212}",
        u"\u2798": r"\ding{216}",
        u"\u2799": r"\ding{217}",
        u"\u279A": r"\ding{218}",
        u"\u279B": r"\ding{219}",
        u"\u279C": r"\ding{220}",
        u"\u279D": r"\ding{221}",
        u"\u279E": r"\ding{222}",
        u"\u279F": r"\ding{223}",
        u"\u27A0": r"\ding{224}",
        u"\u27A1": r"\ding{225}",
        u"\u27A2": r"\ding{226}",
        u"\u27A3": r"\ding{227}",
        u"\u27A4": r"\ding{228}",
        u"\u27A5": r"\ding{229}",
        u"\u27A6": r"\ding{230}",
        u"\u27A7": r"\ding{231}",
        u"\u27A8": r"\ding{232}",
        u"\u27A9": r"\ding{233}",
        u"\u27AA": r"\ding{234}",
        u"\u27AB": r"\ding{235}",
        u"\u27AC": r"\ding{236}",
        u"\u27AD": r"\ding{237}",
        u"\u27AE": r"\ding{238}",
        u"\u27AF": r"\ding{239}",
        u"\u27B1": r"\ding{241}",
        u"\u27B2": r"\ding{242}",
        u"\u27B3": r"\ding{243}",
        u"\u27B4": r"\ding{244}",
        u"\u27B5": r"\ding{245}",
        u"\u27B6": r"\ding{246}",
        u"\u27B7": r"\ding{247}",
        u"\u27B8": r"\ding{248}",
        u"\u27B9": r"\ding{249}",
        u"\u27BA": r"\ding{250}",
        u"\u27BB": r"\ding{251}",
        u"\u27BC": r"\ding{252}",
        u"\u27BD": r"\ding{253}",
        u"\u27BE": r"\ding{254}",
        u"\u27F5": r"\longleftarrow{}",
        u"\u27F6": r"\longrightarrow{}",
        u"\u27F7": r"\longleftrightarrow{}",
        u"\u27F8": r"\Longleftarrow{}",
        u"\u27F9": r"\Longrightarrow{}",
        u"\u27FA": r"\Longleftrightarrow{}",
        u"\u27FC": r"\longmapsto{}",
        u"\u27FF": r"\sim\joinrel\leadsto",
        u"\u2905": r"\ElsevierGlyph{E212}",
        # note non-raw string (\u confuses Python)
        u"\u2912": "\\UpArrowBar{}",
        u"\u2913": r"\DownArrowBar{}",
        u"\u2923": r"\ElsevierGlyph{E20C}",
        u"\u2924": r"\ElsevierGlyph{E20D}",
        u"\u2925": r"\ElsevierGlyph{E20B}",
        u"\u2926": r"\ElsevierGlyph{E20A}",
        u"\u2927": r"\ElsevierGlyph{E211}",
        u"\u2928": r"\ElsevierGlyph{E20E}",
        u"\u2929": r"\ElsevierGlyph{E20F}",
        u"\u292A": r"\ElsevierGlyph{E210}",
        u"\u2933": r"\ElsevierGlyph{E21C}",
        u"\u2936": r"\ElsevierGlyph{E21A}",
        u"\u2937": r"\ElsevierGlyph{E219}",
        u"\u2940": r"\Elolarr{}",
        u"\u2941": r"\Elorarr{}",
        u"\u2942": r"\ElzRlarr{}",
        u"\u2944": r"\ElzrLarr{}",
        u"\u2947": r"\Elzrarrx{}",
        u"\u294E": r"\LeftRightVector{}",
        u"\u294F": r"\RightUpDownVector{}",
        u"\u2950": r"\DownLeftRightVector{}",
        u"\u2951": r"\LeftUpDownVector{}",
        u"\u2952": r"\LeftVectorBar{}",
        u"\u2953": r"\RightVectorBar{}",
        u"\u2954": r"\RightUpVectorBar{}",
        u"\u2955": r"\RightDownVectorBar{}",
        u"\u2956": r"\DownLeftVectorBar{}",
        u"\u2957": r"\DownRightVectorBar{}",
        u"\u2958": r"\LeftUpVectorBar{}",
        u"\u2959": r"\LeftDownVectorBar{}",
        u"\u295A": r"\LeftTeeVector{}",
        u"\u295B": r"\RightTeeVector{}",
        u"\u295C": r"\RightUpTeeVector{}",
        u"\u295D": r"\RightDownTeeVector{}",
        u"\u295E": r"\DownLeftTeeVector{}",
        u"\u295F": r"\DownRightTeeVector{}",
        u"\u2960": r"\LeftUpTeeVector{}",
        u"\u2961": r"\LeftDownTeeVector{}",
        # note non-raw string (\u confuses Python)
        u"\u296E": "\\UpEquilibrium{}",
        u"\u296F": r"\ReverseUpEquilibrium{}",
        u"\u2970": r"\RoundImplies{}",
        u"\u297C": r"\ElsevierGlyph{E214}",
        u"\u297D": r"\ElsevierGlyph{E215}",
        u"\u2980": r"\Elztfnc{}",
        u"\u2985": r"\ElsevierGlyph{3018}",
        u"\u2986": r"\Elroang{}",
        u"\u2993": r"&lt;\kern-0.58em(",
        u"\u2994": r"\ElsevierGlyph{E291}",
        u"\u2999": r"\Elzddfnc{}",
        u"\u299C": r"\Angle{}",
        u"\u29A0": r"\Elzlpargt{}",
        u"\u29B5": r"\ElsevierGlyph{E260}",
        u"\u29B6": r"\ElsevierGlyph{E61B}",
        u"\u29CA": r"\ElzLap{}",
        u"\u29CB": r"\Elzdefas{}",
        u"\u29CF": r"\LeftTriangleBar{}",
        u"\u29D0": r"\RightTriangleBar{}",
        u"\u29DC": r"\ElsevierGlyph{E372}",
        u"\u29EB": r"\blacklozenge{}",
        u"\u29F4": r"\RuleDelayed{}",
        u"\u2A04": r"\Elxuplus{}",
        u"\u2A05": r"\ElzThr{}",
        u"\u2A06": r"\Elxsqcup{}",
        u"\u2A07": r"\ElzInf{}",
        u"\u2A08": r"\ElzSup{}",
        u"\u2A0D": r"\ElzCint{}",
        u"\u2A0F": r"\clockoint{}",
        u"\u2A10": r"\ElsevierGlyph{E395}",
        u"\u2A16": r"\sqrint{}",
        u"\u2A25": r"\ElsevierGlyph{E25A}",
        u"\u2A2A": r"\ElsevierGlyph{E25B}",
        u"\u2A2D": r"\ElsevierGlyph{E25C}",
        u"\u2A2E": r"\ElsevierGlyph{E25D}",
        u"\u2A2F": r"\ElzTimes{}",
        u"\u2A34": r"\ElsevierGlyph{E25E}",
        u"\u2A35": r"\ElsevierGlyph{E25E}",
        u"\u2A3C": r"\ElsevierGlyph{E259}",
        u"\u2A3F": r"\amalg{}",
        u"\u2A53": r"\ElzAnd{}",
        u"\u2A54": r"\ElzOr{}",
        u"\u2A55": r"\ElsevierGlyph{E36E}",
        u"\u2A56": r"\ElOr{}",
        u"\u2A5E": r"\perspcorrespond{}",
        u"\u2A5F": r"\Elzminhat{}",
        u"\u2A63": r"\ElsevierGlyph{225A}",
        u"\u2A6E": r"\stackrel{*}{=}",
        u"\u2A75": r"\Equal{}",
        u"\u2A7D": r"\leqslant{}",
        u"\u2A7E": r"\geqslant{}",
        u"\u2A85": r"\lessapprox{}",
        u"\u2A86": r"\gtrapprox{}",
        u"\u2A87": r"\lneq{}",
        u"\u2A88": r"\gneq{}",
        u"\u2A89": r"\lnapprox{}",
        u"\u2A8A": r"\gnapprox{}",
        u"\u2A8B": r"\lesseqqgtr{}",
        u"\u2A8C": r"\gtreqqless{}",
        u"\u2A95": r"\eqslantless{}",
        u"\u2A96": r"\eqslantgtr{}",
        u"\u2A9D": r"\Pisymbol{ppi020}{117}",
        u"\u2A9E": r"\Pisymbol{ppi020}{105}",
        u"\u2AA1": r"\NestedLessLess{}",
        u"\u2AA2": r"\NestedGreaterGreater{}",
        u"\u2AAF": r"\preceq{}",
        u"\u2AB0": r"\succeq{}",
        u"\u2AB5": r"\precneqq{}",
        u"\u2AB6": r"\succneqq{}",
        u"\u2AB7": r"\precapprox{}",
        u"\u2AB8": r"\succapprox{}",
        u"\u2AB9": r"\precnapprox{}",
        u"\u2ABA": r"\succnapprox{}",
        u"\u2AC5": r"\subseteqq{}",
        u"\u2AC6": r"\supseteqq{}",
        u"\u2ACB": r"\subsetneqq{}",
        u"\u2ACC": r"\supsetneqq{}",
        u"\u2AEB": r"\ElsevierGlyph{E30D}",
        u"\u2AF6": r"\Elztdcol{}",
        u"\u2AFD": r"{{/}\!\!{/}}",
        u"\u300A": r"\ElsevierGlyph{300A}",
        u"\u300B": r"\ElsevierGlyph{300B}",
        u"\u3018": r"\ElsevierGlyph{3018}",
        u"\u3019": r"\ElsevierGlyph{3019}",
        u"\u301A": r"\openbracketleft{}",
        u"\u301B": r"\openbracketright{}",
        u"\uFB00": r"ff",
        u"\uFB01": r"fi",
        u"\uFB02": r"fl",
        u"\uFB03": r"ffi",
        u"\uFB04": r"ffl",
        u"\uD400": r"\mathbf{A}",
        u"\uD401": r"\mathbf{B}",
        u"\uD402": r"\mathbf{C}",
        u"\uD403": r"\mathbf{D}",
        u"\uD404": r"\mathbf{E}",
        u"\uD405": r"\mathbf{F}",
        u"\uD406": r"\mathbf{G}",
        u"\uD407": r"\mathbf{H}",
        u"\uD408": r"\mathbf{I}",
        u"\uD409": r"\mathbf{J}",
        u"\uD40A": r"\mathbf{K}",
        u"\uD40B": r"\mathbf{L}",
        u"\uD40C": r"\mathbf{M}",
        u"\uD40D": r"\mathbf{N}",
        u"\uD40E": r"\mathbf{O}",
        u"\uD40F": r"\mathbf{P}",
        u"\uD410": r"\mathbf{Q}",
        u"\uD411": r"\mathbf{R}",
        u"\uD412": r"\mathbf{S}",
        u"\uD413": r"\mathbf{T}",
        u"\uD414": r"\mathbf{U}",
        u"\uD415": r"\mathbf{V}",
        u"\uD416": r"\mathbf{W}",
        u"\uD417": r"\mathbf{X}",
        u"\uD418": r"\mathbf{Y}",
        u"\uD419": r"\mathbf{Z}",
        u"\uD41A": r"\mathbf{a}",
        u"\uD41B": r"\mathbf{b}",
        u"\uD41C": r"\mathbf{c}",
        u"\uD41D": r"\mathbf{d}",
        u"\uD41E": r"\mathbf{e}",
        u"\uD41F": r"\mathbf{f}",
        u"\uD420": r"\mathbf{g}",
        u"\uD421": r"\mathbf{h}",
        u"\uD422": r"\mathbf{i}",
        u"\uD423": r"\mathbf{j}",
        u"\uD424": r"\mathbf{k}",
        u"\uD425": r"\mathbf{l}",
        u"\uD426": r"\mathbf{m}",
        u"\uD427": r"\mathbf{n}",
        u"\uD428": r"\mathbf{o}",
        u"\uD429": r"\mathbf{p}",
        u"\uD42A": r"\mathbf{q}",
        u"\uD42B": r"\mathbf{r}",
        u"\uD42C": r"\mathbf{s}",
        u"\uD42D": r"\mathbf{t}",
        u"\uD42E": r"\mathbf{u}",
        u"\uD42F": r"\mathbf{v}",
        u"\uD430": r"\mathbf{w}",
        u"\uD431": r"\mathbf{x}",
        u"\uD432": r"\mathbf{y}",
        u"\uD433": r"\mathbf{z}",
        u"\uD434": r"\mathsl{A}",
        u"\uD435": r"\mathsl{B}",
        u"\uD436": r"\mathsl{C}",
        u"\uD437": r"\mathsl{D}",
        u"\uD438": r"\mathsl{E}",
        u"\uD439": r"\mathsl{F}",
        u"\uD43A": r"\mathsl{G}",
        u"\uD43B": r"\mathsl{H}",
        u"\uD43C": r"\mathsl{I}",
        u"\uD43D": r"\mathsl{J}",
        u"\uD43E": r"\mathsl{K}",
        u"\uD43F": r"\mathsl{L}",
        u"\uD440": r"\mathsl{M}",
        u"\uD441": r"\mathsl{N}",
        u"\uD442": r"\mathsl{O}",
        u"\uD443": r"\mathsl{P}",
        u"\uD444": r"\mathsl{Q}",
        u"\uD445": r"\mathsl{R}",
        u"\uD446": r"\mathsl{S}",
        u"\uD447": r"\mathsl{T}",
        u"\uD448": r"\mathsl{U}",
        u"\uD449": r"\mathsl{V}",
        u"\uD44A": r"\mathsl{W}",
        u"\uD44B": r"\mathsl{X}",
        u"\uD44C": r"\mathsl{Y}",
        u"\uD44D": r"\mathsl{Z}",
        u"\uD44E": r"\mathsl{a}",
        u"\uD44F": r"\mathsl{b}",
        u"\uD450": r"\mathsl{c}",
        u"\uD451": r"\mathsl{d}",
        u"\uD452": r"\mathsl{e}",
        u"\uD453": r"\mathsl{f}",
        u"\uD454": r"\mathsl{g}",
        u"\uD456": r"\mathsl{i}",
        u"\uD457": r"\mathsl{j}",
        u"\uD458": r"\mathsl{k}",
        u"\uD459": r"\mathsl{l}",
        u"\uD45A": r"\mathsl{m}",
        u"\uD45B": r"\mathsl{n}",
        u"\uD45C": r"\mathsl{o}",
        u"\uD45D": r"\mathsl{p}",
        u"\uD45E": r"\mathsl{q}",
        u"\uD45F": r"\mathsl{r}",
        u"\uD460": r"\mathsl{s}",
        u"\uD461": r"\mathsl{t}",
        u"\uD462": r"\mathsl{u}",
        u"\uD463": r"\mathsl{v}",
        u"\uD464": r"\mathsl{w}",
        u"\uD465": r"\mathsl{x}",
        u"\uD466": r"\mathsl{y}",
        u"\uD467": r"\mathsl{z}",
        u"\uD468": r"\mathbit{A}",
        u"\uD469": r"\mathbit{B}",
        u"\uD46A": r"\mathbit{C}",
        u"\uD46B": r"\mathbit{D}",
        u"\uD46C": r"\mathbit{E}",
        u"\uD46D": r"\mathbit{F}",
        u"\uD46E": r"\mathbit{G}",
        u"\uD46F": r"\mathbit{H}",
        u"\uD470": r"\mathbit{I}",
        u"\uD471": r"\mathbit{J}",
        u"\uD472": r"\mathbit{K}",
        u"\uD473": r"\mathbit{L}",
        u"\uD474": r"\mathbit{M}",
        u"\uD475": r"\mathbit{N}",
        u"\uD476": r"\mathbit{O}",
        u"\uD477": r"\mathbit{P}",
        u"\uD478": r"\mathbit{Q}",
        u"\uD479": r"\mathbit{R}",
        u"\uD47A": r"\mathbit{S}",
        u"\uD47B": r"\mathbit{T}",
        u"\uD47C": r"\mathbit{U}",
        u"\uD47D": r"\mathbit{V}",
        u"\uD47E": r"\mathbit{W}",
        u"\uD47F": r"\mathbit{X}",
        u"\uD480": r"\mathbit{Y}",
        u"\uD481": r"\mathbit{Z}",
        u"\uD482": r"\mathbit{a}",
        u"\uD483": r"\mathbit{b}",
        u"\uD484": r"\mathbit{c}",
        u"\uD485": r"\mathbit{d}",
        u"\uD486": r"\mathbit{e}",
        u"\uD487": r"\mathbit{f}",
        u"\uD488": r"\mathbit{g}",
        u"\uD489": r"\mathbit{h}",
        u"\uD48A": r"\mathbit{i}",
        u"\uD48B": r"\mathbit{j}",
        u"\uD48C": r"\mathbit{k}",
        u"\uD48D": r"\mathbit{l}",
        u"\uD48E": r"\mathbit{m}",
        u"\uD48F": r"\mathbit{n}",
        u"\uD490": r"\mathbit{o}",
        u"\uD491": r"\mathbit{p}",
        u"\uD492": r"\mathbit{q}",
        u"\uD493": r"\mathbit{r}",
        u"\uD494": r"\mathbit{s}",
        u"\uD495": r"\mathbit{t}",
        u"\uD496": r"\mathbit{u}",
        u"\uD497": r"\mathbit{v}",
        u"\uD498": r"\mathbit{w}",
        u"\uD499": r"\mathbit{x}",
        u"\uD49A": r"\mathbit{y}",
        u"\uD49B": r"\mathbit{z}",
        u"\uD49C": r"\mathscr{A}",
        u"\uD49E": r"\mathscr{C}",
        u"\uD49F": r"\mathscr{D}",
        u"\uD4A2": r"\mathscr{G}",
        u"\uD4A5": r"\mathscr{J}",
        u"\uD4A6": r"\mathscr{K}",
        u"\uD4A9": r"\mathscr{N}",
        u"\uD4AA": r"\mathscr{O}",
        u"\uD4AB": r"\mathscr{P}",
        u"\uD4AC": r"\mathscr{Q}",
        u"\uD4AE": r"\mathscr{S}",
        u"\uD4AF": r"\mathscr{T}",
        u"\uD4B0": r"\mathscr{U}",
        u"\uD4B1": r"\mathscr{V}",
        u"\uD4B2": r"\mathscr{W}",
        u"\uD4B3": r"\mathscr{X}",
        u"\uD4B4": r"\mathscr{Y}",
        u"\uD4B5": r"\mathscr{Z}",
        u"\uD4B6": r"\mathscr{a}",
        u"\uD4B7": r"\mathscr{b}",
        u"\uD4B8": r"\mathscr{c}",
        u"\uD4B9": r"\mathscr{d}",
        u"\uD4BB": r"\mathscr{f}",
        u"\uD4BD": r"\mathscr{h}",
        u"\uD4BE": r"\mathscr{i}",
        u"\uD4BF": r"\mathscr{j}",
        u"\uD4C0": r"\mathscr{k}",
        u"\uD4C1": r"\mathscr{l}",
        u"\uD4C2": r"\mathscr{m}",
        u"\uD4C3": r"\mathscr{n}",
        u"\uD4C5": r"\mathscr{p}",
        u"\uD4C6": r"\mathscr{q}",
        u"\uD4C7": r"\mathscr{r}",
        u"\uD4C8": r"\mathscr{s}",
        u"\uD4C9": r"\mathscr{t}",
        u"\uD4CA": r"\mathscr{u}",
        u"\uD4CB": r"\mathscr{v}",
        u"\uD4CC": r"\mathscr{w}",
        u"\uD4CD": r"\mathscr{x}",
        u"\uD4CE": r"\mathscr{y}",
        u"\uD4CF": r"\mathscr{z}",
        u"\uD4D0": r"\mathmit{A}",
        u"\uD4D1": r"\mathmit{B}",
        u"\uD4D2": r"\mathmit{C}",
        u"\uD4D3": r"\mathmit{D}",
        u"\uD4D4": r"\mathmit{E}",
        u"\uD4D5": r"\mathmit{F}",
        u"\uD4D6": r"\mathmit{G}",
        u"\uD4D7": r"\mathmit{H}",
        u"\uD4D8": r"\mathmit{I}",
        u"\uD4D9": r"\mathmit{J}",
        u"\uD4DA": r"\mathmit{K}",
        u"\uD4DB": r"\mathmit{L}",
        u"\uD4DC": r"\mathmit{M}",
        u"\uD4DD": r"\mathmit{N}",
        u"\uD4DE": r"\mathmit{O}",
        u"\uD4DF": r"\mathmit{P}",
        u"\uD4E0": r"\mathmit{Q}",
        u"\uD4E1": r"\mathmit{R}",
        u"\uD4E2": r"\mathmit{S}",
        u"\uD4E3": r"\mathmit{T}",
        u"\uD4E4": r"\mathmit{U}",
        u"\uD4E5": r"\mathmit{V}",
        u"\uD4E6": r"\mathmit{W}",
        u"\uD4E7": r"\mathmit{X}",
        u"\uD4E8": r"\mathmit{Y}",
        u"\uD4E9": r"\mathmit{Z}",
        u"\uD4EA": r"\mathmit{a}",
        u"\uD4EB": r"\mathmit{b}",
        u"\uD4EC": r"\mathmit{c}",
        u"\uD4ED": r"\mathmit{d}",
        u"\uD4EE": r"\mathmit{e}",
        u"\uD4EF": r"\mathmit{f}",
        u"\uD4F0": r"\mathmit{g}",
        u"\uD4F1": r"\mathmit{h}",
        u"\uD4F2": r"\mathmit{i}",
        u"\uD4F3": r"\mathmit{j}",
        u"\uD4F4": r"\mathmit{k}",
        u"\uD4F5": r"\mathmit{l}",
        u"\uD4F6": r"\mathmit{m}",
        u"\uD4F7": r"\mathmit{n}",
        u"\uD4F8": r"\mathmit{o}",
        u"\uD4F9": r"\mathmit{p}",
        u"\uD4FA": r"\mathmit{q}",
        u"\uD4FB": r"\mathmit{r}",
        u"\uD4FC": r"\mathmit{s}",
        u"\uD4FD": r"\mathmit{t}",
        u"\uD4FE": r"\mathmit{u}",
        u"\uD4FF": r"\mathmit{v}",
        u"\uD500": r"\mathmit{w}",
        u"\uD501": r"\mathmit{x}",
        u"\uD502": r"\mathmit{y}",
        u"\uD503": r"\mathmit{z}",
        u"\uD504": r"\mathfrak{A}",
        u"\uD505": r"\mathfrak{B}",
        u"\uD507": r"\mathfrak{D}",
        u"\uD508": r"\mathfrak{E}",
        u"\uD509": r"\mathfrak{F}",
        u"\uD50A": r"\mathfrak{G}",
        u"\uD50D": r"\mathfrak{J}",
        u"\uD50E": r"\mathfrak{K}",
        u"\uD50F": r"\mathfrak{L}",
        u"\uD510": r"\mathfrak{M}",
        u"\uD511": r"\mathfrak{N}",
        u"\uD512": r"\mathfrak{O}",
        u"\uD513": r"\mathfrak{P}",
        u"\uD514": r"\mathfrak{Q}",
        u"\uD516": r"\mathfrak{S}",
        u"\uD517": r"\mathfrak{T}",
        u"\uD518": r"\mathfrak{U}",
        u"\uD519": r"\mathfrak{V}",
        u"\uD51A": r"\mathfrak{W}",
        u"\uD51B": r"\mathfrak{X}",
        u"\uD51C": r"\mathfrak{Y}",
        u"\uD51E": r"\mathfrak{a}",
        u"\uD51F": r"\mathfrak{b}",
        u"\uD520": r"\mathfrak{c}",
        u"\uD521": r"\mathfrak{d}",
        u"\uD522": r"\mathfrak{e}",
        u"\uD523": r"\mathfrak{f}",
        u"\uD524": r"\mathfrak{g}",
        u"\uD525": r"\mathfrak{h}",
        u"\uD526": r"\mathfrak{i}",
        u"\uD527": r"\mathfrak{j}",
        u"\uD528": r"\mathfrak{k}",
        u"\uD529": r"\mathfrak{l}",
        u"\uD52A": r"\mathfrak{m}",
        u"\uD52B": r"\mathfrak{n}",
        u"\uD52C": r"\mathfrak{o}",
        u"\uD52D": r"\mathfrak{p}",
        u"\uD52E": r"\mathfrak{q}",
        u"\uD52F": r"\mathfrak{r}",
        u"\uD530": r"\mathfrak{s}",
        u"\uD531": r"\mathfrak{t}",
        u"\uD532": r"\mathfrak{u}",
        u"\uD533": r"\mathfrak{v}",
        u"\uD534": r"\mathfrak{w}",
        u"\uD535": r"\mathfrak{x}",
        u"\uD536": r"\mathfrak{y}",
        u"\uD537": r"\mathfrak{z}",
        u"\uD538": r"\mathbb{A}",
        u"\uD539": r"\mathbb{B}",
        u"\uD53B": r"\mathbb{D}",
        u"\uD53C": r"\mathbb{E}",
        u"\uD53D": r"\mathbb{F}",
        u"\uD53E": r"\mathbb{G}",
        u"\uD540": r"\mathbb{I}",
        u"\uD541": r"\mathbb{J}",
        u"\uD542": r"\mathbb{K}",
        u"\uD543": r"\mathbb{L}",
        u"\uD544": r"\mathbb{M}",
        u"\uD546": r"\mathbb{O}",
        u"\uD54A": r"\mathbb{S}",
        u"\uD54B": r"\mathbb{T}",
        u"\uD54C": r"\mathbb{U}",
        u"\uD54D": r"\mathbb{V}",
        u"\uD54E": r"\mathbb{W}",
        u"\uD54F": r"\mathbb{X}",
        u"\uD550": r"\mathbb{Y}",
        u"\uD552": r"\mathbb{a}",
        u"\uD553": r"\mathbb{b}",
        u"\uD554": r"\mathbb{c}",
        u"\uD555": r"\mathbb{d}",
        u"\uD556": r"\mathbb{e}",
        u"\uD557": r"\mathbb{f}",
        u"\uD558": r"\mathbb{g}",
        u"\uD559": r"\mathbb{h}",
        u"\uD55A": r"\mathbb{i}",
        u"\uD55B": r"\mathbb{j}",
        u"\uD55C": r"\mathbb{k}",
        u"\uD55D": r"\mathbb{l}",
        u"\uD55E": r"\mathbb{m}",
        u"\uD55F": r"\mathbb{n}",
        u"\uD560": r"\mathbb{o}",
        u"\uD561": r"\mathbb{p}",
        u"\uD562": r"\mathbb{q}",
        u"\uD563": r"\mathbb{r}",
        u"\uD564": r"\mathbb{s}",
        u"\uD565": r"\mathbb{t}",
        u"\uD566": r"\mathbb{u}",
        u"\uD567": r"\mathbb{v}",
        u"\uD568": r"\mathbb{w}",
        u"\uD569": r"\mathbb{x}",
        u"\uD56A": r"\mathbb{y}",
        u"\uD56B": r"\mathbb{z}",
        u"\uD56C": r"\mathslbb{A}",
        u"\uD56D": r"\mathslbb{B}",
        u"\uD56E": r"\mathslbb{C}",
        u"\uD56F": r"\mathslbb{D}",
        u"\uD570": r"\mathslbb{E}",
        u"\uD571": r"\mathslbb{F}",
        u"\uD572": r"\mathslbb{G}",
        u"\uD573": r"\mathslbb{H}",
        u"\uD574": r"\mathslbb{I}",
        u"\uD575": r"\mathslbb{J}",
        u"\uD576": r"\mathslbb{K}",
        u"\uD577": r"\mathslbb{L}",
        u"\uD578": r"\mathslbb{M}",
        u"\uD579": r"\mathslbb{N}",
        u"\uD57A": r"\mathslbb{O}",
        u"\uD57B": r"\mathslbb{P}",
        u"\uD57C": r"\mathslbb{Q}",
        u"\uD57D": r"\mathslbb{R}",
        u"\uD57E": r"\mathslbb{S}",
        u"\uD57F": r"\mathslbb{T}",
        u"\uD580": r"\mathslbb{U}",
        u"\uD581": r"\mathslbb{V}",
        u"\uD582": r"\mathslbb{W}",
        u"\uD583": r"\mathslbb{X}",
        u"\uD584": r"\mathslbb{Y}",
        u"\uD585": r"\mathslbb{Z}",
        u"\uD586": r"\mathslbb{a}",
        u"\uD587": r"\mathslbb{b}",
        u"\uD588": r"\mathslbb{c}",
        u"\uD589": r"\mathslbb{d}",
        u"\uD58A": r"\mathslbb{e}",
        u"\uD58B": r"\mathslbb{f}",
        u"\uD58C": r"\mathslbb{g}",
        u"\uD58D": r"\mathslbb{h}",
        u"\uD58E": r"\mathslbb{i}",
        u"\uD58F": r"\mathslbb{j}",
        u"\uD590": r"\mathslbb{k}",
        u"\uD591": r"\mathslbb{l}",
        u"\uD592": r"\mathslbb{m}",
        u"\uD593": r"\mathslbb{n}",
        u"\uD594": r"\mathslbb{o}",
        u"\uD595": r"\mathslbb{p}",
        u"\uD596": r"\mathslbb{q}",
        u"\uD597": r"\mathslbb{r}",
        u"\uD598": r"\mathslbb{s}",
        u"\uD599": r"\mathslbb{t}",
        u"\uD59A": r"\mathslbb{u}",
        u"\uD59B": r"\mathslbb{v}",
        u"\uD59C": r"\mathslbb{w}",
        u"\uD59D": r"\mathslbb{x}",
        u"\uD59E": r"\mathslbb{y}",
        u"\uD59F": r"\mathslbb{z}",
        u"\uD5A0": r"\mathsf{A}",
        u"\uD5A1": r"\mathsf{B}",
        u"\uD5A2": r"\mathsf{C}",
        u"\uD5A3": r"\mathsf{D}",
        u"\uD5A4": r"\mathsf{E}",
        u"\uD5A5": r"\mathsf{F}",
        u"\uD5A6": r"\mathsf{G}",
        u"\uD5A7": r"\mathsf{H}",
        u"\uD5A8": r"\mathsf{I}",
        u"\uD5A9": r"\mathsf{J}",
        u"\uD5AA": r"\mathsf{K}",
        u"\uD5AB": r"\mathsf{L}",
        u"\uD5AC": r"\mathsf{M}",
        u"\uD5AD": r"\mathsf{N}",
        u"\uD5AE": r"\mathsf{O}",
        u"\uD5AF": r"\mathsf{P}",
        u"\uD5B0": r"\mathsf{Q}",
        u"\uD5B1": r"\mathsf{R}",
        u"\uD5B2": r"\mathsf{S}",
        u"\uD5B3": r"\mathsf{T}",
        u"\uD5B4": r"\mathsf{U}",
        u"\uD5B5": r"\mathsf{V}",
        u"\uD5B6": r"\mathsf{W}",
        u"\uD5B7": r"\mathsf{X}",
        u"\uD5B8": r"\mathsf{Y}",
        u"\uD5B9": r"\mathsf{Z}",
        u"\uD5BA": r"\mathsf{a}",
        u"\uD5BB": r"\mathsf{b}",
        u"\uD5BC": r"\mathsf{c}",
        u"\uD5BD": r"\mathsf{d}",
        u"\uD5BE": r"\mathsf{e}",
        u"\uD5BF": r"\mathsf{f}",
        u"\uD5C0": r"\mathsf{g}",
        u"\uD5C1": r"\mathsf{h}",
        u"\uD5C2": r"\mathsf{i}",
        u"\uD5C3": r"\mathsf{j}",
        u"\uD5C4": r"\mathsf{k}",
        u"\uD5C5": r"\mathsf{l}",
        u"\uD5C6": r"\mathsf{m}",
        u"\uD5C7": r"\mathsf{n}",
        u"\uD5C8": r"\mathsf{o}",
        u"\uD5C9": r"\mathsf{p}",
        u"\uD5CA": r"\mathsf{q}",
        u"\uD5CB": r"\mathsf{r}",
        u"\uD5CC": r"\mathsf{s}",
        u"\uD5CD": r"\mathsf{t}",
        u"\uD5CE": r"\mathsf{u}",
        u"\uD5CF": r"\mathsf{v}",
        u"\uD5D0": r"\mathsf{w}",
        u"\uD5D1": r"\mathsf{x}",
        u"\uD5D2": r"\mathsf{y}",
        u"\uD5D3": r"\mathsf{z}",
        u"\uD5D4": r"\mathsfbf{A}",
        u"\uD5D5": r"\mathsfbf{B}",
        u"\uD5D6": r"\mathsfbf{C}",
        u"\uD5D7": r"\mathsfbf{D}",
        u"\uD5D8": r"\mathsfbf{E}",
        u"\uD5D9": r"\mathsfbf{F}",
        u"\uD5DA": r"\mathsfbf{G}",
        u"\uD5DB": r"\mathsfbf{H}",
        u"\uD5DC": r"\mathsfbf{I}",
        u"\uD5DD": r"\mathsfbf{J}",
        u"\uD5DE": r"\mathsfbf{K}",
        u"\uD5DF": r"\mathsfbf{L}",
        u"\uD5E0": r"\mathsfbf{M}",
        u"\uD5E1": r"\mathsfbf{N}",
        u"\uD5E2": r"\mathsfbf{O}",
        u"\uD5E3": r"\mathsfbf{P}",
        u"\uD5E4": r"\mathsfbf{Q}",
        u"\uD5E5": r"\mathsfbf{R}",
        u"\uD5E6": r"\mathsfbf{S}",
        u"\uD5E7": r"\mathsfbf{T}",
        u"\uD5E8": r"\mathsfbf{U}",
        u"\uD5E9": r"\mathsfbf{V}",
        u"\uD5EA": r"\mathsfbf{W}",
        u"\uD5EB": r"\mathsfbf{X}",
        u"\uD5EC": r"\mathsfbf{Y}",
        u"\uD5ED": r"\mathsfbf{Z}",
        u"\uD5EE": r"\mathsfbf{a}",
        u"\uD5EF": r"\mathsfbf{b}",
        u"\uD5F0": r"\mathsfbf{c}",
        u"\uD5F1": r"\mathsfbf{d}",
        u"\uD5F2": r"\mathsfbf{e}",
        u"\uD5F3": r"\mathsfbf{f}",
        u"\uD5F4": r"\mathsfbf{g}",
        u"\uD5F5": r"\mathsfbf{h}",
        u"\uD5F6": r"\mathsfbf{i}",
        u"\uD5F7": r"\mathsfbf{j}",
        u"\uD5F8": r"\mathsfbf{k}",
        u"\uD5F9": r"\mathsfbf{l}",
        u"\uD5FA": r"\mathsfbf{m}",
        u"\uD5FB": r"\mathsfbf{n}",
        u"\uD5FC": r"\mathsfbf{o}",
        u"\uD5FD": r"\mathsfbf{p}",
        u"\uD5FE": r"\mathsfbf{q}",
        u"\uD5FF": r"\mathsfbf{r}",
        u"\uD600": r"\mathsfbf{s}",
        u"\uD601": r"\mathsfbf{t}",
        u"\uD602": r"\mathsfbf{u}",
        u"\uD603": r"\mathsfbf{v}",
        u"\uD604": r"\mathsfbf{w}",
        u"\uD605": r"\mathsfbf{x}",
        u"\uD606": r"\mathsfbf{y}",
        u"\uD607": r"\mathsfbf{z}",
        u"\uD608": r"\mathsfsl{A}",
        u"\uD609": r"\mathsfsl{B}",
        u"\uD60A": r"\mathsfsl{C}",
        u"\uD60B": r"\mathsfsl{D}",
        u"\uD60C": r"\mathsfsl{E}",
        u"\uD60D": r"\mathsfsl{F}",
        u"\uD60E": r"\mathsfsl{G}",
        u"\uD60F": r"\mathsfsl{H}",
        u"\uD610": r"\mathsfsl{I}",
        u"\uD611": r"\mathsfsl{J}",
        u"\uD612": r"\mathsfsl{K}",
        u"\uD613": r"\mathsfsl{L}",
        u"\uD614": r"\mathsfsl{M}",
        u"\uD615": r"\mathsfsl{N}",
        u"\uD616": r"\mathsfsl{O}",
        u"\uD617": r"\mathsfsl{P}",
        u"\uD618": r"\mathsfsl{Q}",
        u"\uD619": r"\mathsfsl{R}",
        u"\uD61A": r"\mathsfsl{S}",
        u"\uD61B": r"\mathsfsl{T}",
        u"\uD61C": r"\mathsfsl{U}",
        u"\uD61D": r"\mathsfsl{V}",
        u"\uD61E": r"\mathsfsl{W}",
        u"\uD61F": r"\mathsfsl{X}",
        u"\uD620": r"\mathsfsl{Y}",
        u"\uD621": r"\mathsfsl{Z}",
        u"\uD622": r"\mathsfsl{a}",
        u"\uD623": r"\mathsfsl{b}",
        u"\uD624": r"\mathsfsl{c}",
        u"\uD625": r"\mathsfsl{d}",
        u"\uD626": r"\mathsfsl{e}",
        u"\uD627": r"\mathsfsl{f}",
        u"\uD628": r"\mathsfsl{g}",
        u"\uD629": r"\mathsfsl{h}",
        u"\uD62A": r"\mathsfsl{i}",
        u"\uD62B": r"\mathsfsl{j}",
        u"\uD62C": r"\mathsfsl{k}",
        u"\uD62D": r"\mathsfsl{l}",
        u"\uD62E": r"\mathsfsl{m}",
        u"\uD62F": r"\mathsfsl{n}",
        u"\uD630": r"\mathsfsl{o}",
        u"\uD631": r"\mathsfsl{p}",
        u"\uD632": r"\mathsfsl{q}",
        u"\uD633": r"\mathsfsl{r}",
        u"\uD634": r"\mathsfsl{s}",
        u"\uD635": r"\mathsfsl{t}",
        u"\uD636": r"\mathsfsl{u}",
        u"\uD637": r"\mathsfsl{v}",
        u"\uD638": r"\mathsfsl{w}",
        u"\uD639": r"\mathsfsl{x}",
        u"\uD63A": r"\mathsfsl{y}",
        u"\uD63B": r"\mathsfsl{z}",
        u"\uD63C": r"\mathsfbfsl{A}",
        u"\uD63D": r"\mathsfbfsl{B}",
        u"\uD63E": r"\mathsfbfsl{C}",
        u"\uD63F": r"\mathsfbfsl{D}",
        u"\uD640": r"\mathsfbfsl{E}",
        u"\uD641": r"\mathsfbfsl{F}",
        u"\uD642": r"\mathsfbfsl{G}",
        u"\uD643": r"\mathsfbfsl{H}",
        u"\uD644": r"\mathsfbfsl{I}",
        u"\uD645": r"\mathsfbfsl{J}",
        u"\uD646": r"\mathsfbfsl{K}",
        u"\uD647": r"\mathsfbfsl{L}",
        u"\uD648": r"\mathsfbfsl{M}",
        u"\uD649": r"\mathsfbfsl{N}",
        u"\uD64A": r"\mathsfbfsl{O}",
        u"\uD64B": r"\mathsfbfsl{P}",
        u"\uD64C": r"\mathsfbfsl{Q}",
        u"\uD64D": r"\mathsfbfsl{R}",
        u"\uD64E": r"\mathsfbfsl{S}",
        u"\uD64F": r"\mathsfbfsl{T}",
        u"\uD650": r"\mathsfbfsl{U}",
        u"\uD651": r"\mathsfbfsl{V}",
        u"\uD652": r"\mathsfbfsl{W}",
        u"\uD653": r"\mathsfbfsl{X}",
        u"\uD654": r"\mathsfbfsl{Y}",
        u"\uD655": r"\mathsfbfsl{Z}",
        u"\uD656": r"\mathsfbfsl{a}",
        u"\uD657": r"\mathsfbfsl{b}",
        u"\uD658": r"\mathsfbfsl{c}",
        u"\uD659": r"\mathsfbfsl{d}",
        u"\uD65A": r"\mathsfbfsl{e}",
        u"\uD65B": r"\mathsfbfsl{f}",
        u"\uD65C": r"\mathsfbfsl{g}",
        u"\uD65D": r"\mathsfbfsl{h}",
        u"\uD65E": r"\mathsfbfsl{i}",
        u"\uD65F": r"\mathsfbfsl{j}",
        u"\uD660": r"\mathsfbfsl{k}",
        u"\uD661": r"\mathsfbfsl{l}",
        u"\uD662": r"\mathsfbfsl{m}",
        u"\uD663": r"\mathsfbfsl{n}",
        u"\uD664": r"\mathsfbfsl{o}",
        u"\uD665": r"\mathsfbfsl{p}",
        u"\uD666": r"\mathsfbfsl{q}",
        u"\uD667": r"\mathsfbfsl{r}",
        u"\uD668": r"\mathsfbfsl{s}",
        u"\uD669": r"\mathsfbfsl{t}",
        u"\uD66A": r"\mathsfbfsl{u}",
        u"\uD66B": r"\mathsfbfsl{v}",
        u"\uD66C": r"\mathsfbfsl{w}",
        u"\uD66D": r"\mathsfbfsl{x}",
        u"\uD66E": r"\mathsfbfsl{y}",
        u"\uD66F": r"\mathsfbfsl{z}",
        u"\uD670": r"\mathtt{A}",
        u"\uD671": r"\mathtt{B}",
        u"\uD672": r"\mathtt{C}",
        u"\uD673": r"\mathtt{D}",
        u"\uD674": r"\mathtt{E}",
        u"\uD675": r"\mathtt{F}",
        u"\uD676": r"\mathtt{G}",
        u"\uD677": r"\mathtt{H}",
        u"\uD678": r"\mathtt{I}",
        u"\uD679": r"\mathtt{J}",
        u"\uD67A": r"\mathtt{K}",
        u"\uD67B": r"\mathtt{L}",
        u"\uD67C": r"\mathtt{M}",
        u"\uD67D": r"\mathtt{N}",
        u"\uD67E": r"\mathtt{O}",
        u"\uD67F": r"\mathtt{P}",
        u"\uD680": r"\mathtt{Q}",
        u"\uD681": r"\mathtt{R}",
        u"\uD682": r"\mathtt{S}",
        u"\uD683": r"\mathtt{T}",
        u"\uD684": r"\mathtt{U}",
        u"\uD685": r"\mathtt{V}",
        u"\uD686": r"\mathtt{W}",
        u"\uD687": r"\mathtt{X}",
        u"\uD688": r"\mathtt{Y}",
        u"\uD689": r"\mathtt{Z}",
        u"\uD68A": r"\mathtt{a}",
        u"\uD68B": r"\mathtt{b}",
        u"\uD68C": r"\mathtt{c}",
        u"\uD68D": r"\mathtt{d}",
        u"\uD68E": r"\mathtt{e}",
        u"\uD68F": r"\mathtt{f}",
        u"\uD690": r"\mathtt{g}",
        u"\uD691": r"\mathtt{h}",
        u"\uD692": r"\mathtt{i}",
        u"\uD693": r"\mathtt{j}",
        u"\uD694": r"\mathtt{k}",
        u"\uD695": r"\mathtt{l}",
        u"\uD696": r"\mathtt{m}",
        u"\uD697": r"\mathtt{n}",
        u"\uD698": r"\mathtt{o}",
        u"\uD699": r"\mathtt{p}",
        u"\uD69A": r"\mathtt{q}",
        u"\uD69B": r"\mathtt{r}",
        u"\uD69C": r"\mathtt{s}",
        u"\uD69D": r"\mathtt{t}",
        u"\uD69E": r"\mathtt{u}",
        u"\uD69F": r"\mathtt{v}",
        u"\uD6A0": r"\mathtt{w}",
        u"\uD6A1": r"\mathtt{x}",
        u"\uD6A2": r"\mathtt{y}",
        u"\uD6A3": r"\mathtt{z}",
        u"\uD6A8": r"\mathbf{\Alpha}",
        u"\uD6A9": r"\mathbf{\Beta}",
        u"\uD6AA": r"\mathbf{\Gamma}",
        u"\uD6AB": r"\mathbf{\Delta}",
        u"\uD6AC": r"\mathbf{\Epsilon}",
        u"\uD6AD": r"\mathbf{\Zeta}",
        u"\uD6AE": r"\mathbf{\Eta}",
        u"\uD6AF": r"\mathbf{\Theta}",
        u"\uD6B0": r"\mathbf{\Iota}",
        u"\uD6B1": r"\mathbf{\Kappa}",
        u"\uD6B2": r"\mathbf{\Lambda}",
        u"\uD6B5": r"\mathbf{\Xi}",
        u"\uD6B7": r"\mathbf{\Pi}",
        u"\uD6B8": r"\mathbf{\Rho}",
        u"\uD6B9": r"\mathbf{\vartheta}",
        u"\uD6BA": r"\mathbf{\Sigma}",
        u"\uD6BB": r"\mathbf{\Tau}",
        # note non-raw string (\u confuses Python)
        u"\uD6BC": "\\mathbf{\\Upsilon}",
        u"\uD6BD": r"\mathbf{\Phi}",
        u"\uD6BE": r"\mathbf{\Chi}",
        u"\uD6BF": r"\mathbf{\Psi}",
        u"\uD6C0": r"\mathbf{\Omega}",
        u"\uD6C1": r"\mathbf{\nabla}",
        u"\uD6C2": r"\mathbf{\Alpha}",
        u"\uD6C3": r"\mathbf{\Beta}",
        u"\uD6C4": r"\mathbf{\Gamma}",
        u"\uD6C5": r"\mathbf{\Delta}",
        u"\uD6C6": r"\mathbf{\Epsilon}",
        u"\uD6C7": r"\mathbf{\Zeta}",
        u"\uD6C8": r"\mathbf{\Eta}",
        u"\uD6C9": r"\mathbf{\theta}",
        u"\uD6CA": r"\mathbf{\Iota}",
        u"\uD6CB": r"\mathbf{\Kappa}",
        u"\uD6CC": r"\mathbf{\Lambda}",
        u"\uD6CF": r"\mathbf{\Xi}",
        u"\uD6D1": r"\mathbf{\Pi}",
        u"\uD6D2": r"\mathbf{\Rho}",
        u"\uD6D3": r"\mathbf{\varsigma}",
        u"\uD6D4": r"\mathbf{\Sigma}",
        u"\uD6D5": r"\mathbf{\Tau}",
        # note non-raw string (\u confuses Python)
        u"\uD6D6": "\\mathbf{\\Upsilon}",
        u"\uD6D7": r"\mathbf{\Phi}",
        u"\uD6D8": r"\mathbf{\Chi}",
        u"\uD6D9": r"\mathbf{\Psi}",
        u"\uD6DA": r"\mathbf{\Omega}",
        u"\uD6DB": r"\partial{}",
        u"\uD6DC": r"\in",
        u"\uD6DD": r"\mathbf{\vartheta}",
        u"\uD6DE": r"\mathbf{\varkappa}",
        u"\uD6DF": r"\mathbf{\phi}",
        u"\uD6E0": r"\mathbf{\varrho}",
        u"\uD6E1": r"\mathbf{\varpi}",
        u"\uD6E2": r"\mathsl{\Alpha}",
        u"\uD6E3": r"\mathsl{\Beta}",
        u"\uD6E4": r"\mathsl{\Gamma}",
        u"\uD6E5": r"\mathsl{\Delta}",
        u"\uD6E6": r"\mathsl{\Epsilon}",
        u"\uD6E7": r"\mathsl{\Zeta}",
        u"\uD6E8": r"\mathsl{\Eta}",
        u"\uD6E9": r"\mathsl{\Theta}",
        u"\uD6EA": r"\mathsl{\Iota}",
        u"\uD6EB": r"\mathsl{\Kappa}",
        u"\uD6EC": r"\mathsl{\Lambda}",
        u"\uD6EF": r"\mathsl{\Xi}",
        u"\uD6F1": r"\mathsl{\Pi}",
        u"\uD6F2": r"\mathsl{\Rho}",
        u"\uD6F3": r"\mathsl{\vartheta}",
        u"\uD6F4": r"\mathsl{\Sigma}",
        u"\uD6F5": r"\mathsl{\Tau}",
        # note non-raw string (\u confuses Python)
        u"\uD6F6": "\mathsl{\\Upsilon}",
        u"\uD6F7": r"\mathsl{\Phi}",
        u"\uD6F8": r"\mathsl{\Chi}",
        u"\uD6F9": r"\mathsl{\Psi}",
        u"\uD6FA": r"\mathsl{\Omega}",
        u"\uD6FB": r"\mathsl{\nabla}",
        u"\uD6FC": r"\mathsl{\Alpha}",
        u"\uD6FD": r"\mathsl{\Beta}",
        u"\uD6FE": r"\mathsl{\Gamma}",
        u"\uD6FF": r"\mathsl{\Delta}",
        u"\uD700": r"\mathsl{\Epsilon}",
        u"\uD701": r"\mathsl{\Zeta}",
        u"\uD702": r"\mathsl{\Eta}",
        u"\uD703": r"\mathsl{\Theta}",
        u"\uD704": r"\mathsl{\Iota}",
        u"\uD705": r"\mathsl{\Kappa}",
        u"\uD706": r"\mathsl{\Lambda}",
        u"\uD709": r"\mathsl{\Xi}",
        u"\uD70B": r"\mathsl{\Pi}",
        u"\uD70C": r"\mathsl{\Rho}",
        u"\uD70D": r"\mathsl{\varsigma}",
        u"\uD70E": r"\mathsl{\Sigma}",
        u"\uD70F": r"\mathsl{\Tau}",
        # note non-raw string (\u confuses Python)
        u"\uD710": "\\mathsl{\\Upsilon}",
        u"\uD711": r"\mathsl{\Phi}",
        u"\uD712": r"\mathsl{\Chi}",
        u"\uD713": r"\mathsl{\Psi}",
        u"\uD714": r"\mathsl{\Omega}",
        u"\uD715": r"\partial{}",
        u"\uD716": r"\in",
        u"\uD717": r"\mathsl{\vartheta}",
        u"\uD718": r"\mathsl{\varkappa}",
        u"\uD719": r"\mathsl{\phi}",
        u"\uD71A": r"\mathsl{\varrho}",
        u"\uD71B": r"\mathsl{\varpi}",
        u"\uD71C": r"\mathbit{\Alpha}",
        u"\uD71D": r"\mathbit{\Beta}",
        u"\uD71E": r"\mathbit{\Gamma}",
        u"\uD71F": r"\mathbit{\Delta}",
        u"\uD720": r"\mathbit{\Epsilon}",
        u"\uD721": r"\mathbit{\Zeta}",
        u"\uD722": r"\mathbit{\Eta}",
        u"\uD723": r"\mathbit{\Theta}",
        u"\uD724": r"\mathbit{\Iota}",
        u"\uD725": r"\mathbit{\Kappa}",
        u"\uD726": r"\mathbit{\Lambda}",
        u"\uD729": r"\mathbit{\Xi}",
        u"\uD72B": r"\mathbit{\Pi}",
        u"\uD72C": r"\mathbit{\Rho}",
        u"\uD72D": r"\mathbit{O}",
        u"\uD72E": r"\mathbit{\Sigma}",
        u"\uD72F": r"\mathbit{\Tau}",
        # note non-raw string (\u confuses Python)
        u"\uD730": "\\mathbit{\\Upsilon}",
        u"\uD731": r"\mathbit{\Phi}",
        u"\uD732": r"\mathbit{\Chi}",
        u"\uD733": r"\mathbit{\Psi}",
        u"\uD734": r"\mathbit{\Omega}",
        u"\uD735": r"\mathbit{\nabla}",
        u"\uD736": r"\mathbit{\Alpha}",
        u"\uD737": r"\mathbit{\Beta}",
        u"\uD738": r"\mathbit{\Gamma}",
        u"\uD739": r"\mathbit{\Delta}",
        u"\uD73A": r"\mathbit{\Epsilon}",
        u"\uD73B": r"\mathbit{\Zeta}",
        u"\uD73C": r"\mathbit{\Eta}",
        u"\uD73D": r"\mathbit{\Theta}",
        u"\uD73E": r"\mathbit{\Iota}",
        u"\uD73F": r"\mathbit{\Kappa}",
        u"\uD740": r"\mathbit{\Lambda}",
        u"\uD743": r"\mathbit{\Xi}",
        u"\uD745": r"\mathbit{\Pi}",
        u"\uD746": r"\mathbit{\Rho}",
        u"\uD747": r"\mathbit{\varsigma}",
        u"\uD748": r"\mathbit{\Sigma}",
        u"\uD749": r"\mathbit{\Tau}",
        # note non-raw string (\u confuses Python)
        u"\uD74A": "\\mathbit{\\Upsilon}",
        u"\uD74B": r"\mathbit{\Phi}",
        u"\uD74C": r"\mathbit{\Chi}",
        u"\uD74D": r"\mathbit{\Psi}",
        u"\uD74E": r"\mathbit{\Omega}",
        u"\uD74F": r"\partial{}",
        u"\uD750": r"\in",
        u"\uD751": r"\mathbit{\vartheta}",
        u"\uD752": r"\mathbit{\varkappa}",
        u"\uD753": r"\mathbit{\phi}",
        u"\uD754": r"\mathbit{\varrho}",
        u"\uD755": r"\mathbit{\varpi}",
        u"\uD756": r"\mathsfbf{\Alpha}",
        u"\uD757": r"\mathsfbf{\Beta}",
        u"\uD758": r"\mathsfbf{\Gamma}",
        u"\uD759": r"\mathsfbf{\Delta}",
        u"\uD75A": r"\mathsfbf{\Epsilon}",
        u"\uD75B": r"\mathsfbf{\Zeta}",
        u"\uD75C": r"\mathsfbf{\Eta}",
        u"\uD75D": r"\mathsfbf{\Theta}",
        u"\uD75E": r"\mathsfbf{\Iota}",
        u"\uD75F": r"\mathsfbf{\Kappa}",
        u"\uD760": r"\mathsfbf{\Lambda}",
        u"\uD763": r"\mathsfbf{\Xi}",
        u"\uD765": r"\mathsfbf{\Pi}",
        u"\uD766": r"\mathsfbf{\Rho}",
        u"\uD767": r"\mathsfbf{\vartheta}",
        u"\uD768": r"\mathsfbf{\Sigma}",
        u"\uD769": r"\mathsfbf{\Tau}",
        # note non-raw string (\u confuses Python)
        u"\uD76A": "\\mathsfbf{\\Upsilon}",
        u"\uD76B": r"\mathsfbf{\Phi}",
        u"\uD76C": r"\mathsfbf{\Chi}",
        u"\uD76D": r"\mathsfbf{\Psi}",
        u"\uD76E": r"\mathsfbf{\Omega}",
        u"\uD76F": r"\mathsfbf{\nabla}",
        u"\uD770": r"\mathsfbf{\Alpha}",
        u"\uD771": r"\mathsfbf{\Beta}",
        u"\uD772": r"\mathsfbf{\Gamma}",
        u"\uD773": r"\mathsfbf{\Delta}",
        u"\uD774": r"\mathsfbf{\Epsilon}",
        u"\uD775": r"\mathsfbf{\Zeta}",
        u"\uD776": r"\mathsfbf{\Eta}",
        u"\uD777": r"\mathsfbf{\Theta}",
        u"\uD778": r"\mathsfbf{\Iota}",
        u"\uD779": r"\mathsfbf{\Kappa}",
        u"\uD77A": r"\mathsfbf{\Lambda}",
        u"\uD77D": r"\mathsfbf{\Xi}",
        u"\uD77F": r"\mathsfbf{\Pi}",
        u"\uD780": r"\mathsfbf{\Rho}",
        u"\uD781": r"\mathsfbf{\varsigma}",
        u"\uD782": r"\mathsfbf{\Sigma}",
        u"\uD783": r"\mathsfbf{\Tau}",
        # note non-raw string (\u confuses Python)
        u"\uD784": "\\mathsfbf{\\Upsilon}",
        u"\uD785": r"\mathsfbf{\Phi}",
        u"\uD786": r"\mathsfbf{\Chi}",
        u"\uD787": r"\mathsfbf{\Psi}",
        u"\uD788": r"\mathsfbf{\Omega}",
        u"\uD789": r"\partial{}",
        u"\uD78A": r"\in",
        u"\uD78B": r"\mathsfbf{\vartheta}",
        u"\uD78C": r"\mathsfbf{\varkappa}",
        u"\uD78D": r"\mathsfbf{\phi}",
        u"\uD78E": r"\mathsfbf{\varrho}",
        u"\uD78F": r"\mathsfbf{\varpi}",
        u"\uD790": r"\mathsfbfsl{\Alpha}",
        u"\uD791": r"\mathsfbfsl{\Beta}",
        u"\uD792": r"\mathsfbfsl{\Gamma}",
        u"\uD793": r"\mathsfbfsl{\Delta}",
        u"\uD794": r"\mathsfbfsl{\Epsilon}",
        u"\uD795": r"\mathsfbfsl{\Zeta}",
        u"\uD796": r"\mathsfbfsl{\Eta}",
        u"\uD797": r"\mathsfbfsl{\vartheta}",
        u"\uD798": r"\mathsfbfsl{\Iota}",
        u"\uD799": r"\mathsfbfsl{\Kappa}",
        u"\uD79A": r"\mathsfbfsl{\Lambda}",
        u"\uD79D": r"\mathsfbfsl{\Xi}",
        u"\uD79F": r"\mathsfbfsl{\Pi}",
        u"\uD7A0": r"\mathsfbfsl{\Rho}",
        u"\uD7A1": r"\mathsfbfsl{\vartheta}",
        u"\uD7A2": r"\mathsfbfsl{\Sigma}",
        u"\uD7A3": r"\mathsfbfsl{\Tau}",
        # note non-raw string (\u confuses Python)
        u"\uD7A4": "\\mathsfbfsl{\\Upsilon}",
        u"\uD7A5": r"\mathsfbfsl{\Phi}",
        u"\uD7A6": r"\mathsfbfsl{\Chi}",
        u"\uD7A7": r"\mathsfbfsl{\Psi}",
        u"\uD7A8": r"\mathsfbfsl{\Omega}",
        u"\uD7A9": r"\mathsfbfsl{\nabla}",
        u"\uD7AA": r"\mathsfbfsl{\Alpha}",
        u"\uD7AB": r"\mathsfbfsl{\Beta}",
        u"\uD7AC": r"\mathsfbfsl{\Gamma}",
        u"\uD7AD": r"\mathsfbfsl{\Delta}",
        u"\uD7AE": r"\mathsfbfsl{\Epsilon}",
        u"\uD7AF": r"\mathsfbfsl{\Zeta}",
        u"\uD7B0": r"\mathsfbfsl{\Eta}",
        u"\uD7B1": r"\mathsfbfsl{\vartheta}",
        u"\uD7B2": r"\mathsfbfsl{\Iota}",
        u"\uD7B3": r"\mathsfbfsl{\Kappa}",
        u"\uD7B4": r"\mathsfbfsl{\Lambda}",
        u"\uD7B7": r"\mathsfbfsl{\Xi}",
        u"\uD7B9": r"\mathsfbfsl{\Pi}",
        u"\uD7BA": r"\mathsfbfsl{\Rho}",
        u"\uD7BB": r"\mathsfbfsl{\varsigma}",
        u"\uD7BC": r"\mathsfbfsl{\Sigma}",
        u"\uD7BD": r"\mathsfbfsl{\Tau}",
        # note non-raw string (\u confuses Python)
        u"\uD7BE": "\\mathsfbfsl{\\Upsilon}",
        u"\uD7BF": r"\mathsfbfsl{\Phi}",
        u"\uD7C0": r"\mathsfbfsl{\Chi}",
        u"\uD7C1": r"\mathsfbfsl{\Psi}",
        u"\uD7C2": r"\mathsfbfsl{\Omega}",
        u"\uD7C3": r"\partial{}",
        u"\uD7C4": r"\in",
        u"\uD7C5": r"\mathsfbfsl{\vartheta}",
        u"\uD7C6": r"\mathsfbfsl{\varkappa}",
        u"\uD7C7": r"\mathsfbfsl{\phi}",
        u"\uD7C8": r"\mathsfbfsl{\varrho}",
        u"\uD7C9": r"\mathsfbfsl{\varpi}",
        u"\uD7CE": r"\mathbf{0}",
        u"\uD7CF": r"\mathbf{1}",
        u"\uD7D0": r"\mathbf{2}",
        u"\uD7D1": r"\mathbf{3}",
        u"\uD7D2": r"\mathbf{4}",
        u"\uD7D3": r"\mathbf{5}",
        u"\uD7D4": r"\mathbf{6}",
        u"\uD7D5": r"\mathbf{7}",
        u"\uD7D6": r"\mathbf{8}",
        u"\uD7D7": r"\mathbf{9}",
        u"\uD7D8": r"\mathbb{0}",
        u"\uD7D9": r"\mathbb{1}",
        u"\uD7DA": r"\mathbb{2}",
        u"\uD7DB": r"\mathbb{3}",
        u"\uD7DC": r"\mathbb{4}",
        u"\uD7DD": r"\mathbb{5}",
        u"\uD7DE": r"\mathbb{6}",
        u"\uD7DF": r"\mathbb{7}",
        u"\uD7E0": r"\mathbb{8}",
        u"\uD7E1": r"\mathbb{9}",
        u"\uD7E2": r"\mathsf{0}",
        u"\uD7E3": r"\mathsf{1}",
        u"\uD7E4": r"\mathsf{2}",
        u"\uD7E5": r"\mathsf{3}",
        u"\uD7E6": r"\mathsf{4}",
        u"\uD7E7": r"\mathsf{5}",
        u"\uD7E8": r"\mathsf{6}",
        u"\uD7E9": r"\mathsf{7}",
        u"\uD7EA": r"\mathsf{8}",
        u"\uD7EB": r"\mathsf{9}",
        u"\uD7EC": r"\mathsfbf{0}",
        u"\uD7ED": r"\mathsfbf{1}",
        u"\uD7EE": r"\mathsfbf{2}",
        u"\uD7EF": r"\mathsfbf{3}",
        u"\uD7F0": r"\mathsfbf{4}",
        u"\uD7F1": r"\mathsfbf{5}",
        u"\uD7F2": r"\mathsfbf{6}",
        u"\uD7F3": r"\mathsfbf{7}",
        u"\uD7F4": r"\mathsfbf{8}",
        u"\uD7F5": r"\mathsfbf{9}",
        u"\uD7F6": r"\mathtt{0}",
        u"\uD7F7": r"\mathtt{1}",
        u"\uD7F8": r"\mathtt{2}",
        u"\uD7F9": r"\mathtt{3}",
        u"\uD7FA": r"\mathtt{4}",
        u"\uD7FB": r"\mathtt{5}",
        u"\uD7FC": r"\mathtt{6}",
        u"\uD7FD": r"\mathtt{7}",
        u"\uD7FE": r"\mathtt{8}",
        u"\uD7FF": r"\mathtt{9}",
    }

    # Some multi-character items in the source dataset that we can't
    # translate. This is probably not exhaustive.
    #    u"\u2009\u200A\u200A": r"\;",
    #    u"\u2242\u0338": r"\NotEqualTilde{}",
    #    u"\u224B\u0338": r"\not\apid{}",
    #    u"\u224E\u0338": r"\NotHumpDownHump{}",
    #    u"\u224F\u0338": r"\NotHumpEqual{}",
    #    u"\u2250\u0338": r"\not\doteq",
    #    u"\u2268\uFE00": r"\lvertneqq{}",
    #    u"\u2269\uFE00": r"\gvertneqq{}",
    #    u"\u226A\u0338": r"\NotLessLess{}",
    #    u"\u226B\u0338": r"\NotGreaterGreater{}",
    #    u"\u227E\u0338": r"\NotPrecedesTilde{}",
    #    u"\u227F\u0338": r"\NotSucceedsTilde{}",
    #    u"\u228A\uFE00": r"\varsubsetneqq{}",
    #    u"\u228B\uFE00": r"\varsupsetneq{}",
    #    u"\u228F\u0338": r"\NotSquareSubset{}",
    #    u"\u2290\u0338": r"\NotSquareSuperset{}",
    #    u"\u2933\u0338": r"\ElsevierGlyph{E21D}",
    #    u"\u29CF\u0338": r"\NotLeftTriangleBar{}",
    #    u"\u29D0\u0338": r"\NotRightTriangleBar{}",
    #    u"\u2A7D\u0338": r"\nleqslant{}",
    #    u"\u2A7E\u0338": r"\ngeqslant{}",
    #    u"\u2AA1\u0338": r"\NotNestedLessLess{}",
    #    u"\u2AA2\u0338": r"\NotNestedGreaterGreater{}",
    #    u"\u2AAF\u0338": r"\not\preceq{}",
    #    u"\u2AB0\u0338": r"\not\succeq{}",
    #    u"\u2AC5\u0338": r"\nsubseteqq{}",
    #    u"\u2AC6\u0338": r"\nsupseteqq",
    #    u"\u2AFD\u20E5": r"{\rlap{\textbackslash}{{/}\!\!{/}}}",

    unicode_to_latex_table = dict(
        (ord(k), v)
        for k, v in unicode_to_latex_table_base.items()
    )
    return text.translate(unicode_to_latex_table)
