/* -*- Mode: Vala; indent-tabs-mode: nil; tab-width: 2 -*- */
/*
    This file is part of Déjà Dup.
    For copyright information, see AUTHORS.

    Déjà Dup is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Déjà Dup is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Déjà Dup.  If not, see <http://www.gnu.org/licenses/>.
*/

using GLib;

namespace DejaDup {

// Convenience class for adding automatic backup switch to pref shells
public class PreferencesPeriodicSwitch : Gtk.Switch
{
  DejaDup.FilteredSettings settings;
  construct
  {
    settings = DejaDup.get_settings();
    settings.bind(DejaDup.PERIODIC_KEY, this, "active", SettingsBindFlags.GET);
    state_set.connect(request_background);
  }

  bool request_background(bool state)
  {
    if (state) {
      var bg = new Background();
      if (!bg.request_autostart(this)) {
        this.sensitive = !bg.permission_refused;
        this.active = false;
        return true; // don't change state, skip default handler
      }
    }

    settings.set_boolean(DejaDup.PERIODIC_KEY, state);
    return false;
  }
}

public class Preferences : Gtk.Grid
{
  DejaDupApp _app;
  public DejaDupApp app {
    get { return _app; }
    set {
      _app = value;
      _app.notify["op"].connect(() => {
        restore_button.sensitive = _app.op == null;
        backup_button.sensitive = _app.op == null;
      });
      restore_button.sensitive = _app.op == null;
      backup_button.sensitive = _app.op == null;
    }
  }

  Gtk.Button backup_button;
  Gtk.Button restore_button;
  const int PAGE_HMARGIN = 12;
  const int PAGE_VMARGIN = 18;

  private ConfigPeriod config_period;
  private ConfigDelete config_delete;
  private ConfigLabelBackupDate last_backup_date;
  private ConfigLabelBackupDate next_backup_date;

  ~Preferences() {
    config_period = null;
    config_delete = null;
    last_backup_date = null;
    next_backup_date = null;
  }

  Gtk.Widget make_settings_page()
  {
    var settings_page = new Gtk.Box(Gtk.Orientation.VERTICAL, 0);
    Gtk.Stack stack = new Gtk.Stack();
    Gtk.Widget w;
    Gtk.Grid table;
    int row;
    int i = 0;

    var settings = DejaDup.get_settings();

    row = 0;

    var last_backup_label = new Gtk.Label("");
    last_backup_date = new DejaDup.ConfigLabelBackupDate(DejaDup.ConfigLabelBackupDate.Kind.LAST);
    last_backup_date.bind_property("pretty_date", last_backup_label, "label", BindingFlags.SYNC_CREATE);
    var last_backup_row = new Hdy.ActionRow();
    last_backup_row.title = "Last backup";
    last_backup_row.activatable = false;
    last_backup_row.add_action(last_backup_label);

    restore_button = new Gtk.Button.from_icon_name("document-open-symbolic");
    restore_button.valign = Gtk.Align.CENTER;
    restore_button.clicked.connect((b) => {app.restore();});
    var restore_row = new Hdy.ActionRow();
    restore_row.title = _("_Restore");
    restore_row.subtitle = _("Browse for existing backups");
    restore_row.use_underline = true;
    restore_row.activatable_widget = restore_button;
    restore_row.add_action(restore_button);

    var last_backup_group = new Hdy.PreferencesGroup();
    last_backup_group.title = _("Last backup");
    last_backup_group.add(last_backup_row);
    last_backup_group.add(restore_row);

    var next_backup_label = new Gtk.Label("Today");
    next_backup_date = new DejaDup.ConfigLabelBackupDate(DejaDup.ConfigLabelBackupDate.Kind.NEXT);
    next_backup_date.bind_property("pretty_date", next_backup_label, "label", BindingFlags.SYNC_CREATE);
    var next_backup_row = new Hdy.ActionRow();
    next_backup_row.title = "Next backup";
    next_backup_row.activatable = false;
    next_backup_row.add_action(next_backup_label);

    backup_button = new Gtk.Button.from_icon_name(Config.ICON_NAME + "-symbolic");
    backup_button.valign = Gtk.Align.CENTER;
    backup_button.clicked.connect((b) => {app.backup();});
    var backup_row = new Hdy.ActionRow();
    backup_row.title = _("_Backup now");
    // w = new DejaDup.ConfigLabelDescription(DejaDup.ConfigLabelDescription.Kind.BACKUP);
    backup_row.subtitle = _("Browse for existing backups");
    backup_row.use_underline = true;
    backup_row.activatable_widget = backup_button;
    backup_row.add_action(backup_button);

    var next_backup_group = new Hdy.PreferencesGroup();
    next_backup_group.title = _("Next backup");
    next_backup_group.add(next_backup_row);
    next_backup_group.add(backup_row);

    // Scheduling group
    var automatic_backup_switch = new DejaDup.PreferencesPeriodicSwitch();
    automatic_backup_switch.valign = Gtk.Align.CENTER;
    var automatic_backup_row = new Hdy.ActionRow();
    automatic_backup_row.title = _("_Automatic backup");
    automatic_backup_row.use_underline = true;
    automatic_backup_row.activatable_widget = automatic_backup_switch;
    automatic_backup_row.add_action(automatic_backup_switch);

    var every_row = new Hdy.ComboRow();
    // translators: as in "Every day"
    every_row.title = _("_Every");
    every_row.use_underline = true;
    // We keep it around because we don't want to use it as a widget but as a controller.
    config_period = new DejaDup.ConfigPeriod(every_row, DejaDup.PERIODIC_PERIOD_KEY);
    settings.bind(DejaDup.PERIODIC_KEY, every_row, "sensitive", SettingsBindFlags.GET);

    var keep_row = new Hdy.ComboRow();
    keep_row.title = C_("verb", "_Keep");
    keep_row.use_underline = true;
    // We keep it around because we don't want to use it as a widget but as a controller.
    config_delete = new DejaDup.ConfigDelete(keep_row, DejaDup.DELETE_AFTER_KEY);

    var low_on_space_label = new Gtk.Label(_("Old backups will be deleted earlier if the storage location is low on space."));
    low_on_space_label.margin = 6;
    low_on_space_label.justify = Gtk.Justification.CENTER;
    low_on_space_label.wrap = true;

    var scheduling_group = new Hdy.PreferencesGroup();
    scheduling_group.title = _("Scheduling");
    scheduling_group.add(automatic_backup_row);
    scheduling_group.add(every_row);
    scheduling_group.add(keep_row);
    scheduling_group.add(low_on_space_label);

    var page = new Hdy.PreferencesPage();
    page.add(last_backup_group);
    page.add(next_backup_group);
    page.add(scheduling_group);

    stack.add(page);
    stack.child_set(page,
                    "title", _("Scheduling"),
                    "icon-name", "document-open-recent-symbolic",
                    null);
    ++i;

    // Reset page
    table = new_panel();

    w = new DejaDup.ConfigList(DejaDup.INCLUDE_LIST_KEY);
    w.expand = true;
    table.add(w);

    stack.add(table);
    stack.child_set(table,
                    "title", _("Save"),
                    "icon-name", "document-save-symbolic",
                    null);
    ++i;

    // Reset page
    table = new_panel();

    w = new DejaDup.ConfigList(DejaDup.EXCLUDE_LIST_KEY);
    w.expand = true;
    table.add(w);

    stack.add(table);
    stack.child_set(table,
                    "title", _("Ignore"),
                    "icon-name", "eye-not-looking-symbolic",
                    null);
    ++i;

    // Storage location group
    var location = new DejaDup.ConfigLocation(false, false, new Gtk.SizeGroup(Gtk.SizeGroupMode.HORIZONTAL));
    location.valign = Gtk.Align.CENTER;

    var storage_location_row = new Hdy.ActionRow();
    storage_location_row.title = _("_Storage location");
    storage_location_row.use_underline = true;
    storage_location_row.activatable_widget = location;
    storage_location_row.add_action(location);

    var storage_location_group = new Hdy.PreferencesGroup();
    storage_location_group.add(storage_location_row);

    var storage_location_preferences_group = new Hdy.PreferencesGroup();
    storage_location_preferences_group.add(location.extras);

    page = new Hdy.PreferencesPage();
    page.add(storage_location_group);
    page.add(storage_location_preferences_group);

    stack.add(page);
    // Translators: storage as in "where to store the backup"
    stack.child_set(page,
                    "title", _("Storage"),
                    "icon-name", Config.ICON_NAME + "-symbolic",
                    null);
    ++i;

    stack.show_all(); // can't switch to pages that aren't shown

    stack.expand = true;
    settings_page.add(stack);

    var view_switcher = new Hdy.ViewSwitcherBar();
    view_switcher.reveal = true;
    view_switcher.stack = stack;
    settings_page.add(view_switcher);

    settings_page.show();
    return settings_page;
  }

  Gtk.Grid new_panel()
  {
    var table = new Gtk.Grid();
    table.margin_start = PAGE_HMARGIN;
    table.margin_end = PAGE_HMARGIN;
    table.margin_top = PAGE_VMARGIN;
    table.margin_bottom = PAGE_VMARGIN;
    return table;
  }

  construct {
    add(make_settings_page());
  }
}

}
