/// Contains a set of most categories registered in Desktop Menu Specification.
/// Some categories are mapped to multiple categories according to the specification.
///
/// All categories are in lower-case and without all word separators for easier matching.
///
/// Please sort them in the same order as in the FreeDesktop website, and sort
/// multiple "aliases" for the same category alphabetically.
///
/// Sources:
/// - W3C: https://github.com/w3c/manifest/wiki/Categories#standardized-categories
/// - FreeDesktop: https://specifications.freedesktop.org/menu-spec/menu-spec-1.0.html#category-registry
///
#[cfg(any(platform_linux, platform_bsd))]
pub static XDG_CATEGORIES: phf::Map<&'static str, &'static [&'static str]> = phf::phf_map! {
    // Main Categories
    "audiovideo" => &["AudioVideo"],
    "audio" => &["Audio", "AudioVideo"],
    "music" => &["Audio", "AudioVideo"],
    "video" => &["Video", "AudioVideo"],
    "developertools" => &["Development"],
    "development" => &["Development"],
    "developmenttools" => &["Development"],
    "education" => &["Education"],
    "game" => &["Game"],
    "games" => &["Game"],
    "design" => &["Graphics"],
    "graphics" => &["Graphics"],
    "network" => &["Network"],
    "office" => &["Office"],
    "options" => &["Settings"],
    "settings" => &["Settings"],
    "system" => &["System"],
    "personalization" => &["Utility"],
    "utilities" => &["Utility"],
    "utility" => &["Utility"],

    // Additional Categories
    "building" => &["Building", "Development"],
    "debugger" => &["Debugger", "Development"],
    "ide" => &["IDE", "Development"],
    "guidesigner" => &["GUIDesigner", "Development"],
    "profiler" => &["Profiling", "Development"],
    "profiling" => &["Profiling", "Development"],
    "revisioncontrol" => &["RevisionControl", "Development"],
    "versioncontrol" => &["RevisionControl", "Development"],
    "translation" => &["Translation"],
    "translator" => &["Translation"],
    "calendar" => &["Calendar", "Office"],
    "contactmanagement" => &["ContactManagement", "Office"],
    "database" => &["Database"],
    "dictionaries" => &["Dictionary"],
    "dictionary" => &["Dictionary"],
    "chart" => &["Chart", "Office"],
    "charts" => &["Chart", "Office"],
    "email" => &["Email", "Office", "Network"],
    "finance" => &["Finance", "Office"],
    "flowchart" => &["FlowChart", "Office"],
    "flowcharts" => &["FlowChart", "Office"],
    "pda" => &["PDA", "Office"],
    "projectmanagement" => &["ProjectManagement", "Development", "Office"],
    "presentation" => &["Presentation", "Office"],
    "presentations" => &["Presentation", "Office"],
    "spreadsheet" => &["Spreadsheet", "Office"],
    "spreadsheets" => &["Spreadsheet", "Office"],
    "wordprocessor" => &["WordProcessor", "Office"],
    "2dgraphics" => &["2DGraphics", "Graphics"],
    "vectorgraphics" => &["VectorGraphics", "2DGraphics", "Graphics"],
    "rastergraphics" => &["RasterGraphics", "2DGraphics", "Graphics"],
    "3dgraphics" => &["3DGraphics", "Graphics"],
    "scanning" => &["Scanning"],
    "ocr" => &["OCR", "Scanning"],
    "photo" => &["Photography"],
    "photography" => &["Photography"],
    "publishing" => &["Publishing"],
    "viewer" => &["Viewer"],
    "texttool" => &["TextTools", "Utility"],
    "texttools" => &["TextTools", "Utility"],
    "desktopsettings" => &["DesktopSettings", "Settings"],
    "hardwaresettings" => &["HardwareSettings", "Settings"],
    "printing" => &["Printing"],
    "packagemanager" => &["PackageManager"],
    "dialup" => &["Dialup", "Network"],
    "instantmessaging" => &["InstantMessaging", "Network"],
    "chat" => &["Chat", "Network"],
    "social" => &["InstantMessaging", "Chat", "Network"],
    "ircclient" => &["IRCClient", "Network"],
    "filetransfer" => &["FileTransfer", "Network"],
    "hamradio" => &["HamRadio", "Network"],
    "magazines" => &["News"],
    "news" => &["News"],
    "newspapers" => &["News"],
    "p2p" => &["P2P", "Network"],
    "remoteaccess" => &["RemoteAccess", "Network"],
    "telephony" => &["Telephony", "Network"],
    "telephonytools" => &["TelephonyTools", "Utility"],
    "videoconference" => &["VideoConference", "Network"],
    "browser" => &["WebBrowser", "Network"],
    "webbrowser" => &["WebBrowser", "Network"],
    "webdevelopment" => &["WebDevelopment", "Development", "Network"],
    "midi" => &["Midi", "Audio", "AudioVideo"],
    "mixer" => &["Mixer", "Audio", "AudioVideo"],
    "sequencer" => &["Sequencer", "Audio", "AudioVideo"],
    "tuner" => &["Tuner", "Audio", "AudioVideo"],
    "tv" => &["TV", "Video", "AudioVideo"],
    "audioediting" => &["AudioVideoEditing", "Audio", "AudioVideo"],
    "videoediting" => &["AudioVideoEditing", "Video", "AudioVideo"],
    "audiovideoediting" => &["AudioVideoEditing", "AudioVideo"],
    "player" => &["Player", "AudioVideo"],
    "recorder" => &["Recorder", "AudioVideo"],
    "discburning" => &["DiscBurning", "AudioVideo"],
    "actiongame" => &["ActionGame", "Game"],
    "adventuregame" => &["AdventureGame", "Game"],
    "arcadegame" => &["ArcadeGame", "Game"],
    "boardgame" => &["BoardGame", "Game"],
    "blocksgame" => &["BlocksGame", "Game"],
    "cardgame" => &["CardGame", "Game"],
    "kidsgame" => &["KidsGame", "Game"],
    "logicgame" => &["LogicGame", "Game"],
    "roleplaying" => &["RolePlaying", "Game"],
    "roleplayinggame" => &["RolePlaying", "Game"],
    "rpg" => &["RolePlaying", "Game"],
    "simulation" => &["Simulation", "Game"],
    "simulationgame" => &["Simulation", "Game"],
    "sportsgame" => &["SportsGame", "Game"],
    "strategygame" => &["StrategyGame", "Game"],
    "art" => &["Art"],
    "construction" => &["Construction"],
    "languages" => &["Languages"],
    "science" => &["Science"],
    "artificialintelligence" => &["ArtificialIntelligence", "Science"],
    "astronomy" => &["Astronomy", "Science"],
    "biology" => &["Biology", "Science"],
    "chemistry" => &["Chemistry", "Science"],
    "computerscience" => &["ComputerScience", "Science"],
    "datavisualization" => &["DataVisualization", "ComputerScience", "Science"],
    "economy" => &["Economy"],
    "electricity" => &["Electricity"],
    "geography" => &["Geography", "Science"],
    "geology" => &["Geology", "Science"],
    "geoscience" => &["Geoscience", "Science"],
    "history" => &["History", "Science"],
    "imageprocessing" => &["ImageProcessing"],
    "book" => &["Literature"],
    "books" => &["Literature"],
    "literature" => &["Literature"],
    "math" => &["Math", "Science"],
    "numericalanalysis" => &["NumericalAnalysis", "Math", "Science"],
    "medical" => &["MedicalSoftware", "Science"],
    "medicalsoftware" => &["MedicalSoftware", "Science"],
    "physics" => &["Physics", "Science"],
    "robotics" => &["Robotics", "Science"],
    "fitness" => &["Sports"],
    "sport" => &["Sports"],
    "sports" => &["Sports"],
    "parallelcomputing" => &["ParallelComputing", "ComputerScience", "Science"],
    "amusement" => &["Amusement"],
    "entertainment" => &["Amusement"],
    "archiving" => &["Archiving", "Utility"],
    "compression" => &["Compression", "Archiving", "Utility"],
    "electronics" => &["Electronics"],
    "emulation" => &["Emulator"],
    "emulator" => &["Emulator"],
    "engineering" => &["Engineering"],
    "filetools" => &["FileTools", "System", "Utility"],
    "filemanager" => &["FileManager", "FileTools", "System", "Utility"],
    "terminalemulator" => &["TerminalEmulator", "System"],
    "filesystem" => &["Filesystem", "System"],
    "monitor" => &["Monitor", "System"],
    "security" => &["Security"],
    "accessibility" => &["Accessibility"],
    "calculator" => &["Calculator", "Utility"],
    "clock" => &["Clock", "Utility"],
    "texteditor" => &["TextEditor", "Utility"],
    "documentation" => &["Documentation"],
};

/// Contains mappings from W3C categories, including some additional
/// non-standard categories, to the Apple application categories list.
///
/// All categories are in lower-case and without all word separators for easier matching.
///
/// Please sort them in the same order as in the Apple website, and sort
/// multiple "aliases" for the same category alphabetically.
///
/// Sources:
/// - W3C: https://github.com/w3c/manifest/wiki/Categories#standardized-categories
/// - Apple: https://developer.apple.com/documentation/bundleresources/information_property_list/lsapplicationcategorytype
///
#[cfg(platform_macos)]
pub static MACOS_CATEGORIES: phf::Map<&'static str, &'static str> = phf::phf_map! {
    "business" => "public.app-category.business",
    "developertools" => "public.app-category.developer-tools",
    "development" => "public.app-category.developer-tools",
    "developmenttools" => "public.app-category.developer-tools",
    "education" => "public.app-category.education",
    "amusement" => "public.app-category.entertainment",
    "entertainment" => "public.app-category.entertainment",
    "finance" => "public.app-category.finance",
    "game" => "public.app-category.games",
    "games" => "public.app-category.games",
    "actiongame" => "public.app-category.action-games",
    "adventuregame" => "public.app-category.adventure-games",
    "arcadegame" => "public.app-category.arcade-games",
    "boardgame" => "public.app-category.board-games",
    "cardgame" => "public.app-category.card-games",
    "casinogame" => "public.app-category.casino-games",
    "dicegame" => "public.app-category.dice-games",
    "educationalgame" => "public.app-category.educational-games",
    "familygame" => "public.app-category.family-games",
    "kidsgame" => "public.app-category.kids-games",
    "musicgame" => "public.app-category.music-games",
    "puzzlegame" => "public.app-category.puzzle-games",
    "racinggames" => "public.app-category.racing-games",
    "roleplaying" => "public.app-category.role-playing-games",
    "roleplayinggame" => "public.app-category.role-playing-games",
    "rpg" => "public.app-category.role-playing-games",
    "simulation" => "public.app-category.simulation-games",
    "simulationgame" => "public.app-category.simulation-games",
    "sportsgame" => "public.app-category.sports-games",
    "strategygame" => "public.app-category.strategy-games",
    "triviagame" => "public.app-category.trivia-games",
    "wordgame" => "public.app-category.word-games",
    "design" => "public.app-category.graphics-design",
    "graphics" => "public.app-category.graphics-design",
    "health" => "public.app-category.healthcare-fitness",
    "fitness" => "public.app-category.healthcare-fitness",
    "lifestyle" => "public.app-category.lifestyle",
    "medical" => "public.app-category.medical",
    "medicalsoftware" => "public.app-category.medical",
    "music" => "public.app-category.music",
    "news" => "public.app-category.news",
    "photo" => "public.app-category.photography",
    "photography" => "public.app-category.photography",
    "productivity" => "public.app-category.productivity",
    "reference" => "public.app-category.reference",
    "social" => "public.app-category.social-networking",
    "sport" => "public.app-category.sports",
    "sports" => "public.app-category.sports",
    "travel" => "public.app-category.travel",
    "personalization" => "public.app-category.utilities",
    "utilities" => "public.app-category.utilities",
    "utility" => "public.app-category.utilities",
    "video" => "public.app-category.video",
    "weather" => "public.app-category.weather",
};

/// Contains mappings from W3C categories, including some additional
/// non-standard categories, to the PortableApps.com categories list.
///
/// All categories are in lower-case and without all word separators for easier matching.
///
/// Please sort them in the same order as in the PortableApps.com website,
/// and sort multiple "aliases" for the same category alphabetically.
///
/// Sources:
/// - W3C: https://github.com/w3c/manifest/wiki/Categories#standardized-categories
/// - PortableApps.com: https://portableapps.com/development/portableapps.com_format#appinfo
///
#[cfg(all(platform_windows, feature = "portable"))]
pub static PORTABLEAPPS_CATEGORIES: phf::Map<&'static str, &'static str> = phf::phf_map! {
    "accessibility" => "Accessibility",
    "developertools" => "Development",
    "development" => "Development",
    "developmenttools" => "Development",
    "education" => "Education",
    "game" => "Games",
    "games" => "Games",
    "design" => "Graphics & Pictures",
    "graphics" => "Graphics & Pictures",
    "internet" => "Internet",
    "audio" => "Music & Video",
    "music" => "Music & Video",
    "video" => "Music & Video",
    "office" => "Office",
    "security" => "Security",
    "personalization" => "Utilities",
    "utilities" => "Utilities",
    "utility" => "Utilities",
};
