# -*- coding: utf-8 -*-
#
# Copyright (C) 2018 Matthias Klumpp <matthias.klumpp@puri.sm>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, see <http://www.gnu.org/licenses/>.

import os
import subprocess
import shlex
import json
import logging as log
from glob import glob


ENCRYPT_BYPASS_INITRAMFS_HOOK = '/usr/share/initramfs-tools/hooks/bypass_encrypt_hook'
CRYPTTAB_FILE = '/etc/crypttab'

KEYFILE = '/crypto_keyfile.bin'
KEYFILE_OLD = '/crypto_keyfile.bin.old'

CALAMARES_LUKS_PARTITION_FILE = '/encrypted_partitions.json'

CRYPTTAB_HEADER = """# /etc/crypttab: mappings for encrypted partitions.
#
# Each mapped device will be created in /dev/mapper, so your /etc/fstab
# should use the /dev/mapper/<name> paths for encrypted devices.
#
# See crypttab(5) for the supported syntax.
#
# NOTE: Do not list your root (/) partition here, it must be set up
#       beforehand by the initramfs (/etc/mkinitcpio.conf). The same applies
#       to encrypted swap, which should be set up with mkinitcpio-openswap
#       for resume support.
#
# <name>               <device>                         <password> <options>"""


def run_command(command, input=None):
    if not isinstance(command, list):
        command = shlex.split(command)

    if not input:
        input = None
    elif isinstance(input, str):
        input = input.encode('utf-8')
    elif not isinstance(input, bytes):
        input = input.read()

    try:
        pipe = subprocess.Popen(command,
                                shell=False,
                                stdin=subprocess.PIPE,
                                stdout=subprocess.PIPE,
                                stderr=subprocess.PIPE,
                                )
    except OSError:
        return (None, None, -1)

    (output, stderr) = pipe.communicate(input=input)
    (output, stderr) = (c.decode('utf-8', errors='ignore') for c in (output, stderr))
    return (output, stderr, pipe.returncode)


def get_device_id_alias(device):
    all_disk_paths = glob('/dev/disk/by-id/*')

    device_id_alias = device
    for d in all_disk_paths:
        # resolve alias links to direct /dev nodes
        dev_path = os.path.realpath(d)
        if dev_path == device:
            device_id_alias = d
            break

    return device_id_alias


def generate_crypttab_line_info(partition, crypttab_options, use_password=False):
    """ Generates information for each crypttab entry. """
    if 'luksMapperName' not in partition or 'luksUuid' not in partition:
        return None

    mapper_name = partition['luksMapperName']
    luks_uuid = partition['luksUuid']
    if not mapper_name or not luks_uuid:
        return None

    if partition['fs'] == 'linuxswap':
        # we always re-encrypt swap partitions with a random password on boot
        # this makes suspend-to-disk impossible, but improves security slightly
        return dict(
            name=mapper_name,
            device=get_device_id_alias(partition['device']),
            password='/dev/urandom',
            options='swap,cipher=aes-xts-plain64,size=256,noearly')

    if use_password:
        pwd_s = 'none'
    else:
        pwd_s = '/crypto_keyfile.bin'

    return dict(
        name=mapper_name,
        device='UUID=' + luks_uuid,
        password=pwd_s,
        options=crypttab_options)


def print_crypttab_line(dct, file=None):
    """ Prints line to '/etc/crypttab' file. """
    line = "{:21} {:<45} {} {}".format(dct['name'],
                                       dct['device'],
                                       dct['password'],
                                       dct['options'])

    print(line, file=file)


def set_initial_disk_password(new_password):
    if not os.path.isfile(ENCRYPT_BYPASS_INITRAMFS_HOOK) or not os.path.isfile(CALAMARES_LUKS_PARTITION_FILE):
        log.warning('Encrypt bypass initramfs hook does not exist, script will not do anything')
        return False

    luks_partitions = json.load(open(CALAMARES_LUKS_PARTITION_FILE, 'r'))
    luks_root_device = luks_partitions.get('rootDevice')
    additional_luks_devices = luks_partitions.get('additionalDevices', [])

    if not new_password:
        raise Exception("New disk password is empty")
    if not luks_root_device:
        raise Exception("No LUKS root device set in encrypted_partitions.json")

    # move old keyfile out of the way
    os.rename(KEYFILE, KEYFILE_OLD)

    # Generate random keyfile
    out, err, ret = run_command(['dd',
                                 'bs=512',
                                 'count=4',
                                 'if=/dev/urandom',
                                 'of=/crypto_keyfile.bin'])
    if ret != 0:
        raise Exception("Unable to create new crypto_keyfile.bin: {} - {}".format(out, err))

    # make a list of all partitions we deal with
    partitions = additional_luks_devices.copy()
    partitions.insert(0, luks_root_device)

    for partition in partitions:
        if partition['fs'] == 'linuxswap':
            continue  # at time, we encrypt swap with a random password at boot time

        out, err, ret = run_command(['cryptsetup',
                                     'luksAddKey',
                                     partition['device'],
                                     KEYFILE,
                                     '--key-file', KEYFILE_OLD])
        if ret != 0:
            raise Exception("Unable to add key file: {} - {}".format(out, err))

        out, err, ret = run_command(['cryptsetup',
                                     'luksRemoveKey',
                                     partition['device'],
                                     KEYFILE_OLD])
        if ret != 0:
            raise Exception("Unable to remove old key file: {} - {}".format(out, err))

        # only add a real password to the root partition.
        # once that is unlocked, the system has access to the keyfile and can automatically
        # decrypt all other partitions
        if partition['device'] == luks_root_device['device']:
            out, err, ret = run_command(['cryptsetup',
                                         'luksAddKey',
                                         partition['device'],
                                         '--key-file', KEYFILE,
                                         '-q'],
                                        input=new_password)
            if ret != 0:
                raise Exception("Unable to add key: {} - {}".format(out, err))

    out, err, ret = run_command(['chmod',
                                 'g-rwx,o-rwx',
                                 KEYFILE])
    if ret != 0:
        raise Exception("Unable to set permissions on key file: {} - {}".format(out, err))

    os.remove(KEYFILE_OLD)
    os.remove(ENCRYPT_BYPASS_INITRAMFS_HOOK)
    os.remove(CALAMARES_LUKS_PARTITION_FILE)

    # rewrite crypttab with the new options
    with open(CRYPTTAB_FILE, 'w') as crypttab_f:
        print(CRYPTTAB_HEADER, file=crypttab_f)

        # the root partition is decrypted using a password, not a keyfile
        dct = generate_crypttab_line_info(luks_root_device, 'luks', True)
        if dct:
            print_crypttab_line(dct, file=crypttab_f)

        # all other partitions are decrypted using a keyfile
        for partition in additional_luks_devices:
            dct = generate_crypttab_line_info(partition, 'luks,keyscript=/bin/cat')

            if dct:
                print_crypttab_line(dct, file=crypttab_f)

        crypttab_f.flush()
        os.fsync(crypttab_f.fileno())

    return True


def update_initramfs():
    # FIXME: workaround for cryptsetup picking up swap by UUID (because it may be mounted via that)
    # when updating initramfs. We must not do that, since swap is always reformatted on boot,
    # which chabges its UUID.
    # For now, we just unmount swap at this stage to be safe. Subsequent initramfs updates after a reboot
    # are safe
    out, err, ret = run_command(['swapoff', '-a'])
    if ret != 0:
        log.error("Unable to disable swap: {} - {}".format(out, err))

    # update initramfs to make changes permanent
    out, err, ret = run_command(['update-initramfs', '-u'])
    if ret != 0:
        raise Exception("Unable to update initramfs: {} - {}".format(out, err))

    # enable swap after the initramfs update
    out, err, ret = run_command(['swapon', '-a'])
    if ret != 0:
        log.error("Unable to enable swap: {} - {}".format(out, err))

    return True
