# -*- coding: utf-8 -*-
#
# Copyright (C) 2018 Matthias Klumpp <matthias.klumpp@puri.sm>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, see <http://www.gnu.org/licenses/>.

import dbus.service
import logging as log
import sys
from threading import Timer, Thread
from .crypto_setup import set_initial_disk_password, update_initramfs


DBUS_BUS_NAME = 'net.pureos.InitDiskCrypto'
DBUS_PATH = '/'
DBUS_INTERFACE_NAME = 'net.pureos.InitDiskCrypto'


class PermissionDeniedByPolicy(dbus.DBusException):
    _dbus_error_name = 'net.pureos.initdiskcrypto.applypassword'


class InitDiskCryptoDBus(dbus.service.Object):

    def __init__(self, bus, mainloop, enforce_polkit=True):
        # used in _check_policykit_priviledge
        self._dbus_info = None
        self._polkit = None
        # useful for testing
        self._enforce_polkit = enforce_polkit
        self._mainloop = mainloop

        # init dbus service
        bus_name = dbus.service.BusName(DBUS_INTERFACE_NAME, bus=bus)
        dbus.service.Object.__init__(self, bus_name, DBUS_PATH)

        self._timer = self._get_timer()
        self._timer.start()

        self._dpu_thread = None
        self._password_set = False

        log.debug('waiting for connections')

    def _get_timer(self):
        return Timer(40, self._timeout)

    def _timeout(self):
        log.debug('Service timed out due to inactivity.')
        if self._dpu_thread and self._dpu_thread.is_alive():
            self._dpu_thread.join()
        self._mainloop.quit()
        sys.exit(0)

    def _disk_password_update_thread(self, password):
        try:
            ret = set_initial_disk_password(password)
            update_initramfs()
        except Exception as e:
            log.error(str(e))
            ret = False

        # reset timeout timer
        self._timer = self._get_timer()
        self._timer.start()

        self.Finished(ret)

    @dbus.service.signal(dbus_interface=DBUS_INTERFACE_NAME, signature='')
    def AuthFailed(self):
        ''' Signal emitted when authorization has failed '''
        log.debug('Auth failed signal')

    @dbus.service.signal(dbus_interface=DBUS_INTERFACE_NAME, signature='b')
    def Finished(self, success):
        ''' Signal emitted when action was completed successfully '''
        log.debug('Finished signal. Success: {}'.format(success))

    @dbus.service.method(DBUS_INTERFACE_NAME,
                         sender_keyword='sender', connection_keyword='conn',
                         in_signature='s', out_signature='b')
    def SetInitialDiskPassword(self, password, sender=None, conn=None):
        if self._password_set:
            return False

        # don't time out while doing something
        self._timer.cancel()

        self._check_policykit_privilege(
            sender, conn, 'net.pureos.initdiskcrypto.applypassword')

        self._password_set = True  # disallow running this function again
        self._dpu_thread = Thread(target=self._disk_password_update_thread, args=(password, ))
        self._dpu_thread.start()

        return True

    def _check_policykit_privilege(self, sender, conn, privilege):
        '''Verify that sender has a given PolicyKit privilege.

        sender is the sender's (private) D-BUS name, such as ':1:42'
        (sender_keyword in @dbus.service.methods). conn is
        the dbus.Connection object (connection_keyword in
        @dbus.service.methods). privilege is the PolicyKit privilege string.

        This method returns if the caller is privileged, and otherwise throws a
        PermissionDeniedByPolicy exception.
        '''
        if sender is None and conn is None:
            # called locally, not through D-BUS
            return
        if not self._enforce_polkit:
            # that happens for testing purposes when running on the session
            # bus, and it does not make sense to restrict operations here
            return

        # get peer PID
        if self._dbus_info is None:
            self._dbus_info = dbus.Interface(conn.get_object('org.freedesktop.DBus',
                                                             '/org/freedesktop/DBus/Bus', False), 'org.freedesktop.DBus')
        pid = self._dbus_info.GetConnectionUnixProcessID(sender)

        # query PolicyKit
        if self._polkit is None:
            self._polkit = dbus.Interface(dbus.SystemBus().get_object(
                'org.freedesktop.PolicyKit1',
                '/org/freedesktop/PolicyKit1/Authority', False),
                'org.freedesktop.PolicyKit1.Authority')
        try:
            # we don't need is_challenge return here, since we call with AllowUserInteraction
            (is_auth, _, details) = self._polkit.CheckAuthorization(
                ('system-bus-name', {'name': dbus.String(sender, variant_level=1)}),
                privilege, {'': ''}, dbus.UInt32(1), '', timeout=600)
        except dbus.DBusException as e:
            if e._dbus_error_name == 'org.freedesktop.DBus.Error.ServiceUnknown':
                # polkitd timed out, connect again
                self._polkit = None
                return self._check_polkit_privilege(sender, conn, privilege)
            else:
                raise

        if not is_auth:
            log.debug('_check_polkit_privilege: sender %s on connection %s pid %i is not authorized for %s: %s' %
                      (sender, conn, pid, privilege, str(details)))
            self.AuthFailed()
            raise PermissionDeniedByPolicy(privilege)
