#!/usr/bin/python

"""Fake energy meter that calculates energy use by integrating power use."""
# (C) Copyright IBM Corp. 2008-2009
# Licensed under the GPLv2.
import pwrkap_data
import thread
import datetime
import time
import traceback

class power_energy_meter(pwrkap_data.energy_meter):
	"""Fake energy meter that uses power meter to estimate energy use."""
	def __init__(self, pmeter, threaded):
		self.power_meter = pmeter
		self.energy_use = None
		self.last_update = None
		self.keep_running = True
		self.threaded = threaded

		# This must run last
		if self.threaded:
			thread.start_new_thread(self.run, ())

	def __del__(self):
		self.keep_running = False

	def __setstate__(self, data):
		self.__dict__ = data
		self.last_update = None
		if self.threaded:
			thread.start_new_thread(self.run, ())

	def run(self):
		"""Periodically update energy use."""
		self.keep_running = True
		while self.keep_running:
			try:
				self.update_energy_use()
			except Exception, e:
				print e
				traceback.print_exc()
			time.sleep(1)

	def update_energy_use(self):
		"""Update the energy use counter."""
		now = datetime.datetime.utcnow()

		if self.last_update == None:
			self.energy_use = 0
			self.last_update = now
			return

		td = now - self.last_update
		time_delta =  td.microseconds + \
			     (td.seconds * 1000000) + \
			     (td.days * 86400000000)
		if time_delta == 0:
			time_delta = 1
		time_delta = float(time_delta) / 1000000

		rate = self.power_meter.read()
		self.energy_use = self.energy_use + (rate * time_delta)
		self.last_update = now

	def read(self):
		if not self.threaded:
			self.update_energy_use()
			time.sleep(2)
			self.update_energy_use()
		return self.energy_use

	def get_latency(self):
		return 0;

	def inventory(self):
		return ("power_energy_meter", {})

class energy_power_meter(pwrkap_data.power_meter):
	"""Fake power meter that uses energy meter to estimate energy use."""
	def __init__(self, emeter, threaded):
		self.energy_meter = emeter
		self.last_energy_use = None
		self.last_update = None
		self.power_use = None
		self.keep_running = True
		self.threaded = threaded

		# This must run last
		if self.threaded:
			thread.start_new_thread(self.run, ())

	def __del__(self):
		self.keep_running = False

	def __setstate__(self, data):
		self.__dict__ = data
		self.last_update = None
		if self.threaded:
			thread.start_new_thread(self.run, ())

	def run(self):
		"""Periodically update energy use."""
		self.keep_running = True
		while self.keep_running:
			try:
				self.update_energy_use()
			except Exception, e:
				print e
				traceback.print_exc()
			time.sleep(1)

	def update_power_use(self):
		"""Update the power use counter."""
		now = datetime.datetime.utcnow()

		if self.last_update == None:
			self.last_energy_use = self.energy_meter.read()
			self.power_use = 0
			self.last_update = now
			return

		td = now - self.last_update
		time_delta =  td.microseconds + \
			     (td.seconds * 1000000) + \
			     (td.days * 86400000000)
		if time_delta == 0:
			time_delta = 1
		time_delta = float(time_delta) / 1000000

		energy_now = self.energy_meter.read()
		self.power_use = float(energy_now - self.last_energy_use) / time_delta
		self.last_energy_use = energy_now
		self.last_update = now

	def read(self):
		if not self.threaded:
			self.update_power_use()
			time.sleep(2)
			self.update_power_use()
		return self.power_use

	def get_latency(self):
		return 0;

	def inventory(self):
		return ("energy_power_meter", {})

