# Licensed under a 3-clause BSD style license - see LICENSE.rst

# "test_ufunc.py" is auto-generated by erfa_generator.py from the template
# "test_ufunc.py.templ". Do *not* edit "test_ufunc.py" directly, instead edit
# "test_ufunc.py.templ" and run erfa_generator.py from the source directory to
# update it.

"""
Basic tests of the ERFA library routines.

These are just the tests bundled with ERFA itself, in ``t_erfa_c.c``,
but translated to python to make sure that the code returns the same numbers.

The viv and vid functions emulate the corresponding functions in ``t_erfa_c.c``.

The tests are skipped if a system library is used that does not match in
version number, since in that case the precise numbers may have changed
(e.g., due to the earth orientation changes between 1.7.2 and 1.7.3).
"""

import numpy as np
import pytest

import erfa
from erfa import ufunc as erfa_ufunc


if not erfa.__version__.startswith(erfa.version.erfa_version):
    pytest.skip("skipping for system library with mismatched version",
                allow_module_level=True)


def viv(ival, ivalok, func, test, _):
    """Validate an integer result."""
    assert ival == ivalok, f"{func} failed: {test} want {ivalok} got {ival}"


def vvd(val, valok, dval, func, test, _):
    """Validate a double result."""
    a = val - valok
    assert a == 0.0 or abs(a) <= abs(dval), (
        f"{func} failed: {test} want {valok:.20g} got {val:.20g} "
        f"(1/{abs(valok / a):.3g})")


status = np.zeros((), dtype=int)
# <--------------------------Actual test-wrapping code------------------------>


def test_a2af():
    s, idmsf = erfa_ufunc.a2af(4, 2.345)
    viv(s[0], b'+', "a2af", "s", status)
    viv(idmsf[0],  134, "a2af", "0", status)
    viv(idmsf[1],   21, "a2af", "1", status)
    viv(idmsf[2],   30, "a2af", "2", status)
    viv(idmsf[3], 9706, "a2af", "3", status)


def test_a2tf():
    s, ihmsf = erfa_ufunc.a2tf(4, -3.01234)
    viv(s[0], b'-', "a2tf", "s", status)
    viv(ihmsf[0],   11, "a2tf", "0", status)
    viv(ihmsf[1],   30, "a2tf", "1", status)
    viv(ihmsf[2],   22, "a2tf", "2", status)
    viv(ihmsf[3], 6484, "a2tf", "3", status)


def test_ab():
    pnat = np.empty((3,), float)
    v = np.empty((3,), float)
    ppr = np.empty((3,), float)
    pnat[0] = -0.76321968546737951
    pnat[1] = -0.60869453983060384
    pnat[2] = -0.21676408580639883
    v[0] =  2.1044018893653786e-5
    v[1] = -8.9108923304429319e-5
    v[2] = -3.8633714797716569e-5
    s = 0.99980921395708788
    bm1 = 0.99999999506209258
    ppr = erfa_ufunc.ab(pnat, v, s, bm1)
    vvd(ppr[0], -0.7631631094219556269, 1e-12, "ab", "1", status)
    vvd(ppr[1], -0.6087553082505590832, 1e-12, "ab", "2", status)
    vvd(ppr[2], -0.2167926269368471279, 1e-12, "ab", "3", status)


def test_ae2hd():
    a = 5.5
    e = 1.1
    p = 0.7
    h, d = erfa_ufunc.ae2hd(a, e, p)
    vvd(h, 0.5933291115507309663, 1e-14, "ae2hd", "h", status)
    vvd(d, 0.9613934761647817620, 1e-14, "ae2hd", "d", status)


def test_af2a():
    a, j = erfa_ufunc.af2a('-', 45, 13, 27.2)
    vvd(a, -0.7893115794313644842, 1e-12, "af2a", "a", status)
    viv(j, 0, "af2a", "j", status)


def test_anp():
    vvd(erfa_ufunc.anp(-0.1), 6.183185307179586477, 1e-12, "anp", "", status)


def test_anpm():
    vvd(erfa_ufunc.anpm(-4.0), 2.283185307179586477, 1e-12, "anpm", "", status)


def test_apcg():
    ebpv = np.empty((), erfa_ufunc.dt_pv)
    ehp = np.empty((3,), float)
    date1 = 2456165.5
    date2 = 0.401182685
    ebpv['p'][0] = 0.901310875
    ebpv['p'][1] = -0.417402664
    ebpv['p'][2] = -0.180982288
    ebpv['v'][0] = 0.00742727954
    ebpv['v'][1] = 0.0140507459
    ebpv['v'][2] = 0.00609045792
    ehp[0] = 0.903358544
    ehp[1] = -0.415395237
    ehp[2] = -0.180084014
    astrom = erfa_ufunc.apcg(date1, date2, ebpv, ehp)
    astrom = astrom.view(np.recarray)
    vvd(astrom.pmt, 12.65133794027378508, 1e-11, "apcg", "pmt", status)
    vvd(astrom.eb[0], 0.901310875, 1e-12, "apcg", "eb(1)", status)
    vvd(astrom.eb[1], -0.417402664, 1e-12, "apcg", "eb(2)", status)
    vvd(astrom.eb[2], -0.180982288, 1e-12, "apcg", "eb(3)", status)
    vvd(astrom.eh[0], 0.8940025429324143045, 1e-12, "apcg", "eh(1)", status)
    vvd(astrom.eh[1], -0.4110930268679817955, 1e-12, "apcg", "eh(2)", status)
    vvd(astrom.eh[2], -0.1782189004872870264, 1e-12, "apcg", "eh(3)", status)
    vvd(astrom.em, 1.010465295811013146, 1e-12, "apcg", "em", status)
    vvd(astrom.v[0], 0.4289638913597693554e-4, 1e-16, "apcg", "v(1)", status)
    vvd(astrom.v[1], 0.8115034051581320575e-4, 1e-16, "apcg", "v(2)", status)
    vvd(astrom.v[2], 0.3517555136380563427e-4, 1e-16, "apcg", "v(3)", status)
    vvd(astrom.bm1, 0.9999999951686012981, 1e-12, "apcg", "bm1", status)
    vvd(astrom.bpn[0][0], 1.0, 0.0, "apcg", "bpn(1,1)", status)
    vvd(astrom.bpn[1][0], 0.0, 0.0, "apcg", "bpn(2,1)", status)
    vvd(astrom.bpn[2][0], 0.0, 0.0, "apcg", "bpn(3,1)", status)
    vvd(astrom.bpn[0][1], 0.0, 0.0, "apcg", "bpn(1,2)", status)
    vvd(astrom.bpn[1][1], 1.0, 0.0, "apcg", "bpn(2,2)", status)
    vvd(astrom.bpn[2][1], 0.0, 0.0, "apcg", "bpn(3,2)", status)
    vvd(astrom.bpn[0][2], 0.0, 0.0, "apcg", "bpn(1,3)", status)
    vvd(astrom.bpn[1][2], 0.0, 0.0, "apcg", "bpn(2,3)", status)
    vvd(astrom.bpn[2][2], 1.0, 0.0, "apcg", "bpn(3,3)", status)


def test_apcg13():
    date1 = 2456165.5
    date2 = 0.401182685
    astrom = erfa_ufunc.apcg13(date1, date2)
    astrom = astrom.view(np.recarray)
    vvd(astrom.pmt, 12.65133794027378508, 1e-11, "apcg13", "pmt", status)
    vvd(astrom.eb[0], 0.9013108747340644755, 1e-12, "apcg13", "eb(1)", status)
    vvd(astrom.eb[1], -0.4174026640406119957, 1e-12, "apcg13", "eb(2)", status)
    vvd(astrom.eb[2], -0.1809822877867817771, 1e-12, "apcg13", "eb(3)", status)
    vvd(astrom.eh[0], 0.8940025429255499549, 1e-12, "apcg13", "eh(1)", status)
    vvd(astrom.eh[1], -0.4110930268331896318, 1e-12, "apcg13", "eh(2)", status)
    vvd(astrom.eh[2], -0.1782189006019749850, 1e-12, "apcg13", "eh(3)", status)
    vvd(astrom.em, 1.010465295964664178, 1e-12, "apcg13", "em", status)
    vvd(astrom.v[0], 0.4289638912941341125e-4, 1e-16, "apcg13", "v(1)", status)
    vvd(astrom.v[1], 0.8115034032405042132e-4, 1e-16, "apcg13", "v(2)", status)
    vvd(astrom.v[2], 0.3517555135536470279e-4, 1e-16, "apcg13", "v(3)", status)
    vvd(astrom.bm1, 0.9999999951686013142, 1e-12, "apcg13", "bm1", status)
    vvd(astrom.bpn[0][0], 1.0, 0.0, "apcg13", "bpn(1,1)", status)
    vvd(astrom.bpn[1][0], 0.0, 0.0, "apcg13", "bpn(2,1)", status)
    vvd(astrom.bpn[2][0], 0.0, 0.0, "apcg13", "bpn(3,1)", status)
    vvd(astrom.bpn[0][1], 0.0, 0.0, "apcg13", "bpn(1,2)", status)
    vvd(astrom.bpn[1][1], 1.0, 0.0, "apcg13", "bpn(2,2)", status)
    vvd(astrom.bpn[2][1], 0.0, 0.0, "apcg13", "bpn(3,2)", status)
    vvd(astrom.bpn[0][2], 0.0, 0.0, "apcg13", "bpn(1,3)", status)
    vvd(astrom.bpn[1][2], 0.0, 0.0, "apcg13", "bpn(2,3)", status)
    vvd(astrom.bpn[2][2], 1.0, 0.0, "apcg13", "bpn(3,3)", status)


def test_apci():
    ebpv = np.empty((), erfa_ufunc.dt_pv)
    ehp = np.empty((3,), float)
    date1 = 2456165.5
    date2 = 0.401182685
    ebpv['p'][0] = 0.901310875
    ebpv['p'][1] = -0.417402664
    ebpv['p'][2] = -0.180982288
    ebpv['v'][0] = 0.00742727954
    ebpv['v'][1] = 0.0140507459
    ebpv['v'][2] = 0.00609045792
    ehp[0] = 0.903358544
    ehp[1] = -0.415395237
    ehp[2] = -0.180084014
    x = 0.0013122272
    y = -2.92808623e-5
    s = 3.05749468e-8
    astrom = erfa_ufunc.apci(date1, date2, ebpv, ehp, x, y, s)
    astrom = astrom.view(np.recarray)
    vvd(astrom.pmt, 12.65133794027378508, 1e-11, "apci", "pmt", status)
    vvd(astrom.eb[0], 0.901310875, 1e-12, "apci", "eb(1)", status)
    vvd(astrom.eb[1], -0.417402664, 1e-12, "apci", "eb(2)", status)
    vvd(astrom.eb[2], -0.180982288, 1e-12, "apci", "eb(3)", status)
    vvd(astrom.eh[0], 0.8940025429324143045, 1e-12, "apci", "eh(1)", status)
    vvd(astrom.eh[1], -0.4110930268679817955, 1e-12, "apci", "eh(2)", status)
    vvd(astrom.eh[2], -0.1782189004872870264, 1e-12, "apci", "eh(3)", status)
    vvd(astrom.em, 1.010465295811013146, 1e-12, "apci", "em", status)
    vvd(astrom.v[0], 0.4289638913597693554e-4, 1e-16, "apci", "v(1)", status)
    vvd(astrom.v[1], 0.8115034051581320575e-4, 1e-16, "apci", "v(2)", status)
    vvd(astrom.v[2], 0.3517555136380563427e-4, 1e-16, "apci", "v(3)", status)
    vvd(astrom.bm1, 0.9999999951686012981, 1e-12, "apci", "bm1", status)
    vvd(astrom.bpn[0][0], 0.9999991390295159156, 1e-12, "apci", "bpn(1,1)", status)
    vvd(astrom.bpn[1][0], 0.4978650072505016932e-7, 1e-12, "apci", "bpn(2,1)", status)
    vvd(astrom.bpn[2][0], 0.1312227200000000000e-2, 1e-12, "apci", "bpn(3,1)", status)
    vvd(astrom.bpn[0][1], -0.1136336653771609630e-7, 1e-12, "apci", "bpn(1,2)", status)
    vvd(astrom.bpn[1][1], 0.9999999995713154868, 1e-12, "apci", "bpn(2,2)", status)
    vvd(astrom.bpn[2][1], -0.2928086230000000000e-4, 1e-12, "apci", "bpn(3,2)", status)
    vvd(astrom.bpn[0][2], -0.1312227200895260194e-2, 1e-12, "apci", "bpn(1,3)", status)
    vvd(astrom.bpn[1][2], 0.2928082217872315680e-4, 1e-12, "apci", "bpn(2,3)", status)
    vvd(astrom.bpn[2][2], 0.9999991386008323373, 1e-12, "apci", "bpn(3,3)", status)


def test_apci13():
    date1 = 2456165.5
    date2 = 0.401182685
    astrom, eo = erfa_ufunc.apci13(date1, date2)
    astrom = astrom.view(np.recarray)
    vvd(astrom.pmt, 12.65133794027378508, 1e-11, "apci13", "pmt", status)
    vvd(astrom.eb[0], 0.9013108747340644755, 1e-12, "apci13", "eb(1)", status)
    vvd(astrom.eb[1], -0.4174026640406119957, 1e-12, "apci13", "eb(2)", status)
    vvd(astrom.eb[2], -0.1809822877867817771, 1e-12, "apci13", "eb(3)", status)
    vvd(astrom.eh[0], 0.8940025429255499549, 1e-12, "apci13", "eh(1)", status)
    vvd(astrom.eh[1], -0.4110930268331896318, 1e-12, "apci13", "eh(2)", status)
    vvd(astrom.eh[2], -0.1782189006019749850, 1e-12, "apci13", "eh(3)", status)
    vvd(astrom.em, 1.010465295964664178, 1e-12, "apci13", "em", status)
    vvd(astrom.v[0], 0.4289638912941341125e-4, 1e-16, "apci13", "v(1)", status)
    vvd(astrom.v[1], 0.8115034032405042132e-4, 1e-16, "apci13", "v(2)", status)
    vvd(astrom.v[2], 0.3517555135536470279e-4, 1e-16, "apci13", "v(3)", status)
    vvd(astrom.bm1, 0.9999999951686013142, 1e-12, "apci13", "bm1", status)
    vvd(astrom.bpn[0][0], 0.9999992060376761710, 1e-12, "apci13", "bpn(1,1)", status)
    vvd(astrom.bpn[1][0], 0.4124244860106037157e-7, 1e-12, "apci13", "bpn(2,1)", status)
    vvd(astrom.bpn[2][0], 0.1260128571051709670e-2, 1e-12, "apci13", "bpn(3,1)", status)
    vvd(astrom.bpn[0][1], -0.1282291987222130690e-7, 1e-12, "apci13", "bpn(1,2)", status)
    vvd(astrom.bpn[1][1], 0.9999999997456835325, 1e-12, "apci13", "bpn(2,2)", status)
    vvd(astrom.bpn[2][1], -0.2255288829420524935e-4, 1e-12, "apci13", "bpn(3,2)", status)
    vvd(astrom.bpn[0][2], -0.1260128571661374559e-2, 1e-12, "apci13", "bpn(1,3)", status)
    vvd(astrom.bpn[1][2], 0.2255285422953395494e-4, 1e-12, "apci13", "bpn(2,3)", status)
    vvd(astrom.bpn[2][2], 0.9999992057833604343, 1e-12, "apci13", "bpn(3,3)", status)
    vvd(eo, -0.2900618712657375647e-2, 1e-12, "apci13", "eo", status)


def test_apco():
    ebpv = np.empty((), erfa_ufunc.dt_pv)
    ehp = np.empty((3,), float)
    date1 = 2456384.5
    date2 = 0.970031644
    ebpv['p'][0] = -0.974170438
    ebpv['p'][1] = -0.211520082
    ebpv['p'][2] = -0.0917583024
    ebpv['v'][0] = 0.00364365824
    ebpv['v'][1] = -0.0154287319
    ebpv['v'][2] = -0.00668922024
    ehp[0] = -0.973458265
    ehp[1] = -0.209215307
    ehp[2] = -0.0906996477
    x = 0.0013122272
    y = -2.92808623e-5
    s = 3.05749468e-8
    theta = 3.14540971
    elong = -0.527800806
    phi = -1.2345856
    hm = 2738.0
    xp = 2.47230737e-7
    yp = 1.82640464e-6
    sp = -3.01974337e-11
    refa = 0.000201418779
    refb = -2.36140831e-7
    astrom = erfa_ufunc.apco(date1, date2, ebpv, ehp, x, y, s, theta, elong, phi, hm, xp, yp, sp, refa, refb)
    astrom = astrom.view(np.recarray)
    vvd(astrom.pmt, 13.25248468622587269, 1e-11, "apco", "pmt", status)
    vvd(astrom.eb[0], -0.9741827110630322720, 1e-12, "apco", "eb(1)", status)
    vvd(astrom.eb[1], -0.2115130190135344832, 1e-12, "apco", "eb(2)", status)
    vvd(astrom.eb[2], -0.09179840186949532298, 1e-12, "apco", "eb(3)", status)
    vvd(astrom.eh[0], -0.9736425571689739035, 1e-12, "apco", "eh(1)", status)
    vvd(astrom.eh[1], -0.2092452125849330936, 1e-12, "apco", "eh(2)", status)
    vvd(astrom.eh[2], -0.09075578152243272599, 1e-12, "apco", "eh(3)", status)
    vvd(astrom.em, 0.9998233241709957653, 1e-12, "apco", "em", status)
    vvd(astrom.v[0], 0.2078704992916728762e-4, 1e-16, "apco", "v(1)", status)
    vvd(astrom.v[1], -0.8955360107151952319e-4, 1e-16, "apco", "v(2)", status)
    vvd(astrom.v[2], -0.3863338994288951082e-4, 1e-16, "apco", "v(3)", status)
    vvd(astrom.bm1, 0.9999999950277561236, 1e-12, "apco", "bm1", status)
    vvd(astrom.bpn[0][0], 0.9999991390295159156, 1e-12, "apco", "bpn(1,1)", status)
    vvd(astrom.bpn[1][0], 0.4978650072505016932e-7, 1e-12, "apco", "bpn(2,1)", status)
    vvd(astrom.bpn[2][0], 0.1312227200000000000e-2, 1e-12, "apco", "bpn(3,1)", status)
    vvd(astrom.bpn[0][1], -0.1136336653771609630e-7, 1e-12, "apco", "bpn(1,2)", status)
    vvd(astrom.bpn[1][1], 0.9999999995713154868, 1e-12, "apco", "bpn(2,2)", status)
    vvd(astrom.bpn[2][1], -0.2928086230000000000e-4, 1e-12, "apco", "bpn(3,2)", status)
    vvd(astrom.bpn[0][2], -0.1312227200895260194e-2, 1e-12, "apco", "bpn(1,3)", status)
    vvd(astrom.bpn[1][2], 0.2928082217872315680e-4, 1e-12, "apco", "bpn(2,3)", status)
    vvd(astrom.bpn[2][2], 0.9999991386008323373, 1e-12, "apco", "bpn(3,3)", status)
    vvd(astrom.along, -0.5278008060295995734, 1e-12, "apco", "along", status)
    vvd(astrom.xpl, 0.1133427418130752958e-5, 1e-17, "apco", "xpl", status)
    vvd(astrom.ypl, 0.1453347595780646207e-5, 1e-17, "apco", "ypl", status)
    vvd(astrom.sphi, -0.9440115679003211329, 1e-12, "apco", "sphi", status)
    vvd(astrom.cphi, 0.3299123514971474711, 1e-12, "apco", "cphi", status)
    vvd(astrom.diurab, 0, 0, "apco", "diurab", status)
    vvd(astrom.eral, 2.617608903970400427, 1e-12, "apco", "eral", status)
    vvd(astrom.refa, 0.2014187790000000000e-3, 1e-15, "apco", "refa", status)
    vvd(astrom.refb, -0.2361408310000000000e-6, 1e-18, "apco", "refb", status)


def test_apco13():
    utc1 = 2456384.5
    utc2 = 0.969254051
    dut1 = 0.1550675
    elong = -0.527800806
    phi = -1.2345856
    hm = 2738.0
    xp = 2.47230737e-7
    yp = 1.82640464e-6
    phpa = 731.0
    tc = 12.8
    rh = 0.59
    wl = 0.55
    astrom, eo, j = erfa_ufunc.apco13(utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    astrom = astrom.view(np.recarray)
    vvd(astrom.pmt, 13.25248468622475727, 1e-11, "apco13", "pmt", status)
    vvd(astrom.eb[0], -0.9741827107320875162, 1e-12, "apco13", "eb(1)", status)
    vvd(astrom.eb[1], -0.2115130190489716682, 1e-12, "apco13", "eb(2)", status)
    vvd(astrom.eb[2], -0.09179840189496755339, 1e-12, "apco13", "eb(3)", status)
    vvd(astrom.eh[0], -0.9736425572586935247, 1e-12, "apco13", "eh(1)", status)
    vvd(astrom.eh[1], -0.2092452121603336166, 1e-12, "apco13", "eh(2)", status)
    vvd(astrom.eh[2], -0.09075578153885665295, 1e-12, "apco13", "eh(3)", status)
    vvd(astrom.em, 0.9998233240913898141, 1e-12, "apco13", "em", status)
    vvd(astrom.v[0], 0.2078704994520489246e-4, 1e-16, "apco13", "v(1)", status)
    vvd(astrom.v[1], -0.8955360133238868938e-4, 1e-16, "apco13", "v(2)", status)
    vvd(astrom.v[2], -0.3863338993055887398e-4, 1e-16, "apco13", "v(3)", status)
    vvd(astrom.bm1, 0.9999999950277561004, 1e-12, "apco13", "bm1", status)
    vvd(astrom.bpn[0][0], 0.9999991390295147999, 1e-12, "apco13", "bpn(1,1)", status)
    vvd(astrom.bpn[1][0], 0.4978650075315529277e-7, 1e-12, "apco13", "bpn(2,1)", status)
    vvd(astrom.bpn[2][0], 0.001312227200850293372, 1e-12, "apco13", "bpn(3,1)", status)
    vvd(astrom.bpn[0][1], -0.1136336652812486604e-7, 1e-12, "apco13", "bpn(1,2)", status)
    vvd(astrom.bpn[1][1], 0.9999999995713154865, 1e-12, "apco13", "bpn(2,2)", status)
    vvd(astrom.bpn[2][1], -0.2928086230975367296e-4, 1e-12, "apco13", "bpn(3,2)", status)
    vvd(astrom.bpn[0][2], -0.001312227201745553566, 1e-12, "apco13", "bpn(1,3)", status)
    vvd(astrom.bpn[1][2], 0.2928082218847679162e-4, 1e-12, "apco13", "bpn(2,3)", status)
    vvd(astrom.bpn[2][2], 0.9999991386008312212, 1e-12, "apco13", "bpn(3,3)", status)
    vvd(astrom.along, -0.5278008060295995733, 1e-12, "apco13", "along", status)
    vvd(astrom.xpl, 0.1133427418130752958e-5, 1e-17, "apco13", "xpl", status)
    vvd(astrom.ypl, 0.1453347595780646207e-5, 1e-17, "apco13", "ypl", status)
    vvd(astrom.sphi, -0.9440115679003211329, 1e-12, "apco13", "sphi", status)
    vvd(astrom.cphi, 0.3299123514971474711, 1e-12, "apco13", "cphi", status)
    vvd(astrom.diurab, 0, 0, "apco13", "diurab", status)
    vvd(astrom.eral, 2.617608909189664000, 1e-12, "apco13", "eral", status)
    vvd(astrom.refa, 0.2014187785940396921e-3, 1e-15, "apco13", "refa", status)
    vvd(astrom.refb, -0.2361408314943696227e-6, 1e-18, "apco13", "refb", status)
    vvd(eo, -0.003020548354802412839, 1e-14, "apco13", "eo", status)
    viv(j, 0, "apco13", "j", status)


def test_apcs():
    pv = np.empty((), erfa_ufunc.dt_pv)
    ebpv = np.empty((), erfa_ufunc.dt_pv)
    ehp = np.empty((3,), float)
    date1 = 2456384.5
    date2 = 0.970031644
    pv['p'][0] = -1836024.09
    pv['p'][1] = 1056607.72
    pv['p'][2] = -5998795.26
    pv['v'][0] = -77.0361767
    pv['v'][1] = -133.310856
    pv['v'][2] = 0.0971855934
    ebpv['p'][0] = -0.974170438
    ebpv['p'][1] = -0.211520082
    ebpv['p'][2] = -0.0917583024
    ebpv['v'][0] = 0.00364365824
    ebpv['v'][1] = -0.0154287319
    ebpv['v'][2] = -0.00668922024
    ehp[0] = -0.973458265
    ehp[1] = -0.209215307
    ehp[2] = -0.0906996477
    astrom = erfa_ufunc.apcs(date1, date2, pv, ebpv, ehp)
    astrom = astrom.view(np.recarray)
    vvd(astrom.pmt, 13.25248468622587269, 1e-11, "apcs", "pmt", status)
    vvd(astrom.eb[0], -0.9741827110629881886, 1e-12, "apcs", "eb(1)", status)
    vvd(astrom.eb[1], -0.2115130190136415986, 1e-12, "apcs", "eb(2)", status)
    vvd(astrom.eb[2], -0.09179840186954412099, 1e-12, "apcs", "eb(3)", status)
    vvd(astrom.eh[0], -0.9736425571689454706, 1e-12, "apcs", "eh(1)", status)
    vvd(astrom.eh[1], -0.2092452125850435930, 1e-12, "apcs", "eh(2)", status)
    vvd(astrom.eh[2], -0.09075578152248299218, 1e-12, "apcs", "eh(3)", status)
    vvd(astrom.em, 0.9998233241709796859, 1e-12, "apcs", "em", status)
    vvd(astrom.v[0], 0.2078704993282685510e-4, 1e-16, "apcs", "v(1)", status)
    vvd(astrom.v[1], -0.8955360106989405683e-4, 1e-16, "apcs", "v(2)", status)
    vvd(astrom.v[2], -0.3863338994289409097e-4, 1e-16, "apcs", "v(3)", status)
    vvd(astrom.bm1, 0.9999999950277561237, 1e-12, "apcs", "bm1", status)
    vvd(astrom.bpn[0][0], 1, 0, "apcs", "bpn(1,1)", status)
    vvd(astrom.bpn[1][0], 0, 0, "apcs", "bpn(2,1)", status)
    vvd(astrom.bpn[2][0], 0, 0, "apcs", "bpn(3,1)", status)
    vvd(astrom.bpn[0][1], 0, 0, "apcs", "bpn(1,2)", status)
    vvd(astrom.bpn[1][1], 1, 0, "apcs", "bpn(2,2)", status)
    vvd(astrom.bpn[2][1], 0, 0, "apcs", "bpn(3,2)", status)
    vvd(astrom.bpn[0][2], 0, 0, "apcs", "bpn(1,3)", status)
    vvd(astrom.bpn[1][2], 0, 0, "apcs", "bpn(2,3)", status)
    vvd(astrom.bpn[2][2], 1, 0, "apcs", "bpn(3,3)", status)


def test_apcs13():
    pv = np.empty((), erfa_ufunc.dt_pv)
    date1 = 2456165.5
    date2 = 0.401182685
    pv['p'][0] = -6241497.16
    pv['p'][1] = 401346.896
    pv['p'][2] = -1251136.04
    pv['v'][0] = -29.264597
    pv['v'][1] = -455.021831
    pv['v'][2] = 0.0266151194
    astrom = erfa_ufunc.apcs13(date1, date2, pv)
    astrom = astrom.view(np.recarray)
    vvd(astrom.pmt, 12.65133794027378508, 1e-11, "apcs13", "pmt", status)
    vvd(astrom.eb[0], 0.9012691529025250644, 1e-12, "apcs13", "eb(1)", status)
    vvd(astrom.eb[1], -0.4173999812023194317, 1e-12, "apcs13", "eb(2)", status)
    vvd(astrom.eb[2], -0.1809906511146429670, 1e-12, "apcs13", "eb(3)", status)
    vvd(astrom.eh[0], 0.8939939101760130792, 1e-12, "apcs13", "eh(1)", status)
    vvd(astrom.eh[1], -0.4111053891734021478, 1e-12, "apcs13", "eh(2)", status)
    vvd(astrom.eh[2], -0.1782336880636997374, 1e-12, "apcs13", "eh(3)", status)
    vvd(astrom.em, 1.010428384373491095, 1e-12, "apcs13", "em", status)
    vvd(astrom.v[0], 0.4279877294121697570e-4, 1e-16, "apcs13", "v(1)", status)
    vvd(astrom.v[1], 0.7963255087052120678e-4, 1e-16, "apcs13", "v(2)", status)
    vvd(astrom.v[2], 0.3517564013384691531e-4, 1e-16, "apcs13", "v(3)", status)
    vvd(astrom.bm1, 0.9999999952947980978, 1e-12, "apcs13", "bm1", status)
    vvd(astrom.bpn[0][0], 1, 0, "apcs13", "bpn(1,1)", status)
    vvd(astrom.bpn[1][0], 0, 0, "apcs13", "bpn(2,1)", status)
    vvd(astrom.bpn[2][0], 0, 0, "apcs13", "bpn(3,1)", status)
    vvd(astrom.bpn[0][1], 0, 0, "apcs13", "bpn(1,2)", status)
    vvd(astrom.bpn[1][1], 1, 0, "apcs13", "bpn(2,2)", status)
    vvd(astrom.bpn[2][1], 0, 0, "apcs13", "bpn(3,2)", status)
    vvd(astrom.bpn[0][2], 0, 0, "apcs13", "bpn(1,3)", status)
    vvd(astrom.bpn[1][2], 0, 0, "apcs13", "bpn(2,3)", status)
    vvd(astrom.bpn[2][2], 1, 0, "apcs13", "bpn(3,3)", status)


def test_aper():
    astrom = np.zeros((), erfa_ufunc.dt_eraASTROM).view(np.recarray)
    astrom.along = 1.234
    theta = 5.678
    astrom = erfa_ufunc.aper(theta, astrom)
    astrom = astrom.view(np.recarray)
    vvd(astrom.eral, 6.912000000000000000, 1e-12, "aper", "pmt", status)


def test_aper13():
    astrom = np.zeros((), erfa_ufunc.dt_eraASTROM).view(np.recarray)
    astrom.along = 1.234
    ut11 = 2456165.5
    ut12 = 0.401182685
    astrom = erfa_ufunc.aper13(ut11, ut12, astrom)
    astrom = astrom.view(np.recarray)
    vvd(astrom.eral, 3.316236661789694933, 1e-12, "aper13", "pmt", status)


def test_apio():
    sp = -3.01974337e-11
    theta = 3.14540971
    elong = -0.527800806
    phi = -1.2345856
    hm = 2738.0
    xp = 2.47230737e-7
    yp = 1.82640464e-6
    refa = 0.000201418779
    refb = -2.36140831e-7
    astrom = erfa_ufunc.apio(sp, theta, elong, phi, hm, xp, yp, refa, refb)
    astrom = astrom.view(np.recarray)
    vvd(astrom.along, -0.5278008060295995734, 1e-12, "apio", "along", status)
    vvd(astrom.xpl, 0.1133427418130752958e-5, 1e-17, "apio", "xpl", status)
    vvd(astrom.ypl, 0.1453347595780646207e-5, 1e-17, "apio", "ypl", status)
    vvd(astrom.sphi, -0.9440115679003211329, 1e-12, "apio", "sphi", status)
    vvd(astrom.cphi, 0.3299123514971474711, 1e-12, "apio", "cphi", status)
    vvd(astrom.diurab, 0.5135843661699913529e-6, 1e-12, "apio", "diurab", status)
    vvd(astrom.eral, 2.617608903970400427, 1e-12, "apio", "eral", status)
    vvd(astrom.refa, 0.2014187790000000000e-3, 1e-15, "apio", "refa", status)
    vvd(astrom.refb, -0.2361408310000000000e-6, 1e-18, "apio", "refb", status)


def test_apio13():
    utc1 = 2456384.5
    utc2 = 0.969254051
    dut1 = 0.1550675
    elong = -0.527800806
    phi = -1.2345856
    hm = 2738.0
    xp = 2.47230737e-7
    yp = 1.82640464e-6
    phpa = 731.0
    tc = 12.8
    rh = 0.59
    wl = 0.55
    astrom, j = erfa_ufunc.apio13(utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    astrom = astrom.view(np.recarray)
    vvd(astrom.along, -0.5278008060295995733, 1e-12, "apio13", "along", status)
    vvd(astrom.xpl, 0.1133427418130752958e-5, 1e-17, "apio13", "xpl", status)
    vvd(astrom.ypl, 0.1453347595780646207e-5, 1e-17, "apio13", "ypl", status)
    vvd(astrom.sphi, -0.9440115679003211329, 1e-12, "apio13", "sphi", status)
    vvd(astrom.cphi, 0.3299123514971474711, 1e-12, "apio13", "cphi", status)
    vvd(astrom.diurab, 0.5135843661699913529e-6, 1e-12, "apio13", "diurab", status)
    vvd(astrom.eral, 2.617608909189664000, 1e-12, "apio13", "eral", status)
    vvd(astrom.refa, 0.2014187785940396921e-3, 1e-15, "apio13", "refa", status)
    vvd(astrom.refb, -0.2361408314943696227e-6, 1e-18, "apio13", "refb", status)
    viv(j, 0, "apio13", "j", status)


def test_atcc13():
    rc = 2.71
    dc = 0.174
    pr = 1e-5
    pd = 5e-6
    px = 0.1
    rv = 55.0
    date1 = 2456165.5
    date2 = 0.401182685
    ra, da = erfa_ufunc.atcc13(rc, dc, pr, pd, px, rv, date1, date2)
    vvd(ra,  2.710126504531372384, 1e-12, "atcc13", "ra", status)
    vvd(da, 0.1740632537628350152, 1e-12, "atcc13", "da", status)


def test_atccq():
    date1 = 2456165.5
    date2 = 0.401182685
    astrom, eo = erfa_ufunc.apci13(date1, date2)
    rc = 2.71
    dc = 0.174
    pr = 1e-5
    pd = 5e-6
    px = 0.1
    rv = 55.0
    ra, da = erfa_ufunc.atccq(rc, dc, pr, pd, px, rv, astrom)
    vvd(ra, 2.710126504531372384, 1e-12, "atccq", "ra", status)
    vvd(da, 0.1740632537628350152, 1e-12, "atccq", "da", status)


def test_atci13():
    rc = 2.71
    dc = 0.174
    pr = 1e-5
    pd = 5e-6
    px = 0.1
    rv = 55.0
    date1 = 2456165.5
    date2 = 0.401182685
    ri, di, eo = erfa_ufunc.atci13(rc, dc, pr, pd, px, rv, date1, date2)
    vvd(ri, 2.710121572968696744, 1e-12, "atci13", "ri", status)
    vvd(di, 0.1729371367219539137, 1e-12, "atci13", "di", status)
    vvd(eo, -0.002900618712657375647, 1e-14, "atci13", "eo", status)


def test_atciq():
    date1 = 2456165.5
    date2 = 0.401182685
    astrom, eo = erfa_ufunc.apci13(date1, date2)
    rc = 2.71
    dc = 0.174
    pr = 1e-5
    pd = 5e-6
    px = 0.1
    rv = 55.0
    ri, di = erfa_ufunc.atciq(rc, dc, pr, pd, px, rv, astrom)
    vvd(ri, 2.710121572968696744, 1e-12, "atciq", "ri", status)
    vvd(di, 0.1729371367219539137, 1e-12, "atciq", "di", status)


def test_atciqn():
    b = np.empty(3, erfa_ufunc.dt_eraLDBODY).view(np.recarray)
    date1 = 2456165.5
    date2 = 0.401182685
    astrom, eo = erfa_ufunc.apci13(date1, date2)
    rc = 2.71
    dc = 0.174
    pr = 1e-5
    pd = 5e-6
    px = 0.1
    rv = 55.0
    b[0].bm = 0.00028574
    b[0].dl = 3e-10
    b[0].pv['p'][0] = -7.81014427
    b[0].pv['p'][1] = -5.60956681
    b[0].pv['p'][2] = -1.98079819
    b[0].pv['v'][0] = 0.0030723249
    b[0].pv['v'][1] = -0.00406995477
    b[0].pv['v'][2] = -0.00181335842
    b[1].bm = 0.00095435
    b[1].dl = 3e-9
    b[1].pv['p'][0] = 0.738098796
    b[1].pv['p'][1] = 4.63658692
    b[1].pv['p'][2] = 1.9693136
    b[1].pv['v'][0] = -0.00755816922
    b[1].pv['v'][1] = 0.00126913722
    b[1].pv['v'][2] = 0.000727999001
    b[2].bm = 1.0
    b[2].dl = 6e-6
    b[2].pv['p'][0] = -0.000712174377
    b[2].pv['p'][1] = -0.00230478303
    b[2].pv['p'][2] = -0.00105865966
    b[2].pv['v'][0] = 6.29235213e-6
    b[2].pv['v'][1] = -3.30888387e-7
    b[2].pv['v'][2] = -2.96486623e-7
    ri, di = erfa_ufunc.atciqn (rc, dc, pr, pd, px, rv, astrom, b)
    vvd(ri, 2.710122008104983335, 1e-12, "atciqn", "ri", status)
    vvd(di, 0.1729371916492767821, 1e-12, "atciqn", "di", status)


def test_atciqz():
    date1 = 2456165.5
    date2 = 0.401182685
    astrom, eo = erfa_ufunc.apci13(date1, date2)
    rc = 2.71
    dc = 0.174
    ri, di = erfa_ufunc.atciqz(rc, dc, astrom)
    vvd(ri, 2.709994899247256984, 1e-12, "atciqz", "ri", status)
    vvd(di, 0.1728740720984931891, 1e-12, "atciqz", "di", status)


def test_atco13():
    rc = 2.71
    dc = 0.174
    pr = 1e-5
    pd = 5e-6
    px = 0.1
    rv = 55.0
    utc1 = 2456384.5
    utc2 = 0.969254051
    dut1 = 0.1550675
    elong = -0.527800806
    phi = -1.2345856
    hm = 2738.0
    xp = 2.47230737e-7
    yp = 1.82640464e-6
    phpa = 731.0
    tc = 12.8
    rh = 0.59
    wl = 0.55
    aob, zob, hob, dob, rob, eo, j = erfa_ufunc.atco13(rc, dc, pr, pd, px, rv, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    vvd(aob, 0.9251774485485515207e-1, 1e-12, "atco13", "aob", status)
    vvd(zob, 1.407661405256499357, 1e-12, "atco13", "zob", status)
    vvd(hob, -0.9265154431529724692e-1, 1e-12, "atco13", "hob", status)
    vvd(dob, 0.1716626560072526200, 1e-12, "atco13", "dob", status)
    vvd(rob, 2.710260453504961012, 1e-12, "atco13", "rob", status)
    vvd(eo, -0.003020548354802412839, 1e-14, "atco13", "eo", status)
    viv(j, 0, "atco13", "j", status)


def test_atic13():
    ri = 2.710121572969038991
    di = 0.1729371367218230438
    date1 = 2456165.5
    date2 = 0.401182685
    rc, dc, eo = erfa_ufunc.atic13(ri, di, date1, date2)
    vvd(rc, 2.710126504531716819, 1e-12, "atic13", "rc", status)
    vvd(dc, 0.1740632537627034482, 1e-12, "atic13", "dc", status)
    vvd(eo, -0.002900618712657375647, 1e-14, "atic13", "eo", status)


def test_aticq():
    date1 = 2456165.5
    date2 = 0.401182685
    astrom, eo = erfa_ufunc.apci13(date1, date2)
    ri = 2.710121572969038991
    di = 0.1729371367218230438
    rc, dc = erfa_ufunc.aticq(ri, di, astrom)
    vvd(rc, 2.710126504531716819, 1e-12, "aticq", "rc", status)
    vvd(dc, 0.1740632537627034482, 1e-12, "aticq", "dc", status)


def test_aticqn():
    b = np.empty(3, erfa_ufunc.dt_eraLDBODY).view(np.recarray)
    date1 = 2456165.5
    date2 = 0.401182685
    astrom, eo = erfa_ufunc.apci13(date1, date2)
    ri = 2.709994899247599271
    di = 0.1728740720983623469
    b[0].bm = 0.00028574
    b[0].dl = 3e-10
    b[0].pv['p'][0] = -7.81014427
    b[0].pv['p'][1] = -5.60956681
    b[0].pv['p'][2] = -1.98079819
    b[0].pv['v'][0] = 0.0030723249
    b[0].pv['v'][1] = -0.00406995477
    b[0].pv['v'][2] = -0.00181335842
    b[1].bm = 0.00095435
    b[1].dl = 3e-9
    b[1].pv['p'][0] = 0.738098796
    b[1].pv['p'][1] = 4.63658692
    b[1].pv['p'][2] = 1.9693136
    b[1].pv['v'][0] = -0.00755816922
    b[1].pv['v'][1] = 0.00126913722
    b[1].pv['v'][2] = 0.000727999001
    b[2].bm = 1.0
    b[2].dl = 6e-6
    b[2].pv['p'][0] = -0.000712174377
    b[2].pv['p'][1] = -0.00230478303
    b[2].pv['p'][2] = -0.00105865966
    b[2].pv['v'][0] = 6.29235213e-6
    b[2].pv['v'][1] = -3.30888387e-7
    b[2].pv['v'][2] = -2.96486623e-7
    rc, dc = erfa_ufunc.aticqn(ri, di, astrom, b)
    vvd(rc, 2.709999575033027333, 1e-12, "eraAtciqn", "rc", status)
    vvd(dc, 0.1739999656316469990, 1e-12, "eraAtciqn", "dc", status)


def test_atio13():
    ri = 2.710121572969038991
    di = 0.1729371367218230438
    utc1 = 2456384.5
    utc2 = 0.969254051
    dut1 = 0.1550675
    elong = -0.527800806
    phi = -1.2345856
    hm = 2738.0
    xp = 2.47230737e-7
    yp = 1.82640464e-6
    phpa = 731.0
    tc = 12.8
    rh = 0.59
    wl = 0.55
    aob, zob, hob, dob, rob, j = erfa_ufunc.atio13(ri, di, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    vvd(aob, 0.9233952224895122499e-1, 1e-12, "atio13", "aob", status)
    vvd(zob, 1.407758704513549991, 1e-12, "atio13", "zob", status)
    vvd(hob, -0.9247619879881698140e-1, 1e-12, "atio13", "hob", status)
    vvd(dob, 0.1717653435756234676, 1e-12, "atio13", "dob", status)
    vvd(rob, 2.710085107988480746, 1e-12, "atio13", "rob", status)
    viv(j, 0, "atio13", "j", status)


def test_atioq():
    utc1 = 2456384.5
    utc2 = 0.969254051
    dut1 = 0.1550675
    elong = -0.527800806
    phi = -1.2345856
    hm = 2738.0
    xp = 2.47230737e-7
    yp = 1.82640464e-6
    phpa = 731.0
    tc = 12.8
    rh = 0.59
    wl = 0.55
    astrom, j = erfa_ufunc.apio13(utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    ri = 2.710121572969038991
    di = 0.1729371367218230438
    aob, zob, hob, dob, rob = erfa_ufunc.atioq(ri, di, astrom)
    vvd(aob, 0.9233952224895122499e-1, 1e-12, "atioq", "aob", status)
    vvd(zob, 1.407758704513549991, 1e-12, "atioq", "zob", status)
    vvd(hob, -0.9247619879881698140e-1, 1e-12, "atioq", "hob", status)
    vvd(dob, 0.1717653435756234676, 1e-12, "atioq", "dob", status)
    vvd(rob, 2.710085107988480746, 1e-12, "atioq", "rob", status)


def test_atoc13():
    utc1 = 2456384.5
    utc2 = 0.969254051
    dut1 = 0.1550675
    elong = -0.527800806
    phi = -1.2345856
    hm = 2738.0
    xp = 2.47230737e-7
    yp = 1.82640464e-6
    phpa = 731.0
    tc = 12.8
    rh = 0.59
    wl = 0.55
    ob1 = 2.710085107986886201
    ob2 = 0.1717653435758265198
    rc, dc, j = erfa_ufunc.atoc13 ("R", ob1, ob2, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    vvd(rc, 2.709956744659136129, 1e-12, "atoc13", "R/rc", status)
    vvd(dc, 0.1741696500898471362, 1e-12, "atoc13", "R/dc", status)
    viv(j, 0, "atoc13", "R/j", status)
    ob1 = -0.09247619879782006106
    ob2 = 0.1717653435758265198
    rc, dc, j = erfa_ufunc.atoc13 ("H", ob1, ob2, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    vvd(rc, 2.709956744659734086, 1e-12, "atoc13", "H/rc", status)
    vvd(dc, 0.1741696500898471362, 1e-12, "atoc13", "H/dc", status)
    viv(j, 0, "atoc13", "H/j", status)
    ob1 = 0.09233952224794989993
    ob2 = 1.407758704513722461
    rc, dc, j = erfa_ufunc.atoc13 ("A", ob1, ob2, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    vvd(rc, 2.709956744659734086, 1e-12, "atoc13", "A/rc", status)
    vvd(dc, 0.1741696500898471366, 1e-12, "atoc13", "A/dc", status)
    viv(j, 0, "atoc13", "A/j", status)


def test_atoi13():
    utc1 = 2456384.5
    utc2 = 0.969254051
    dut1 = 0.1550675
    elong = -0.527800806
    phi = -1.2345856
    hm = 2738.0
    xp = 2.47230737e-7
    yp = 1.82640464e-6
    phpa = 731.0
    tc = 12.8
    rh = 0.59
    wl = 0.55
    ob1 = 2.710085107986886201
    ob2 = 0.1717653435758265198
    ri, di, j = erfa_ufunc.atoi13 ("R", ob1, ob2, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    vvd(ri, 2.710121574447540810, 1e-12, "atoi13", "R/ri", status)
    vvd(di, 0.1729371839116608778, 1e-12, "atoi13", "R/di", status)
    viv(j, 0, "atoi13", "R/J", status)
    ob1 = -0.09247619879782006106
    ob2 = 0.1717653435758265198
    ri, di, j = erfa_ufunc.atoi13 ("H", ob1, ob2, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    vvd(ri, 2.710121574448138676, 1e-12, "atoi13", "H/ri", status)
    vvd(di, 0.1729371839116608778, 1e-12, "atoi13", "H/di", status)
    viv(j, 0, "atoi13", "H/J", status)
    ob1 = 0.09233952224794989993
    ob2 = 1.407758704513722461
    ri, di, j = erfa_ufunc.atoi13 ("A", ob1, ob2, utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    vvd(ri, 2.710121574448138676, 1e-12, "atoi13", "A/ri", status)
    vvd(di, 0.1729371839116608781, 1e-12, "atoi13", "A/di", status)
    viv(j, 0, "atoi13", "A/J", status)


def test_atoiq():
    utc1 = 2456384.5
    utc2 = 0.969254051
    dut1 = 0.1550675
    elong = -0.527800806
    phi = -1.2345856
    hm = 2738.0
    xp = 2.47230737e-7
    yp = 1.82640464e-6
    phpa = 731.0
    tc = 12.8
    rh = 0.59
    wl = 0.55
    astrom, j = erfa_ufunc.apio13(utc1, utc2, dut1, elong, phi, hm, xp, yp, phpa, tc, rh, wl)
    ob1 = 2.710085107986886201
    ob2 = 0.1717653435758265198
    ri, di = erfa_ufunc.atoiq("R", ob1, ob2, astrom)
    vvd(ri, 2.710121574447540810, 1e-12, "atoiq", "R/ri", status)
    vvd(di, 0.17293718391166087785, 1e-12, "atoiq", "R/di", status)
    ob1 = -0.09247619879782006106
    ob2 = 0.1717653435758265198
    ri, di = erfa_ufunc.atoiq("H", ob1, ob2, astrom)
    vvd(ri, 2.710121574448138676, 1e-12, "atoiq", "H/ri", status)
    vvd(di, 0.1729371839116608778, 1e-12, "atoiq", "H/di", status)
    ob1 = 0.09233952224794989993
    ob2 = 1.407758704513722461
    ri, di = erfa_ufunc.atoiq("A", ob1, ob2, astrom)
    vvd(ri, 2.710121574448138676, 1e-12, "atoiq", "A/ri", status)
    vvd(di, 0.1729371839116608781, 1e-12, "atoiq", "A/di", status)


@pytest.mark.xfail(reason='do not yet support no-input ufuncs')
def test_bi00():
    dpsibi, depsbi, dra = erfa_ufunc.bi00()
    vvd(dpsibi, -0.2025309152835086613e-6, 1e-12, "bi00", "dpsibi", status)
    vvd(depsbi, -0.3306041454222147847e-7, 1e-12, "bi00", "depsbi", status)
    vvd(dra, -0.7078279744199225506e-7, 1e-12, "bi00", "dra", status)


def test_bp00():
    rb = np.empty((3, 3), float)
    rp = np.empty((3, 3), float)
    rbp = np.empty((3, 3), float)
    rb, rp, rbp = erfa_ufunc.bp00(2400000.5, 50123.9999)
    vvd(rb[0][0], 0.9999999999999942498, 1e-12, "bp00", "rb11", status)
    vvd(rb[0][1], -0.7078279744199196626e-7, 1e-16, "bp00", "rb12", status)
    vvd(rb[0][2], 0.8056217146976134152e-7, 1e-16, "bp00", "rb13", status)
    vvd(rb[1][0], 0.7078279477857337206e-7, 1e-16, "bp00", "rb21", status)
    vvd(rb[1][1], 0.9999999999999969484, 1e-12, "bp00", "rb22", status)
    vvd(rb[1][2], 0.3306041454222136517e-7, 1e-16, "bp00", "rb23", status)
    vvd(rb[2][0], -0.8056217380986972157e-7, 1e-16, "bp00", "rb31", status)
    vvd(rb[2][1], -0.3306040883980552500e-7, 1e-16, "bp00", "rb32", status)
    vvd(rb[2][2], 0.9999999999999962084, 1e-12, "bp00", "rb33", status)
    vvd(rp[0][0], 0.9999995504864048241, 1e-12, "bp00", "rp11", status)
    vvd(rp[0][1], 0.8696113836207084411e-3, 1e-14, "bp00", "rp12", status)
    vvd(rp[0][2], 0.3778928813389333402e-3, 1e-14, "bp00", "rp13", status)
    vvd(rp[1][0], -0.8696113818227265968e-3, 1e-14, "bp00", "rp21", status)
    vvd(rp[1][1], 0.9999996218879365258, 1e-12, "bp00", "rp22", status)
    vvd(rp[1][2], -0.1690679263009242066e-6, 1e-14, "bp00", "rp23", status)
    vvd(rp[2][0], -0.3778928854764695214e-3, 1e-14, "bp00", "rp31", status)
    vvd(rp[2][1], -0.1595521004195286491e-6, 1e-14, "bp00", "rp32", status)
    vvd(rp[2][2], 0.9999999285984682756, 1e-12, "bp00", "rp33", status)
    vvd(rbp[0][0], 0.9999995505175087260, 1e-12, "bp00", "rbp11", status)
    vvd(rbp[0][1], 0.8695405883617884705e-3, 1e-14, "bp00", "rbp12", status)
    vvd(rbp[0][2], 0.3779734722239007105e-3, 1e-14, "bp00", "rbp13", status)
    vvd(rbp[1][0], -0.8695405990410863719e-3, 1e-14, "bp00", "rbp21", status)
    vvd(rbp[1][1], 0.9999996219494925900, 1e-12, "bp00", "rbp22", status)
    vvd(rbp[1][2], -0.1360775820404982209e-6, 1e-14, "bp00", "rbp23", status)
    vvd(rbp[2][0], -0.3779734476558184991e-3, 1e-14, "bp00", "rbp31", status)
    vvd(rbp[2][1], -0.1925857585832024058e-6, 1e-14, "bp00", "rbp32", status)
    vvd(rbp[2][2], 0.9999999285680153377, 1e-12, "bp00", "rbp33", status)


def test_bp06():
    rb = np.empty((3, 3), float)
    rp = np.empty((3, 3), float)
    rbp = np.empty((3, 3), float)
    rb, rp, rbp = erfa_ufunc.bp06(2400000.5, 50123.9999)
    vvd(rb[0][0], 0.9999999999999942497, 1e-12, "bp06", "rb11", status)
    vvd(rb[0][1], -0.7078368960971557145e-7, 1e-14, "bp06", "rb12", status)
    vvd(rb[0][2], 0.8056213977613185606e-7, 1e-14, "bp06", "rb13", status)
    vvd(rb[1][0], 0.7078368694637674333e-7, 1e-14, "bp06", "rb21", status)
    vvd(rb[1][1], 0.9999999999999969484, 1e-12, "bp06", "rb22", status)
    vvd(rb[1][2], 0.3305943742989134124e-7, 1e-14, "bp06", "rb23", status)
    vvd(rb[2][0], -0.8056214211620056792e-7, 1e-14, "bp06", "rb31", status)
    vvd(rb[2][1], -0.3305943172740586950e-7, 1e-14, "bp06", "rb32", status)
    vvd(rb[2][2], 0.9999999999999962084, 1e-12, "bp06", "rb33", status)
    vvd(rp[0][0], 0.9999995504864960278, 1e-12, "bp06", "rp11", status)
    vvd(rp[0][1], 0.8696112578855404832e-3, 1e-14, "bp06", "rp12", status)
    vvd(rp[0][2], 0.3778929293341390127e-3, 1e-14, "bp06", "rp13", status)
    vvd(rp[1][0], -0.8696112560510186244e-3, 1e-14, "bp06", "rp21", status)
    vvd(rp[1][1], 0.9999996218880458820, 1e-12, "bp06", "rp22", status)
    vvd(rp[1][2], -0.1691646168941896285e-6, 1e-14, "bp06", "rp23", status)
    vvd(rp[2][0], -0.3778929335557603418e-3, 1e-14, "bp06", "rp31", status)
    vvd(rp[2][1], -0.1594554040786495076e-6, 1e-14, "bp06", "rp32", status)
    vvd(rp[2][2], 0.9999999285984501222, 1e-12, "bp06", "rp33", status)
    vvd(rbp[0][0], 0.9999995505176007047, 1e-12, "bp06", "rbp11", status)
    vvd(rbp[0][1], 0.8695404617348208406e-3, 1e-14, "bp06", "rbp12", status)
    vvd(rbp[0][2], 0.3779735201865589104e-3, 1e-14, "bp06", "rbp13", status)
    vvd(rbp[1][0], -0.8695404723772031414e-3, 1e-14, "bp06", "rbp21", status)
    vvd(rbp[1][1], 0.9999996219496027161, 1e-12, "bp06", "rbp22", status)
    vvd(rbp[1][2], -0.1361752497080270143e-6, 1e-14, "bp06", "rbp23", status)
    vvd(rbp[2][0], -0.3779734957034089490e-3, 1e-14, "bp06", "rbp31", status)
    vvd(rbp[2][1], -0.1924880847894457113e-6, 1e-14, "bp06", "rbp32", status)
    vvd(rbp[2][2], 0.9999999285679971958, 1e-12, "bp06", "rbp33", status)


def test_bpn2xy():
    rbpn = np.empty((3, 3), float)
    rbpn[0][0] = 9.999962358680738e-1
    rbpn[0][1] = -2.516417057665452e-3
    rbpn[0][2] = -1.093569785342370e-3
    rbpn[1][0] = 2.516462370370876e-3
    rbpn[1][1] = 9.999968329010883e-1
    rbpn[1][2] = 4.006159587358310e-5
    rbpn[2][0] = 1.093465510215479e-3
    rbpn[2][1] = -4.281337229063151e-5
    rbpn[2][2] = 9.999994012499173e-1
    x, y = erfa_ufunc.bpn2xy(rbpn)
    vvd(x,  1.093465510215479e-3, 1e-12, "bpn2xy", "x", status)
    vvd(y, -4.281337229063151e-5, 1e-12, "bpn2xy", "y", status)


def test_c2i00a():
    rc2i = np.empty((3, 3), float)
    rc2i = erfa_ufunc.c2i00a(2400000.5, 53736.0)
    vvd(rc2i[0][0], 0.9999998323037165557, 1e-12, "c2i00a", "11", status)
    vvd(rc2i[0][1], 0.5581526348992140183e-9, 1e-12, "c2i00a", "12", status)
    vvd(rc2i[0][2], -0.5791308477073443415e-3, 1e-12, "c2i00a", "13", status)
    vvd(rc2i[1][0], -0.2384266227870752452e-7, 1e-12, "c2i00a", "21", status)
    vvd(rc2i[1][1], 0.9999999991917405258, 1e-12, "c2i00a", "22", status)
    vvd(rc2i[1][2], -0.4020594955028209745e-4, 1e-12, "c2i00a", "23", status)
    vvd(rc2i[2][0], 0.5791308472168152904e-3, 1e-12, "c2i00a", "31", status)
    vvd(rc2i[2][1], 0.4020595661591500259e-4, 1e-12, "c2i00a", "32", status)
    vvd(rc2i[2][2], 0.9999998314954572304, 1e-12, "c2i00a", "33", status)


def test_c2i00b():
    rc2i = np.empty((3, 3), float)
    rc2i = erfa_ufunc.c2i00b(2400000.5, 53736.0)
    vvd(rc2i[0][0], 0.9999998323040954356, 1e-12, "c2i00b", "11", status)
    vvd(rc2i[0][1], 0.5581526349131823372e-9, 1e-12, "c2i00b", "12", status)
    vvd(rc2i[0][2], -0.5791301934855394005e-3, 1e-12, "c2i00b", "13", status)
    vvd(rc2i[1][0], -0.2384239285499175543e-7, 1e-12, "c2i00b", "21", status)
    vvd(rc2i[1][1], 0.9999999991917574043, 1e-12, "c2i00b", "22", status)
    vvd(rc2i[1][2], -0.4020552974819030066e-4, 1e-12, "c2i00b", "23", status)
    vvd(rc2i[2][0], 0.5791301929950208873e-3, 1e-12, "c2i00b", "31", status)
    vvd(rc2i[2][1], 0.4020553681373720832e-4, 1e-12, "c2i00b", "32", status)
    vvd(rc2i[2][2], 0.9999998314958529887, 1e-12, "c2i00b", "33", status)


def test_c2i06a():
    rc2i = np.empty((3, 3), float)
    rc2i = erfa_ufunc.c2i06a(2400000.5, 53736.0)
    vvd(rc2i[0][0], 0.9999998323037159379, 1e-12, "c2i06a", "11", status)
    vvd(rc2i[0][1], 0.5581121329587613787e-9, 1e-12, "c2i06a", "12", status)
    vvd(rc2i[0][2], -0.5791308487740529749e-3, 1e-12, "c2i06a", "13", status)
    vvd(rc2i[1][0], -0.2384253169452306581e-7, 1e-12, "c2i06a", "21", status)
    vvd(rc2i[1][1], 0.9999999991917467827, 1e-12, "c2i06a", "22", status)
    vvd(rc2i[1][2], -0.4020579392895682558e-4, 1e-12, "c2i06a", "23", status)
    vvd(rc2i[2][0], 0.5791308482835292617e-3, 1e-12, "c2i06a", "31", status)
    vvd(rc2i[2][1], 0.4020580099454020310e-4, 1e-12, "c2i06a", "32", status)
    vvd(rc2i[2][2], 0.9999998314954628695, 1e-12, "c2i06a", "33", status)


def test_c2ibpn():
    rbpn = np.empty((3, 3), float)
    rc2i = np.empty((3, 3), float)
    rbpn[0][0] = 9.999962358680738e-1
    rbpn[0][1] = -2.516417057665452e-3
    rbpn[0][2] = -1.093569785342370e-3
    rbpn[1][0] = 2.516462370370876e-3
    rbpn[1][1] = 9.999968329010883e-1
    rbpn[1][2] = 4.006159587358310e-5
    rbpn[2][0] = 1.093465510215479e-3
    rbpn[2][1] = -4.281337229063151e-5
    rbpn[2][2] = 9.999994012499173e-1
    rc2i = erfa_ufunc.c2ibpn(2400000.5, 50123.9999, rbpn)
    vvd(rc2i[0][0], 0.9999994021664089977, 1e-12, "c2ibpn", "11", status)
    vvd(rc2i[0][1], -0.3869195948017503664e-8, 1e-12, "c2ibpn", "12", status)
    vvd(rc2i[0][2], -0.1093465511383285076e-2, 1e-12, "c2ibpn", "13", status)
    vvd(rc2i[1][0], 0.5068413965715446111e-7, 1e-12, "c2ibpn", "21", status)
    vvd(rc2i[1][1], 0.9999999990835075686, 1e-12, "c2ibpn", "22", status)
    vvd(rc2i[1][2], 0.4281334246452708915e-4, 1e-12, "c2ibpn", "23", status)
    vvd(rc2i[2][0], 0.1093465510215479000e-2, 1e-12, "c2ibpn", "31", status)
    vvd(rc2i[2][1], -0.4281337229063151000e-4, 1e-12, "c2ibpn", "32", status)
    vvd(rc2i[2][2], 0.9999994012499173103, 1e-12, "c2ibpn", "33", status)


def test_c2ixy():
    rc2i = np.empty((3, 3), float)
    x = 0.5791308486706011000e-3
    y = 0.4020579816732961219e-4
    rc2i = erfa_ufunc.c2ixy(2400000.5, 53736, x, y)
    vvd(rc2i[0][0], 0.9999998323037157138, 1e-12, "c2ixy", "11", status)
    vvd(rc2i[0][1], 0.5581526349032241205e-9, 1e-12, "c2ixy", "12", status)
    vvd(rc2i[0][2], -0.5791308491611263745e-3, 1e-12, "c2ixy", "13", status)
    vvd(rc2i[1][0], -0.2384257057469842953e-7, 1e-12, "c2ixy", "21", status)
    vvd(rc2i[1][1], 0.9999999991917468964, 1e-12, "c2ixy", "22", status)
    vvd(rc2i[1][2], -0.4020579110172324363e-4, 1e-12, "c2ixy", "23", status)
    vvd(rc2i[2][0], 0.5791308486706011000e-3, 1e-12, "c2ixy", "31", status)
    vvd(rc2i[2][1], 0.4020579816732961219e-4, 1e-12, "c2ixy", "32", status)
    vvd(rc2i[2][2], 0.9999998314954627590, 1e-12, "c2ixy", "33", status)


def test_c2ixys():
    rc2i = np.empty((3, 3), float)
    x = 0.5791308486706011000e-3
    y = 0.4020579816732961219e-4
    s = -0.1220040848472271978e-7
    rc2i = erfa_ufunc.c2ixys(x, y, s)
    vvd(rc2i[0][0], 0.9999998323037157138, 1e-12, "c2ixys", "11", status)
    vvd(rc2i[0][1], 0.5581984869168499149e-9, 1e-12, "c2ixys", "12", status)
    vvd(rc2i[0][2], -0.5791308491611282180e-3, 1e-12, "c2ixys", "13", status)
    vvd(rc2i[1][0], -0.2384261642670440317e-7, 1e-12, "c2ixys", "21", status)
    vvd(rc2i[1][1], 0.9999999991917468964, 1e-12, "c2ixys", "22", status)
    vvd(rc2i[1][2], -0.4020579110169668931e-4, 1e-12, "c2ixys", "23", status)
    vvd(rc2i[2][0], 0.5791308486706011000e-3, 1e-12, "c2ixys", "31", status)
    vvd(rc2i[2][1], 0.4020579816732961219e-4, 1e-12, "c2ixys", "32", status)
    vvd(rc2i[2][2], 0.9999998314954627590, 1e-12, "c2ixys", "33", status)


def test_c2s():
    p = np.empty((3,), float)
    p[0] = 100.0
    p[1] = -50.0
    p[2] = 25.0
    theta, phi = erfa_ufunc.c2s(p)
    vvd(theta, -0.4636476090008061162, 1e-14, "c2s", "theta", status)
    vvd(phi, 0.2199879773954594463, 1e-14, "c2s", "phi", status)


def test_c2t00a():
    rc2t = np.empty((3, 3), float)
    tta = 2400000.5
    uta = 2400000.5
    ttb = 53736.0
    utb = 53736.0
    xp = 2.55060238e-7
    yp = 1.860359247e-6
    rc2t = erfa_ufunc.c2t00a(tta, ttb, uta, utb, xp, yp)
    vvd(rc2t[0][0], -0.1810332128307182668, 1e-12, "c2t00a", "11", status)
    vvd(rc2t[0][1], 0.9834769806938457836, 1e-12, "c2t00a", "12", status)
    vvd(rc2t[0][2], 0.6555535638688341725e-4, 1e-12, "c2t00a", "13", status)
    vvd(rc2t[1][0], -0.9834768134135984552, 1e-12, "c2t00a", "21", status)
    vvd(rc2t[1][1], -0.1810332203649520727, 1e-12, "c2t00a", "22", status)
    vvd(rc2t[1][2], 0.5749801116141056317e-3, 1e-12, "c2t00a", "23", status)
    vvd(rc2t[2][0], 0.5773474014081406921e-3, 1e-12, "c2t00a", "31", status)
    vvd(rc2t[2][1], 0.3961832391770163647e-4, 1e-12, "c2t00a", "32", status)
    vvd(rc2t[2][2], 0.9999998325501692289, 1e-12, "c2t00a", "33", status)


def test_c2t00b():
    rc2t = np.empty((3, 3), float)
    tta = 2400000.5
    uta = 2400000.5
    ttb = 53736.0
    utb = 53736.0
    xp = 2.55060238e-7
    yp = 1.860359247e-6
    rc2t = erfa_ufunc.c2t00b(tta, ttb, uta, utb, xp, yp)
    vvd(rc2t[0][0], -0.1810332128439678965, 1e-12, "c2t00b", "11", status)
    vvd(rc2t[0][1], 0.9834769806913872359, 1e-12, "c2t00b", "12", status)
    vvd(rc2t[0][2], 0.6555565082458415611e-4, 1e-12, "c2t00b", "13", status)
    vvd(rc2t[1][0], -0.9834768134115435923, 1e-12, "c2t00b", "21", status)
    vvd(rc2t[1][1], -0.1810332203784001946, 1e-12, "c2t00b", "22", status)
    vvd(rc2t[1][2], 0.5749793922030017230e-3, 1e-12, "c2t00b", "23", status)
    vvd(rc2t[2][0], 0.5773467471863534901e-3, 1e-12, "c2t00b", "31", status)
    vvd(rc2t[2][1], 0.3961790411549945020e-4, 1e-12, "c2t00b", "32", status)
    vvd(rc2t[2][2], 0.9999998325505635738, 1e-12, "c2t00b", "33", status)


def test_c2t06a():
    rc2t = np.empty((3, 3), float)
    tta = 2400000.5
    uta = 2400000.5
    ttb = 53736.0
    utb = 53736.0
    xp = 2.55060238e-7
    yp = 1.860359247e-6
    rc2t = erfa_ufunc.c2t06a(tta, ttb, uta, utb, xp, yp)
    vvd(rc2t[0][0], -0.1810332128305897282, 1e-12, "c2t06a", "11", status)
    vvd(rc2t[0][1], 0.9834769806938592296, 1e-12, "c2t06a", "12", status)
    vvd(rc2t[0][2], 0.6555550962998436505e-4, 1e-12, "c2t06a", "13", status)
    vvd(rc2t[1][0], -0.9834768134136214897, 1e-12, "c2t06a", "21", status)
    vvd(rc2t[1][1], -0.1810332203649130832, 1e-12, "c2t06a", "22", status)
    vvd(rc2t[1][2], 0.5749800844905594110e-3, 1e-12, "c2t06a", "23", status)
    vvd(rc2t[2][0], 0.5773474024748545878e-3, 1e-12, "c2t06a", "31", status)
    vvd(rc2t[2][1], 0.3961816829632690581e-4, 1e-12, "c2t06a", "32", status)
    vvd(rc2t[2][2], 0.9999998325501747785, 1e-12, "c2t06a", "33", status)


def test_c2tcio():
    rc2i = np.empty((3, 3), float)
    rpom = np.empty((3, 3), float)
    rc2t = np.empty((3, 3), float)
    rc2i[0][0] = 0.9999998323037164738
    rc2i[0][1] = 0.5581526271714303683e-9
    rc2i[0][2] = -0.5791308477073443903e-3
    rc2i[1][0] = -0.2384266227524722273e-7
    rc2i[1][1] = 0.9999999991917404296
    rc2i[1][2] = -0.4020594955030704125e-4
    rc2i[2][0] = 0.5791308472168153320e-3
    rc2i[2][1] = 0.4020595661593994396e-4
    rc2i[2][2] = 0.9999998314954572365
    era = 1.75283325530307
    rpom[0][0] = 0.9999999999999674705
    rpom[0][1] = -0.1367174580728847031e-10
    rpom[0][2] = 0.2550602379999972723e-6
    rpom[1][0] = 0.1414624947957029721e-10
    rpom[1][1] = 0.9999999999982694954
    rpom[1][2] = -0.1860359246998866338e-5
    rpom[2][0] = -0.2550602379741215275e-6
    rpom[2][1] = 0.1860359247002413923e-5
    rpom[2][2] = 0.9999999999982369658
    rc2t = erfa_ufunc.c2tcio(rc2i, era, rpom)
    vvd(rc2t[0][0], -0.1810332128307110439, 1e-12, "c2tcio", "11", status)
    vvd(rc2t[0][1], 0.9834769806938470149, 1e-12, "c2tcio", "12", status)
    vvd(rc2t[0][2], 0.6555535638685466874e-4, 1e-12, "c2tcio", "13", status)
    vvd(rc2t[1][0], -0.9834768134135996657, 1e-12, "c2tcio", "21", status)
    vvd(rc2t[1][1], -0.1810332203649448367, 1e-12, "c2tcio", "22", status)
    vvd(rc2t[1][2], 0.5749801116141106528e-3, 1e-12, "c2tcio", "23", status)
    vvd(rc2t[2][0], 0.5773474014081407076e-3, 1e-12, "c2tcio", "31", status)
    vvd(rc2t[2][1], 0.3961832391772658944e-4, 1e-12, "c2tcio", "32", status)
    vvd(rc2t[2][2], 0.9999998325501691969, 1e-12, "c2tcio", "33", status)


def test_c2teqx():
    rbpn = np.empty((3, 3), float)
    rpom = np.empty((3, 3), float)
    rc2t = np.empty((3, 3), float)
    rbpn[0][0] = 0.9999989440476103608
    rbpn[0][1] = -0.1332881761240011518e-2
    rbpn[0][2] = -0.5790767434730085097e-3
    rbpn[1][0] = 0.1332858254308954453e-2
    rbpn[1][1] = 0.9999991109044505944
    rbpn[1][2] = -0.4097782710401555759e-4
    rbpn[2][0] = 0.5791308472168153320e-3
    rbpn[2][1] = 0.4020595661593994396e-4
    rbpn[2][2] = 0.9999998314954572365
    gst = 1.754166138040730516
    rpom[0][0] = 0.9999999999999674705
    rpom[0][1] = -0.1367174580728847031e-10
    rpom[0][2] = 0.2550602379999972723e-6
    rpom[1][0] = 0.1414624947957029721e-10
    rpom[1][1] = 0.9999999999982694954
    rpom[1][2] = -0.1860359246998866338e-5
    rpom[2][0] = -0.2550602379741215275e-6
    rpom[2][1] = 0.1860359247002413923e-5
    rpom[2][2] = 0.9999999999982369658
    rc2t = erfa_ufunc.c2teqx(rbpn, gst, rpom)
    vvd(rc2t[0][0], -0.1810332128528685730, 1e-12, "c2teqx", "11", status)
    vvd(rc2t[0][1], 0.9834769806897685071, 1e-12, "c2teqx", "12", status)
    vvd(rc2t[0][2], 0.6555535639982634449e-4, 1e-12, "c2teqx", "13", status)
    vvd(rc2t[1][0], -0.9834768134095211257, 1e-12, "c2teqx", "21", status)
    vvd(rc2t[1][1], -0.1810332203871023800, 1e-12, "c2teqx", "22", status)
    vvd(rc2t[1][2], 0.5749801116126438962e-3, 1e-12, "c2teqx", "23", status)
    vvd(rc2t[2][0], 0.5773474014081539467e-3, 1e-12, "c2teqx", "31", status)
    vvd(rc2t[2][1], 0.3961832391768640871e-4, 1e-12, "c2teqx", "32", status)
    vvd(rc2t[2][2], 0.9999998325501691969, 1e-12, "c2teqx", "33", status)


def test_c2tpe():
    rc2t = np.empty((3, 3), float)
    tta = 2400000.5
    uta = 2400000.5
    ttb = 53736.0
    utb = 53736.0
    deps = 0.4090789763356509900
    dpsi = -0.9630909107115582393e-5
    xp = 2.55060238e-7
    yp = 1.860359247e-6
    rc2t = erfa_ufunc.c2tpe(tta, ttb, uta, utb, dpsi, deps, xp, yp)
    vvd(rc2t[0][0], -0.1813677995763029394, 1e-12, "c2tpe", "11", status)
    vvd(rc2t[0][1], 0.9023482206891683275, 1e-12, "c2tpe", "12", status)
    vvd(rc2t[0][2], -0.3909902938641085751, 1e-12, "c2tpe", "13", status)
    vvd(rc2t[1][0], -0.9834147641476804807, 1e-12, "c2tpe", "21", status)
    vvd(rc2t[1][1], -0.1659883635434995121, 1e-12, "c2tpe", "22", status)
    vvd(rc2t[1][2], 0.7309763898042819705e-1, 1e-12, "c2tpe", "23", status)
    vvd(rc2t[2][0], 0.1059685430673215247e-2, 1e-12, "c2tpe", "31", status)
    vvd(rc2t[2][1], 0.3977631855605078674, 1e-12, "c2tpe", "32", status)
    vvd(rc2t[2][2], 0.9174875068792735362, 1e-12, "c2tpe", "33", status)


def test_c2txy():
    rc2t = np.empty((3, 3), float)
    tta = 2400000.5
    uta = 2400000.5
    ttb = 53736.0
    utb = 53736.0
    x = 0.5791308486706011000e-3
    y = 0.4020579816732961219e-4
    xp = 2.55060238e-7
    yp = 1.860359247e-6
    rc2t = erfa_ufunc.c2txy(tta, ttb, uta, utb, x, y, xp, yp)
    vvd(rc2t[0][0], -0.1810332128306279253, 1e-12, "c2txy", "11", status)
    vvd(rc2t[0][1], 0.9834769806938520084, 1e-12, "c2txy", "12", status)
    vvd(rc2t[0][2], 0.6555551248057665829e-4, 1e-12, "c2txy", "13", status)
    vvd(rc2t[1][0], -0.9834768134136142314, 1e-12, "c2txy", "21", status)
    vvd(rc2t[1][1], -0.1810332203649529312, 1e-12, "c2txy", "22", status)
    vvd(rc2t[1][2], 0.5749800843594139912e-3, 1e-12, "c2txy", "23", status)
    vvd(rc2t[2][0], 0.5773474028619264494e-3, 1e-12, "c2txy", "31", status)
    vvd(rc2t[2][1], 0.3961816546911624260e-4, 1e-12, "c2txy", "32", status)
    vvd(rc2t[2][2], 0.9999998325501746670, 1e-12, "c2txy", "33", status)


def test_cal2jd():
    djm0, djm, j = erfa_ufunc.cal2jd(2003, 6, 1)
    vvd(djm0, 2400000.5, 0.0, "cal2jd", "djm0", status)
    vvd(djm,    52791.0, 0.0, "cal2jd", "djm", status)
    viv(j, 0, "cal2jd", "j", status)


def test_cp():
    p = np.empty((3,), float)
    c = np.empty((3,), float)
    p[0] = 0.3
    p[1] = 1.2
    p[2] = -2.5
    c = erfa_ufunc.cp(p)
    vvd(c[0],  0.3, 0.0, "cp", "1", status)
    vvd(c[1],  1.2, 0.0, "cp", "2", status)
    vvd(c[2], -2.5, 0.0, "cp", "3", status)


def test_cpv():
    pv = np.empty((), erfa_ufunc.dt_pv)
    c = np.empty((), erfa_ufunc.dt_pv)
    pv['p'][0] = 0.3
    pv['p'][1] = 1.2
    pv['p'][2] = -2.5
    pv['v'][0] = -0.5
    pv['v'][1] = 3.1
    pv['v'][2] = 0.9
    c = erfa_ufunc.cpv(pv)
    vvd(c[0][0],  0.3, 0.0, "cpv", "p1", status)
    vvd(c[0][1],  1.2, 0.0, "cpv", "p2", status)
    vvd(c[0][2], -2.5, 0.0, "cpv", "p3", status)
    vvd(c[1][0], -0.5, 0.0, "cpv", "v1", status)
    vvd(c[1][1],  3.1, 0.0, "cpv", "v2", status)
    vvd(c[1][2],  0.9, 0.0, "cpv", "v3", status)


def test_cr():
    r = np.empty((3, 3), float)
    c = np.empty((3, 3), float)
    r[0][0] = 2.0
    r[0][1] = 3.0
    r[0][2] = 2.0
    r[1][0] = 3.0
    r[1][1] = 2.0
    r[1][2] = 3.0
    r[2][0] = 3.0
    r[2][1] = 4.0
    r[2][2] = 5.0
    c = erfa_ufunc.cr(r)
    vvd(c[0][0], 2.0, 0.0, "cr", "11", status)
    vvd(c[0][1], 3.0, 0.0, "cr", "12", status)
    vvd(c[0][2], 2.0, 0.0, "cr", "13", status)
    vvd(c[1][0], 3.0, 0.0, "cr", "21", status)
    vvd(c[1][1], 2.0, 0.0, "cr", "22", status)
    vvd(c[1][2], 3.0, 0.0, "cr", "23", status)
    vvd(c[2][0], 3.0, 0.0, "cr", "31", status)
    vvd(c[2][1], 4.0, 0.0, "cr", "32", status)
    vvd(c[2][2], 5.0, 0.0, "cr", "33", status)


def test_d2dtf():
    iy, im, id, ihmsf, j = erfa_ufunc.d2dtf("UTC", 5, 2400000.5, 49533.99999)
    viv(iy, 1994, "d2dtf", "y", status)
    viv(im, 6, "d2dtf", "mo", status)
    viv(id, 30, "d2dtf", "d", status)
    viv(ihmsf[0], 23, "d2dtf", "h", status)
    viv(ihmsf[1], 59, "d2dtf", "m", status)
    viv(ihmsf[2], 60, "d2dtf", "s", status)
    viv(ihmsf[3], 13599, "d2dtf", "f", status)
    viv(j, 0, "d2dtf", "j", status)


def test_d2tf():
    s, ihmsf = erfa_ufunc.d2tf(4, -0.987654321)
    viv(s[0], b'-', "d2tf", "s", status)
    viv(ihmsf[0], 23, "d2tf", "0", status)
    viv(ihmsf[1], 42, "d2tf", "1", status)
    viv(ihmsf[2], 13, "d2tf", "2", status)
    viv(ihmsf[3], 3333, "d2tf", "3", status)


def test_dat():
    deltat, j = erfa_ufunc.dat(2003, 6, 1, 0.0)
    vvd(deltat, 32.0, 0.0, "dat", "d1", status)
    viv(j, 0, "dat", "j1", status)
    deltat, j = erfa_ufunc.dat(2008, 1, 17, 0.0)
    vvd(deltat, 33.0, 0.0, "dat", "d2", status)
    viv(j, 0, "dat", "j2", status)
    deltat, j = erfa_ufunc.dat(2017, 9, 1, 0.0)
    vvd(deltat, 37.0, 0.0, "dat", "d3", status)
    viv(j, 0, "dat", "j3", status)


def test_dtdb():
    dtdb = erfa_ufunc.dtdb(2448939.5, 0.123, 0.76543, 5.0123, 5525.242, 3190.0)
    vvd(dtdb, -0.1280368005936998991e-2, 1e-15, "dtdb", "", status)


def test_dtf2d():
    u1, u2, j = erfa_ufunc.dtf2d("UTC", 1994, 6, 30, 23, 59, 60.13599)
    vvd(u1+u2, 2449534.49999, 1e-6, "dtf2d", "u", status)
    viv(j, 0, "dtf2d", "j", status)


def test_eceq06():
    date1 = 2456165.5
    date2 = 0.401182685
    dl = 5.1
    db = -0.9
    dr, dd = erfa_ufunc.eceq06(date1, date2, dl, db)
    vvd(dr, 5.533459733613627767, 1e-14, "eceq06", "dr", status)
    vvd(dd, -1.246542932554480576, 1e-14, "eceq06", "dd", status)


def test_ecm06():
    rm = np.empty((3, 3), float)
    date1 = 2456165.5
    date2 = 0.401182685
    rm = erfa_ufunc.ecm06(date1, date2)
    vvd(rm[0][0], 0.9999952427708701137, 1e-14, "ecm06", "rm11", status)
    vvd(rm[0][1], -0.2829062057663042347e-2, 1e-14, "ecm06", "rm12", status)
    vvd(rm[0][2], -0.1229163741100017629e-2, 1e-14, "ecm06", "rm13", status)
    vvd(rm[1][0], 0.3084546876908653562e-2, 1e-14, "ecm06", "rm21", status)
    vvd(rm[1][1], 0.9174891871550392514, 1e-14, "ecm06", "rm22", status)
    vvd(rm[1][2], 0.3977487611849338124, 1e-14, "ecm06", "rm23", status)
    vvd(rm[2][0], 0.2488512951527405928e-5, 1e-14, "ecm06", "rm31", status)
    vvd(rm[2][1], -0.3977506604161195467, 1e-14, "ecm06", "rm32", status)
    vvd(rm[2][2], 0.9174935488232863071, 1e-14, "ecm06", "rm33", status)


def test_ee00():
    epsa = 0.4090789763356509900
    dpsi = -0.9630909107115582393e-5
    ee = erfa_ufunc.ee00(2400000.5, 53736.0, epsa, dpsi)
    vvd(ee, -0.8834193235367965479e-5, 1e-18, "ee00", "", status)


def test_ee00a():
    ee = erfa_ufunc.ee00a(2400000.5, 53736.0)
    vvd(ee, -0.8834192459222588227e-5, 1e-18, "ee00a", "", status)


def test_ee00b():
    ee = erfa_ufunc.ee00b(2400000.5, 53736.0)
    vvd(ee, -0.8835700060003032831e-5, 1e-18, "ee00b", "", status)


def test_ee06a():
    ee = erfa_ufunc.ee06a(2400000.5, 53736.0)
    vvd(ee, -0.8834195072043790156e-5, 1e-15, "ee06a", "", status)


def test_eect00():
    eect = erfa_ufunc.eect00(2400000.5, 53736.0)
    vvd(eect, 0.2046085004885125264e-8, 1e-20, "eect00", "", status)


def test_eform():
    a, f, j = erfa_ufunc.eform(0)
    viv(j, -1, "eform", "j0", status)
    a, f, j = erfa_ufunc.eform(erfa.WGS84)
    viv(j, 0, "eform", "j1", status)
    vvd(a, 6378137.0, 1e-10, "eform", "a1", status)
    vvd(f, 0.3352810664747480720e-2, 1e-18, "eform", "f1", status)
    a, f, j = erfa_ufunc.eform(erfa.GRS80)
    viv(j, 0, "eform", "j2", status)
    vvd(a, 6378137.0, 1e-10, "eform", "a2", status)
    vvd(f, 0.3352810681182318935e-2, 1e-18, "eform", "f2", status)
    a, f, j = erfa_ufunc.eform(erfa.WGS72)
    viv(j, 0, "eform", "j2", status)
    vvd(a, 6378135.0, 1e-10, "eform", "a3", status)
    vvd(f, 0.3352779454167504862e-2, 1e-18, "eform", "f3", status)
    a, f, j = erfa_ufunc.eform(4)
    viv(j, -1, "eform", "j3", status)


def test_eo06a():
    eo = erfa_ufunc.eo06a(2400000.5, 53736.0)
    vvd(eo, -0.1332882371941833644e-2, 1e-15, "eo06a", "", status)


def test_eors():
    rnpb = np.empty((3, 3), float)
    rnpb[0][0] = 0.9999989440476103608
    rnpb[0][1] = -0.1332881761240011518e-2
    rnpb[0][2] = -0.5790767434730085097e-3
    rnpb[1][0] = 0.1332858254308954453e-2
    rnpb[1][1] = 0.9999991109044505944
    rnpb[1][2] = -0.4097782710401555759e-4
    rnpb[2][0] = 0.5791308472168153320e-3
    rnpb[2][1] = 0.4020595661593994396e-4
    rnpb[2][2] = 0.9999998314954572365
    s = -0.1220040848472271978e-7
    eo = erfa_ufunc.eors(rnpb, s)
    vvd(eo, -0.1332882715130744606e-2, 1e-14, "eors", "", status)


def test_epb():
    epb = erfa_ufunc.epb(2415019.8135, 30103.18648)
    vvd(epb, 1982.418424159278580, 1e-12, "epb", "", status)


def test_epb2jd():
    epb = 1957.3
    djm0, djm = erfa_ufunc.epb2jd(epb)
    vvd(djm0, 2400000.5, 1e-9, "epb2jd", "djm0", status)
    vvd(djm, 35948.1915101513, 1e-9, "epb2jd", "mjd", status)


def test_epj():
    epj = erfa_ufunc.epj(2451545, -7392.5)
    vvd(epj, 1979.760438056125941, 1e-12, "epj", "", status)


def test_epj2jd():
    epj = 1996.8
    djm0, djm = erfa_ufunc.epj2jd(epj)
    vvd(djm0, 2400000.5, 1e-9, "epj2jd", "djm0", status)
    vvd(djm,    50375.7, 1e-9, "epj2jd", "mjd",  status)


def test_epv00():
    pvh = np.empty((), erfa_ufunc.dt_pv)
    pvb = np.empty((), erfa_ufunc.dt_pv)
    pvh, pvb, j = erfa_ufunc.epv00(2400000.5, 53411.52501161)
    vvd(pvh[0][0], -0.7757238809297706813, 1e-14, "epv00", "ph(x)", status)
    vvd(pvh[0][1], 0.5598052241363340596, 1e-14, "epv00", "ph(y)", status)
    vvd(pvh[0][2], 0.2426998466481686993, 1e-14, "epv00", "ph(z)", status)
    vvd(pvh[1][0], -0.1091891824147313846e-1, 1e-15, "epv00", "vh(x)", status)
    vvd(pvh[1][1], -0.1247187268440845008e-1, 1e-15, "epv00", "vh(y)", status)
    vvd(pvh[1][2], -0.5407569418065039061e-2, 1e-15, "epv00", "vh(z)", status)
    vvd(pvb[0][0], -0.7714104440491111971, 1e-14, "epv00", "pb(x)", status)
    vvd(pvb[0][1], 0.5598412061824171323, 1e-14, "epv00", "pb(y)", status)
    vvd(pvb[0][2], 0.2425996277722452400, 1e-14, "epv00", "pb(z)", status)
    vvd(pvb[1][0], -0.1091874268116823295e-1, 1e-15, "epv00", "vb(x)", status)
    vvd(pvb[1][1], -0.1246525461732861538e-1, 1e-15, "epv00", "vb(y)", status)
    vvd(pvb[1][2], -0.5404773180966231279e-2, 1e-15, "epv00", "vb(z)", status)
    viv(j, 0, "epv00", "j", status)


def test_eqec06():
    date1 = 1234.5
    date2 = 2440000.5
    dr = 1.234
    dd = 0.987
    dl, db = erfa_ufunc.eqec06(date1, date2, dr, dd)
    vvd(dl, 1.342509918994654619, 1e-14, "eqec06", "dl", status)
    vvd(db, 0.5926215259704608132, 1e-14, "eqec06", "db", status)


def test_eqeq94():
    eqeq = erfa_ufunc.eqeq94(2400000.5, 41234.0)
    vvd(eqeq, 0.5357758254609256894e-4, 1e-17, "eqeq94", "", status)


def test_era00():
    era00 = erfa_ufunc.era00(2400000.5, 54388.0)
    vvd(era00, 0.4022837240028158102, 1e-12, "era00", "", status)


def test_fad03():
    vvd(erfa_ufunc.fad03(0.80), 1.946709205396925672, 1e-12, "fad03", "", status)


def test_fae03():
    vvd(erfa_ufunc.fae03(0.80), 1.744713738913081846, 1e-12, "fae03", "", status)


def test_faf03():
    vvd(erfa_ufunc.faf03(0.80), 0.2597711366745499518, 1e-12, "faf03", "", status)


def test_faju03():
    vvd(erfa_ufunc.faju03(0.80), 5.275711665202481138, 1e-12, "faju03", "", status)


def test_fal03():
    vvd(erfa_ufunc.fal03(0.80), 5.132369751108684150, 1e-12, "fal03", "", status)


def test_falp03():
    vvd(erfa_ufunc.falp03(0.80), 6.226797973505507345, 1e-12, "falp03", "", status)


def test_fama03():
    vvd(erfa_ufunc.fama03(0.80), 3.275506840277781492, 1e-12, "fama03", "", status)


def test_fame03():
    vvd(erfa_ufunc.fame03(0.80), 5.417338184297289661, 1e-12, "fame03", "", status)


def test_fane03():
    vvd(erfa_ufunc.fane03(0.80), 2.079343830860413523, 1e-12, "fane03", "", status)


def test_faom03():
    vvd(erfa_ufunc.faom03(0.80), -5.973618440951302183, 1e-12, "faom03", "", status)


def test_fapa03():
    vvd(erfa_ufunc.fapa03(0.80), 0.1950884762240000000e-1, 1e-12, "fapa03", "", status)


def test_fasa03():
    vvd(erfa_ufunc.fasa03(0.80), 5.371574539440827046, 1e-12, "fasa03", "", status)


def test_faur03():
    vvd(erfa_ufunc.faur03(0.80), 5.180636450180413523, 1e-12, "faur03", "", status)


def test_fave03():
    vvd(erfa_ufunc.fave03(0.80), 3.424900460533758000, 1e-12, "fave03", "", status)


def test_fk425():
    r1950 = 0.07626899753879587532
    d1950 = -1.137405378399605780
    dr1950 = 0.1973749217849087460e-4
    dd1950 = 0.5659714913272723189e-5
    p1950 = 0.134
    v1950 = 8.7
    r2000, d2000, dr2000, dd2000, p2000, v2000 = erfa_ufunc.fk425(r1950, d1950, dr1950, dd1950, p1950, v1950)
    vvd(r2000, 0.08757989933556446040, 1e-14, "fk425", "r2000", status)
    vvd(d2000, -1.132279113042091895, 1e-12, "fk425", "d2000", status)
    vvd(dr2000, 0.1953670614474396139e-4, 1e-17, "fk425", "dr2000", status)
    vvd(dd2000, 0.5637686678659640164e-5, 1e-18, "fk425", "dd2000", status)
    vvd(p2000, 0.1339919950582767871, 1e-13, "fk425", "p2000", status)
    vvd(v2000, 8.736999669183529069, 1e-12, "fk425", "v2000", status)


def test_fk45z():
    r1950 = 0.01602284975382960982
    d1950 = -0.1164347929099906024
    bepoch = 1954.677617625256806
    r2000, d2000 = erfa_ufunc.fk45z(r1950, d1950, bepoch)
    vvd(r2000, 0.02719295911606862303, 1e-15, "fk45z", "r2000", status)
    vvd(d2000, -0.1115766001565926892, 1e-13, "fk45z", "d2000", status)


def test_fk524():
    r2000 = 0.8723503576487275595
    d2000 = -0.7517076365138887672
    dr2000 = 0.2019447755430472323e-4
    dd2000 = 0.3541563940505160433e-5
    p2000 = 0.1559
    v2000 = 86.87
    r1950, d1950, dr1950, dd1950, p1950, v1950 = erfa_ufunc.fk524(r2000, d2000, dr2000, dd2000, p2000, v2000)
    vvd(r1950, 0.8636359659799603487, 1e-13, "fk524", "r1950", status)
    vvd(d1950, -0.7550281733160843059, 1e-13, "fk524", "d1950", status)
    vvd(dr1950, 0.2023628192747172486e-4, 1e-17, "fk524", "dr1950", status)
    vvd(dd1950, 0.3624459754935334718e-5, 1e-18, "fk524", "dd1950", status)
    vvd(p1950, 0.1560079963299390241, 1e-13, "fk524", "p1950", status)
    vvd(v1950, 86.79606353469163751, 1e-11, "fk524", "v1950", status)


def test_fk52h():
    r5  = 1.76779433
    d5  = -0.2917517103
    dr5 = -1.91851572e-7
    dd5 = -5.8468475e-6
    px5 = 0.379210
    rv5 = -7.6
    rh, dh, drh, ddh, pxh, rvh = erfa_ufunc.fk52h(r5, d5, dr5, dd5, px5, rv5)
    vvd(rh, 1.767794226299947632, 1e-14, "fk52h", "ra", status)
    vvd(dh,  -0.2917516070530391757, 1e-14, "fk52h", "dec", status)
    vvd(drh, -0.1961874125605721270e-6,1e-19, "fk52h", "dr5", status)
    vvd(ddh, -0.58459905176693911e-5, 1e-19, "fk52h", "dd5", status)
    vvd(pxh,  0.37921, 1e-14, "fk52h", "px", status)
    vvd(rvh, -7.6000000940000254, 1e-11, "fk52h", "rv", status)


def test_fk54z():
    r2000 = 0.02719026625066316119
    d2000 = -0.1115815170738754813
    bepoch = 1954.677308160316374
    r1950, d1950, dr1950, dd1950 = erfa_ufunc.fk54z(r2000, d2000, bepoch)
    vvd(r1950, 0.01602015588390065476, 1e-14, "fk54z", "r1950", status)
    vvd(d1950, -0.1164397101110765346, 1e-13, "fk54z", "d1950", status)
    vvd(dr1950, -0.1175712648471090704e-7, 1e-20, "fk54z", "dr1950", status)
    vvd(dd1950, 0.2108109051316431056e-7, 1e-20, "fk54z", "dd1950", status)


@pytest.mark.xfail(reason='do not yet support no-input ufuncs')
def test_fk5hip():
    r5h = np.empty((3, 3), float)
    s5h = np.empty((3,), float)
    r5h, s5h = erfa_ufunc.fk5hip()
    vvd(r5h[0][0], 0.9999999999999928638, 1e-14, "fk5hip", "11", status)
    vvd(r5h[0][1], 0.1110223351022919694e-6, 1e-17, "fk5hip", "12", status)
    vvd(r5h[0][2], 0.4411803962536558154e-7, 1e-17, "fk5hip", "13", status)
    vvd(r5h[1][0], -0.1110223308458746430e-6, 1e-17, "fk5hip", "21", status)
    vvd(r5h[1][1], 0.9999999999999891830, 1e-14, "fk5hip", "22", status)
    vvd(r5h[1][2], -0.9647792498984142358e-7, 1e-17, "fk5hip", "23", status)
    vvd(r5h[2][0], -0.4411805033656962252e-7, 1e-17, "fk5hip", "31", status)
    vvd(r5h[2][1], 0.9647792009175314354e-7, 1e-17, "fk5hip", "32", status)
    vvd(r5h[2][2], 0.9999999999999943728, 1e-14, "fk5hip", "33", status)
    vvd(s5h[0], -0.1454441043328607981e-8, 1e-17, "fk5hip", "s1", status)
    vvd(s5h[1], 0.2908882086657215962e-8, 1e-17, "fk5hip", "s2", status)
    vvd(s5h[2], 0.3393695767766751955e-8, 1e-17, "fk5hip", "s3", status)


def test_fk5hz():
    r5 = 1.76779433
    d5 = -0.2917517103
    rh, dh = erfa_ufunc.fk5hz(r5, d5, 2400000.5, 54479.0)
    vvd(rh,  1.767794191464423978, 1e-12, "fk5hz", "ra", status)
    vvd(dh, -0.2917516001679884419, 1e-12, "fk5hz", "dec", status)


def test_fw2m():
    r = np.empty((3, 3), float)
    gamb = -0.2243387670997992368e-5
    phib = 0.4091014602391312982
    psi  = -0.9501954178013015092e-3
    eps  = 0.4091014316587367472
    r = erfa_ufunc.fw2m(gamb, phib, psi, eps)
    vvd(r[0][0], 0.9999995505176007047, 1e-12, "fw2m", "11", status)
    vvd(r[0][1], 0.8695404617348192957e-3, 1e-12, "fw2m", "12", status)
    vvd(r[0][2], 0.3779735201865582571e-3, 1e-12, "fw2m", "13", status)
    vvd(r[1][0], -0.8695404723772016038e-3, 1e-12, "fw2m", "21", status)
    vvd(r[1][1], 0.9999996219496027161, 1e-12, "fw2m", "22", status)
    vvd(r[1][2], -0.1361752496887100026e-6, 1e-12, "fw2m", "23", status)
    vvd(r[2][0], -0.3779734957034082790e-3, 1e-12, "fw2m", "31", status)
    vvd(r[2][1], -0.1924880848087615651e-6, 1e-12, "fw2m", "32", status)
    vvd(r[2][2], 0.9999999285679971958, 1e-12, "fw2m", "33", status)


def test_fw2xy():
    gamb = -0.2243387670997992368e-5
    phib = 0.4091014602391312982
    psi  = -0.9501954178013015092e-3
    eps  = 0.4091014316587367472
    x, y = erfa_ufunc.fw2xy(gamb, phib, psi, eps)
    vvd(x, -0.3779734957034082790e-3, 1e-14, "fw2xy", "x", status)
    vvd(y, -0.1924880848087615651e-6, 1e-14, "fw2xy", "y", status)


def test_g2icrs():
    dl = 5.5850536063818546461558105
    db = -0.7853981633974483096156608
    dr, dd = erfa_ufunc.g2icrs (dl, db)
    vvd(dr,  5.9338074302227188048671, 1e-14, "g2icrs", "R", status)
    vvd(dd, -1.1784870613579944551541, 1e-14, "g2icrs", "D", status)


def test_gc2gd():
    xyz = np.array([2e6, 3e6, 5.244e6])
    e, p, h, j = erfa_ufunc.gc2gd(0, xyz)
    viv(j, -1, "gc2gd", "j0", status)
    e, p, h, j = erfa_ufunc.gc2gd(erfa.WGS84, xyz)
    viv(j, 0, "gc2gd", "j1", status)
    vvd(e, 0.9827937232473290680, 1e-14, "gc2gd", "e1", status)
    vvd(p, 0.97160184819075459, 1e-14, "gc2gd", "p1", status)
    vvd(h, 331.4172461426059892, 1e-8, "gc2gd", "h1", status)
    e, p, h, j = erfa_ufunc.gc2gd(erfa.GRS80, xyz)
    viv(j, 0, "gc2gd", "j2", status)
    vvd(e, 0.9827937232473290680, 1e-14, "gc2gd", "e2", status)
    vvd(p, 0.97160184820607853, 1e-14, "gc2gd", "p2", status)
    vvd(h, 331.41731754844348, 1e-8, "gc2gd", "h2", status)
    e, p, h, j = erfa_ufunc.gc2gd(erfa.WGS72, xyz)
    viv(j, 0, "gc2gd", "j3", status)
    vvd(e, 0.9827937232473290680, 1e-14, "gc2gd", "e3", status)
    vvd(p, 0.9716018181101511937, 1e-14, "gc2gd", "p3", status)
    vvd(h, 333.2770726130318123, 1e-8, "gc2gd", "h3", status)
    e, p, h, j = erfa_ufunc.gc2gd(4, xyz)
    viv(j, -1, "gc2gd", "j4", status)


def test_gc2gde():
    a = 6378136.0
    f = 0.0033528
    xyz = np.array([2e6, 3e6, 5.244e6])
    e, p, h, j = erfa_ufunc.gc2gde(a, f, xyz)
    viv(j, 0, "gc2gde", "j", status)
    vvd(e, 0.9827937232473290680, 1e-14, "gc2gde", "e", status)
    vvd(p, 0.9716018377570411532, 1e-14, "gc2gde", "p", status)
    vvd(h, 332.36862495764397, 1e-8, "gc2gde", "h", status)


def test_gd2gc():
    e = 3.1
    p = -0.5
    h = 2500.0
    xyz = np.empty((3,), float)
    xyz, j = erfa_ufunc.gd2gc(0, e, p, h)
    viv(j, -1, "gd2gc", "j0", status)
    xyz, j = erfa_ufunc.gd2gc(erfa.WGS84, e, p, h)
    viv(j, 0, "gd2gc", "j1", status)
    vvd(xyz[0], -5599000.5577049947, 1e-7, "gd2gc", "1/1", status)
    vvd(xyz[1], 233011.67223479203, 1e-7, "gd2gc", "2/1", status)
    vvd(xyz[2], -3040909.4706983363, 1e-7, "gd2gc", "3/1", status)
    xyz, j = erfa_ufunc.gd2gc(erfa.GRS80, e, p, h)
    viv(j, 0, "gd2gc", "j2", status)
    vvd(xyz[0], -5599000.5577260984, 1e-7, "gd2gc", "1/2", status)
    vvd(xyz[1], 233011.6722356702949, 1e-7, "gd2gc", "2/2", status)
    vvd(xyz[2], -3040909.4706095476, 1e-7, "gd2gc", "3/2", status)
    xyz, j = erfa_ufunc.gd2gc(erfa.WGS72, e, p, h)
    viv(j, 0, "gd2gc", "j3", status)
    vvd(xyz[0], -5598998.7626301490, 1e-7, "gd2gc", "1/3", status)
    vvd(xyz[1], 233011.5975297822211, 1e-7, "gd2gc", "2/3", status)
    vvd(xyz[2], -3040908.6861467111, 1e-7, "gd2gc", "3/3", status)
    xyz, j = erfa_ufunc.gd2gc(4, e, p, h)
    viv(j, -1, "gd2gc", "j4", status)


def test_gd2gce():
    a = 6378136.0
    f = 0.0033528
    e = 3.1
    p = -0.5
    h = 2500.0
    xyz = np.empty((3,), float)
    xyz, j = erfa_ufunc.gd2gce(a, f, e, p, h)
    viv(j, 0, "gd2gce", "j", status)
    vvd(xyz[0], -5598999.6665116328, 1e-7, "gd2gce", "1", status)
    vvd(xyz[1], 233011.6351463057189, 1e-7, "gd2gce", "2", status)
    vvd(xyz[2], -3040909.0517314132, 1e-7, "gd2gce", "3", status)


def test_gmst00():
    theta = erfa_ufunc.gmst00(2400000.5, 53736.0, 2400000.5, 53736.0)
    vvd(theta, 1.754174972210740592, 1e-12, "gmst00", "", status)


def test_gmst06():
    theta = erfa_ufunc.gmst06(2400000.5, 53736.0, 2400000.5, 53736.0)
    vvd(theta, 1.754174971870091203, 1e-12, "gmst06", "", status)


def test_gmst82():
    theta = erfa_ufunc.gmst82(2400000.5, 53736.0)
    vvd(theta, 1.754174981860675096, 1e-12, "gmst82", "", status)


def test_gst00a():
    theta = erfa_ufunc.gst00a(2400000.5, 53736.0, 2400000.5, 53736.0)
    vvd(theta, 1.754166138018281369, 1e-12, "gst00a", "", status)


def test_gst00b():
    theta = erfa_ufunc.gst00b(2400000.5, 53736.0)
    vvd(theta, 1.754166136510680589, 1e-12, "gst00b", "", status)


def test_gst06():
    rnpb = np.empty((3, 3), float)
    rnpb[0][0] = 0.9999989440476103608
    rnpb[0][1] = -0.1332881761240011518e-2
    rnpb[0][2] = -0.5790767434730085097e-3
    rnpb[1][0] = 0.1332858254308954453e-2
    rnpb[1][1] = 0.9999991109044505944
    rnpb[1][2] = -0.4097782710401555759e-4
    rnpb[2][0] = 0.5791308472168153320e-3
    rnpb[2][1] = 0.4020595661593994396e-4
    rnpb[2][2] = 0.9999998314954572365
    theta = erfa_ufunc.gst06(2400000.5, 53736.0, 2400000.5, 53736.0, rnpb)
    vvd(theta, 1.754166138018167568, 1e-12, "gst06", "", status)


def test_gst06a():
    theta = erfa_ufunc.gst06a(2400000.5, 53736.0, 2400000.5, 53736.0)
    vvd(theta, 1.754166137675019159, 1e-12, "gst06a", "", status)


def test_gst94():
    theta = erfa_ufunc.gst94(2400000.5, 53736.0)
    vvd(theta, 1.754166136020645203, 1e-12, "gst94", "", status)


def test_h2fk5():
    rh  = 1.767794352
    dh  = -0.2917512594
    drh = -2.76413026e-6
    ddh = -5.92994449e-6
    pxh = 0.379210
    rvh = -7.6
    r5, d5, dr5, dd5, px5, rv5 = erfa_ufunc.h2fk5(rh, dh, drh, ddh, pxh, rvh)
    vvd(r5, 1.767794455700065506, 1e-13, "h2fk5", "ra", status)
    vvd(d5, -0.2917513626469638890, 1e-13, "h2fk5", "dec", status)
    vvd(dr5, -0.27597945024511204e-5, 1e-18, "h2fk5", "dr5", status)
    vvd(dd5, -0.59308014093262838e-5, 1e-18, "h2fk5", "dd5", status)
    vvd(px5, 0.37921, 1e-13, "h2fk5", "px", status)
    vvd(rv5, -7.6000001309071126, 1e-11, "h2fk5", "rv", status)


def test_hd2ae():
    h = 1.1
    d = 1.2
    p = 0.3
    a, e = erfa_ufunc.hd2ae(h, d, p)
    vvd(a, 5.916889243730066194, 1e-13, "hd2ae", "a", status)
    vvd(e, 0.4472186304990486228, 1e-14, "hd2ae", "e", status)


def test_hd2pa():
    h = 1.1
    d = 1.2
    p = 0.3
    q = erfa_ufunc.hd2pa(h, d, p)
    vvd(q, 1.906227428001995580, 1e-13, "hd2pa", "q", status)


def test_hfk5z():
    rh = 1.767794352
    dh = -0.2917512594
    r5, d5, dr5, dd5 = erfa_ufunc.hfk5z(rh, dh, 2400000.5, 54479.0)
    vvd(r5, 1.767794490535581026, 1e-13, "hfk5z", "ra", status)
    vvd(d5, -0.2917513695320114258, 1e-14, "hfk5z", "dec", status)
    vvd(dr5, 0.4335890983539243029e-8, 1e-22, "hfk5z", "dr5", status)
    vvd(dd5, -0.8569648841237745902e-9, 1e-23, "hfk5z", "dd5", status)


def test_icrs2g():
    dr = 5.9338074302227188048671087
    dd = -1.1784870613579944551540570
    dl, db = erfa_ufunc.icrs2g (dr, dd)
    vvd(dl,  5.5850536063818546461558, 1e-14, "icrs2g", "L", status)
    vvd(db, -0.7853981633974483096157, 1e-14, "icrs2g", "B", status)


@pytest.mark.xfail(reason='do not yet support no-input ufuncs')
def test_ir():
    r = np.empty((3, 3), float)
    r[0][0] = 2.0
    r[0][1] = 3.0
    r[0][2] = 2.0
    r[1][0] = 3.0
    r[1][1] = 2.0
    r[1][2] = 3.0
    r[2][0] = 3.0
    r[2][1] = 4.0
    r[2][2] = 5.0
    r = erfa_ufunc.ir()
    vvd(r[0][0], 1.0, 0.0, "ir", "11", status)
    vvd(r[0][1], 0.0, 0.0, "ir", "12", status)
    vvd(r[0][2], 0.0, 0.0, "ir", "13", status)
    vvd(r[1][0], 0.0, 0.0, "ir", "21", status)
    vvd(r[1][1], 1.0, 0.0, "ir", "22", status)
    vvd(r[1][2], 0.0, 0.0, "ir", "23", status)
    vvd(r[2][0], 0.0, 0.0, "ir", "31", status)
    vvd(r[2][1], 0.0, 0.0, "ir", "32", status)
    vvd(r[2][2], 1.0, 0.0, "ir", "33", status)


def test_jd2cal():
    dj1 = 2400000.5
    dj2 = 50123.9999
    iy, im, id, fd, j = erfa_ufunc.jd2cal(dj1, dj2)
    viv(iy, 1996, "jd2cal", "y", status)
    viv(im, 2, "jd2cal", "m", status)
    viv(id, 10, "jd2cal", "d", status)
    vvd(fd, 0.9999, 1e-7, "jd2cal", "fd", status)
    viv(j, 0, "jd2cal", "j", status)


def test_jdcalf():
    dj1 = 2400000.5
    dj2 = 50123.9999
    iydmf, j = erfa_ufunc.jdcalf(4, dj1, dj2)
    viv(iydmf[0], 1996, "jdcalf", "y", status)
    viv(iydmf[1], 2, "jdcalf", "m", status)
    viv(iydmf[2], 10, "jdcalf", "d", status)
    viv(iydmf[3], 9999, "jdcalf", "f", status)
    viv(j, 0, "jdcalf", "j", status)


def test_ld():
    p = np.empty((3,), float)
    q = np.empty((3,), float)
    e = np.empty((3,), float)
    p1 = np.empty((3,), float)
    bm = 0.00028574
    p[0] = -0.763276255
    p[1] = -0.608633767
    p[2] = -0.216735543
    q[0] = -0.763276255
    q[1] = -0.608633767
    q[2] = -0.216735543
    e[0] = 0.76700421
    e[1] = 0.605629598
    e[2] = 0.211937094
    em = 8.91276983
    dlim = 3e-10
    p1 = erfa_ufunc.ld(bm, p, q, e, em, dlim)
    vvd(p1[0], -0.7632762548968159627, 1e-12, "ld", "1", status)
    vvd(p1[1], -0.6086337670823762701, 1e-12, "ld", "2", status)
    vvd(p1[2], -0.2167355431320546947, 1e-12, "ld", "3", status)


def test_ldn():
    b = np.empty(3, erfa_ufunc.dt_eraLDBODY).view(np.recarray)
    ob = np.empty((3,), float)
    sc = np.empty((3,), float)
    sn = np.empty((3,), float)
    b[0].bm = 0.00028574
    b[0].dl = 3e-10
    b[0].pv['p'][0] = -7.81014427
    b[0].pv['p'][1] = -5.60956681
    b[0].pv['p'][2] = -1.98079819
    b[0].pv['v'][0] = 0.0030723249
    b[0].pv['v'][1] = -0.00406995477
    b[0].pv['v'][2] = -0.00181335842
    b[1].bm = 0.00095435
    b[1].dl = 3e-9
    b[1].pv['p'][0] = 0.738098796
    b[1].pv['p'][1] = 4.63658692
    b[1].pv['p'][2] = 1.9693136
    b[1].pv['v'][0] = -0.00755816922
    b[1].pv['v'][1] = 0.00126913722
    b[1].pv['v'][2] = 0.000727999001
    b[2].bm = 1.0
    b[2].dl = 6e-6
    b[2].pv['p'][0] = -0.000712174377
    b[2].pv['p'][1] = -0.00230478303
    b[2].pv['p'][2] = -0.00105865966
    b[2].pv['v'][0] = 6.29235213e-6
    b[2].pv['v'][1] = -3.30888387e-7
    b[2].pv['v'][2] = -2.96486623e-7
    ob[0] = -0.974170437
    ob[1] = -0.2115201
    ob[2] = -0.0917583114
    sc[0] = -0.763276255
    sc[1] = -0.608633767
    sc[2] = -0.216735543
    sn = erfa_ufunc.ldn(b, ob, sc)
    vvd(sn[0], -0.7632762579693333866, 1e-12, "ldn", "1", status)
    vvd(sn[1], -0.6086337636093002660, 1e-12, "ldn", "2", status)
    vvd(sn[2], -0.2167355420646328159, 1e-12, "ldn", "3", status)


def test_ldsun():
    p = np.empty((3,), float)
    e = np.empty((3,), float)
    p1 = np.empty((3,), float)
    p[0] = -0.763276255
    p[1] = -0.608633767
    p[2] = -0.216735543
    e[0] = -0.973644023
    e[1] = -0.20925523
    e[2] = -0.0907169552
    em = 0.999809214
    p1 = erfa_ufunc.ldsun(p, e, em)
    vvd(p1[0], -0.7632762580731413169, 1e-12, "ldsun", "1", status)
    vvd(p1[1], -0.6086337635262647900, 1e-12, "ldsun", "2", status)
    vvd(p1[2], -0.2167355419322321302, 1e-12, "ldsun", "3", status)


def test_lteceq():
    epj = 2500.0
    dl = 1.5
    db = 0.6
    dr, dd = erfa_ufunc.lteceq(epj, dl, db)
    vvd(dr, 1.275156021861921167, 1e-14, "lteceq", "dr", status)
    vvd(dd, 0.9966573543519204791, 1e-14, "lteceq", "dd", status)


def test_ltecm():
    rm = np.empty((3, 3), float)
    epj = -3000.0
    rm = erfa_ufunc.ltecm(epj)
    vvd(rm[0][0], 0.3564105644859788825, 1e-14, "ltecm", "rm11", status)
    vvd(rm[0][1], 0.8530575738617682284, 1e-14, "ltecm", "rm12", status)
    vvd(rm[0][2], 0.3811355207795060435, 1e-14, "ltecm", "rm13", status)
    vvd(rm[1][0], -0.9343283469640709942, 1e-14, "ltecm", "rm21", status)
    vvd(rm[1][1], 0.3247830597681745976, 1e-14, "ltecm", "rm22", status)
    vvd(rm[1][2], 0.1467872751535940865, 1e-14, "ltecm", "rm23", status)
    vvd(rm[2][0], 0.1431636191201167793e-2, 1e-14, "ltecm", "rm31", status)
    vvd(rm[2][1], -0.4084222566960599342, 1e-14, "ltecm", "rm32", status)
    vvd(rm[2][2], 0.9127919865189030899, 1e-14, "ltecm", "rm33", status)


def test_lteqec():
    epj = -1500.0
    dr = 1.234
    dd = 0.987
    dl, db = erfa_ufunc.lteqec(epj, dr, dd)
    vvd(dl, 0.5039483649047114859, 1e-14, "lteqec", "dl", status)
    vvd(db, 0.5848534459726224882, 1e-14, "lteqec", "db", status)


def test_ltp():
    rp = np.empty((3, 3), float)
    epj = 1666.666
    rp = erfa_ufunc.ltp(epj)
    vvd(rp[0][0], 0.9967044141159213819, 1e-14, "ltp", "rp11", status)
    vvd(rp[0][1], 0.7437801893193210840e-1, 1e-14, "ltp", "rp12", status)
    vvd(rp[0][2], 0.3237624409345603401e-1, 1e-14, "ltp", "rp13", status)
    vvd(rp[1][0], -0.7437802731819618167e-1, 1e-14, "ltp", "rp21", status)
    vvd(rp[1][1], 0.9972293894454533070, 1e-14, "ltp", "rp22", status)
    vvd(rp[1][2], -0.1205768842723593346e-2, 1e-14, "ltp", "rp23", status)
    vvd(rp[2][0], -0.3237622482766575399e-1, 1e-14, "ltp", "rp31", status)
    vvd(rp[2][1], -0.1206286039697609008e-2, 1e-14, "ltp", "rp32", status)
    vvd(rp[2][2], 0.9994750246704010914, 1e-14, "ltp", "rp33", status)


def test_ltpb():
    rpb = np.empty((3, 3), float)
    epj = 1666.666
    rpb = erfa_ufunc.ltpb(epj)
    vvd(rpb[0][0], 0.9967044167723271851, 1e-14, "ltpb", "rpb11", status)
    vvd(rpb[0][1], 0.7437794731203340345e-1, 1e-14, "ltpb", "rpb12", status)
    vvd(rpb[0][2], 0.3237632684841625547e-1, 1e-14, "ltpb", "rpb13", status)
    vvd(rpb[1][0], -0.7437795663437177152e-1, 1e-14, "ltpb", "rpb21", status)
    vvd(rpb[1][1], 0.9972293947500013666, 1e-14, "ltpb", "rpb22", status)
    vvd(rpb[1][2], -0.1205741865911243235e-2, 1e-14, "ltpb", "rpb23", status)
    vvd(rpb[2][0], -0.3237630543224664992e-1, 1e-14, "ltpb", "rpb31", status)
    vvd(rpb[2][1], -0.1206316791076485295e-2, 1e-14, "ltpb", "rpb32", status)
    vvd(rpb[2][2], 0.9994750220222438819, 1e-14, "ltpb", "rpb33", status)


def test_ltpecl():
    vec = np.empty((3,), float)
    epj = -1500.0
    vec = erfa_ufunc.ltpecl(epj)
    vvd(vec[0], 0.4768625676477096525e-3, 1e-14, "ltpecl", "vec1", status)
    vvd(vec[1], -0.4052259533091875112, 1e-14, "ltpecl", "vec2", status)
    vvd(vec[2], 0.9142164401096448012, 1e-14, "ltpecl", "vec3", status)


def test_ltpequ():
    veq = np.empty((3,), float)
    epj = -2500.0
    veq = erfa_ufunc.ltpequ(epj)
    vvd(veq[0], -0.3586652560237326659, 1e-14, "ltpequ", "veq1", status)
    vvd(veq[1], -0.1996978910771128475, 1e-14, "ltpequ", "veq2", status)
    vvd(veq[2], 0.9118552442250819624, 1e-14, "ltpequ", "veq3", status)


def test_moon98():
    pv = np.empty((), erfa_ufunc.dt_pv)
    pv = erfa_ufunc.moon98(2400000.5, 43999.9)
    vvd(pv['p'][0], -0.2601295959971044180e-2, 1e-11, "moon98", "x 4", status)
    vvd(pv['p'][1], 0.6139750944302742189e-3, 1e-11, "moon98", "y 4", status)
    vvd(pv['p'][2], 0.2640794528229828909e-3, 1e-11, "moon98", "z 4", status)
    vvd(pv['v'][0], -0.1244321506649895021e-3, 1e-11, "moon98", "xd 4", status)
    vvd(pv['v'][1], -0.5219076942678119398e-3, 1e-11, "moon98", "yd 4", status)
    vvd(pv['v'][2], -0.1716132214378462047e-3, 1e-11, "moon98", "zd 4", status)


def test_num00a():
    rmatn = np.empty((3, 3), float)
    rmatn = erfa_ufunc.num00a(2400000.5, 53736.0)
    vvd(rmatn[0][0], 0.9999999999536227949, 1e-12, "num00a", "11", status)
    vvd(rmatn[0][1], 0.8836238544090873336e-5, 1e-12, "num00a", "12", status)
    vvd(rmatn[0][2], 0.3830835237722400669e-5, 1e-12, "num00a", "13", status)
    vvd(rmatn[1][0], -0.8836082880798569274e-5, 1e-12, "num00a", "21", status)
    vvd(rmatn[1][1], 0.9999999991354655028, 1e-12, "num00a", "22", status)
    vvd(rmatn[1][2], -0.4063240865362499850e-4, 1e-12, "num00a", "23", status)
    vvd(rmatn[2][0], -0.3831194272065995866e-5, 1e-12, "num00a", "31", status)
    vvd(rmatn[2][1], 0.4063237480216291775e-4, 1e-12, "num00a", "32", status)
    vvd(rmatn[2][2], 0.9999999991671660338, 1e-12, "num00a", "33", status)


def test_num00b():
    rmatn = np.empty((3, 3), float)
    rmatn = erfa_ufunc.num00b(2400000.5, 53736)
    vvd(rmatn[0][0], 0.9999999999536069682, 1e-12, "num00b", "11", status)
    vvd(rmatn[0][1], 0.8837746144871248011e-5, 1e-12, "num00b", "12", status)
    vvd(rmatn[0][2], 0.3831488838252202945e-5, 1e-12, "num00b", "13", status)
    vvd(rmatn[1][0], -0.8837590456632304720e-5, 1e-12, "num00b", "21", status)
    vvd(rmatn[1][1], 0.9999999991354692733, 1e-12, "num00b", "22", status)
    vvd(rmatn[1][2], -0.4063198798559591654e-4, 1e-12, "num00b", "23", status)
    vvd(rmatn[2][0], -0.3831847930134941271e-5, 1e-12, "num00b", "31", status)
    vvd(rmatn[2][1], 0.4063195412258168380e-4, 1e-12, "num00b", "32", status)
    vvd(rmatn[2][2], 0.9999999991671806225, 1e-12, "num00b", "33", status)


def test_num06a():
    rmatn = np.empty((3, 3), float)
    rmatn = erfa_ufunc.num06a(2400000.5, 53736)
    vvd(rmatn[0][0], 0.9999999999536227668, 1e-12, "num06a", "11", status)
    vvd(rmatn[0][1], 0.8836241998111535233e-5, 1e-12, "num06a", "12", status)
    vvd(rmatn[0][2], 0.3830834608415287707e-5, 1e-12, "num06a", "13", status)
    vvd(rmatn[1][0], -0.8836086334870740138e-5, 1e-12, "num06a", "21", status)
    vvd(rmatn[1][1], 0.9999999991354657474, 1e-12, "num06a", "22", status)
    vvd(rmatn[1][2], -0.4063240188248455065e-4, 1e-12, "num06a", "23", status)
    vvd(rmatn[2][0], -0.3831193642839398128e-5, 1e-12, "num06a", "31", status)
    vvd(rmatn[2][1], 0.4063236803101479770e-4, 1e-12, "num06a", "32", status)
    vvd(rmatn[2][2], 0.9999999991671663114, 1e-12, "num06a", "33", status)


def test_numat():
    rmatn = np.empty((3, 3), float)
    epsa = 0.4090789763356509900
    dpsi = -0.9630909107115582393e-5
    deps = 0.4063239174001678826e-4
    rmatn = erfa_ufunc.numat(epsa, dpsi, deps)
    vvd(rmatn[0][0], 0.9999999999536227949, 1e-12, "numat", "11", status)
    vvd(rmatn[0][1], 0.8836239320236250577e-5, 1e-12, "numat", "12", status)
    vvd(rmatn[0][2], 0.3830833447458251908e-5, 1e-12, "numat", "13", status)
    vvd(rmatn[1][0], -0.8836083657016688588e-5, 1e-12, "numat", "21", status)
    vvd(rmatn[1][1], 0.9999999991354654959, 1e-12, "numat", "22", status)
    vvd(rmatn[1][2], -0.4063240865361857698e-4, 1e-12, "numat", "23", status)
    vvd(rmatn[2][0], -0.3831192481833385226e-5, 1e-12, "numat", "31", status)
    vvd(rmatn[2][1], 0.4063237480216934159e-4, 1e-12, "numat", "32", status)
    vvd(rmatn[2][2], 0.9999999991671660407, 1e-12, "numat", "33", status)


def test_nut00a():
    dpsi, deps = erfa_ufunc.nut00a(2400000.5, 53736.0)
    vvd(dpsi, -0.9630909107115518431e-5, 1e-13, "nut00a", "dpsi", status)
    vvd(deps,  0.4063239174001678710e-4, 1e-13, "nut00a", "deps", status)


def test_nut00b():
    dpsi, deps = erfa_ufunc.nut00b(2400000.5, 53736.0)
    vvd(dpsi, -0.9632552291148362783e-5, 1e-13, "nut00b", "dpsi", status)
    vvd(deps,  0.4063197106621159367e-4, 1e-13, "nut00b", "deps", status)


def test_nut06a():
    dpsi, deps = erfa_ufunc.nut06a(2400000.5, 53736.0)
    vvd(dpsi, -0.9630912025820308797e-5, 1e-13, "nut06a", "dpsi", status)
    vvd(deps,  0.4063238496887249798e-4, 1e-13, "nut06a", "deps", status)


def test_nut80():
    dpsi, deps = erfa_ufunc.nut80(2400000.5, 53736.0)
    vvd(dpsi, -0.9643658353226563966e-5, 1e-13, "nut80", "dpsi", status)
    vvd(deps,  0.4060051006879713322e-4, 1e-13, "nut80", "deps", status)


def test_nutm80():
    rmatn = np.empty((3, 3), float)
    rmatn = erfa_ufunc.nutm80(2400000.5, 53736.0)
    vvd(rmatn[0][0], 0.9999999999534999268, 1e-12, "nutm80", "11", status)
    vvd(rmatn[0][1], 0.8847935789636432161e-5, 1e-12, "nutm80", "12", status)
    vvd(rmatn[0][2], 0.3835906502164019142e-5, 1e-12, "nutm80", "13", status)
    vvd(rmatn[1][0], -0.8847780042583435924e-5, 1e-12, "nutm80", "21", status)
    vvd(rmatn[1][1], 0.9999999991366569963, 1e-12, "nutm80", "22", status)
    vvd(rmatn[1][2], -0.4060052702727130809e-4, 1e-12, "nutm80", "23", status)
    vvd(rmatn[2][0], -0.3836265729708478796e-5, 1e-12, "nutm80", "31", status)
    vvd(rmatn[2][1], 0.4060049308612638555e-4, 1e-12, "nutm80", "32", status)
    vvd(rmatn[2][2], 0.9999999991684415129, 1e-12, "nutm80", "33", status)


def test_obl06():
    vvd(erfa_ufunc.obl06(2400000.5, 54388.0), 0.4090749229387258204, 1e-14, "obl06", "", status)


def test_obl80():
    eps0 = erfa_ufunc.obl80(2400000.5, 54388.0)
    vvd(eps0, 0.4090751347643816218, 1e-14, "obl80", "", status)


def test_p06e():
    eps0, psia, oma, bpa, bqa, pia, bpia, epsa, chia, za, zetaa, thetaa, pa, gam, phi, psi = erfa_ufunc.p06e(2400000.5, 52541.0)
    vvd(eps0, 0.4090926006005828715, 1e-14, "p06e", "eps0", status)
    vvd(psia, 0.6664369630191613431e-3, 1e-14, "p06e", "psia", status)
    vvd(oma , 0.4090925973783255982, 1e-14, "p06e", "oma", status)
    vvd(bpa, 0.5561149371265209445e-6, 1e-14, "p06e", "bpa", status)
    vvd(bqa, -0.6191517193290621270e-5, 1e-14, "p06e", "bqa", status)
    vvd(pia, 0.6216441751884382923e-5, 1e-14, "p06e", "pia", status)
    vvd(bpia, 3.052014180023779882, 1e-14, "p06e", "bpia", status)
    vvd(epsa, 0.4090864054922431688, 1e-14, "p06e", "epsa", status)
    vvd(chia, 0.1387703379530915364e-5, 1e-14, "p06e", "chia", status)
    vvd(za, 0.2921789846651790546e-3, 1e-14, "p06e", "za", status)
    vvd(zetaa, 0.3178773290332009310e-3, 1e-14, "p06e", "zetaa", status)
    vvd(thetaa, 0.2650932701657497181e-3, 1e-14, "p06e", "thetaa", status)
    vvd(pa, 0.6651637681381016288e-3, 1e-14, "p06e", "pa", status)
    vvd(gam, 0.1398077115963754987e-5, 1e-14, "p06e", "gam", status)
    vvd(phi, 0.4090864090837462602, 1e-14, "p06e", "phi", status)
    vvd(psi, 0.6664464807480920325e-3, 1e-14, "p06e", "psi", status)


def test_p2pv():
    p = np.empty((3,), float)
    pv = np.empty((), erfa_ufunc.dt_pv)
    p[0] = 0.25
    p[1] = 1.2
    p[2] = 3.0
    pv['p'][0] = 0.3
    pv['p'][1] = 1.2
    pv['p'][2] = -2.5
    pv['v'][0] = -0.5
    pv['v'][1] = 3.1
    pv['v'][2] = 0.9
    pv = erfa_ufunc.p2pv(p)
    vvd(pv['p'][0], 0.25, 0.0, "p2pv", "p1", status)
    vvd(pv['p'][1], 1.2,  0.0, "p2pv", "p2", status)
    vvd(pv['p'][2], 3.0,  0.0, "p2pv", "p3", status)
    vvd(pv['v'][0], 0.0,  0.0, "p2pv", "v1", status)
    vvd(pv['v'][1], 0.0,  0.0, "p2pv", "v2", status)
    vvd(pv['v'][2], 0.0,  0.0, "p2pv", "v3", status)


def test_p2s():
    p = np.empty((3,), float)
    p[0] = 100.0
    p[1] = -50.0
    p[2] = 25.0
    theta, phi, r = erfa_ufunc.p2s(p)
    vvd(theta, -0.4636476090008061162, 1e-12, "p2s", "theta", status)
    vvd(phi, 0.2199879773954594463, 1e-12, "p2s", "phi", status)
    vvd(r, 114.5643923738960002, 1e-9, "p2s", "r", status)


def test_pap():
    a = np.empty((3,), float)
    b = np.empty((3,), float)
    a[0] = 1.0
    a[1] = 0.1
    a[2] = 0.2
    b[0] = -3.0
    b[1] = 1e-3
    b[2] = 0.2
    theta = erfa_ufunc.pap(a, b)
    vvd(theta, 0.3671514267841113674, 1e-12, "pap", "", status)


def test_pas():
    al = 1.0
    ap = 0.1
    bl = 0.2
    bp = -1.0
    theta = erfa_ufunc.pas(al, ap, bl, bp)
    vvd(theta, -2.724544922932270424, 1e-12, "pas", "", status)


def test_pb06():
    bzeta, bz, btheta = erfa_ufunc.pb06(2400000.5, 50123.9999)
    vvd(bzeta, -0.5092634016326478238e-3, 1e-12, "pb06", "bzeta", status)
    vvd(bz, -0.3602772060566044413e-3, 1e-12, "pb06", "bz", status)
    vvd(btheta, -0.3779735537167811177e-3, 1e-12, "pb06", "btheta", status)


def test_pdp():
    a = np.empty((3,), float)
    b = np.empty((3,), float)
    a[0] = 2.0
    a[1] = 2.0
    a[2] = 3.0
    b[0] = 1.0
    b[1] = 3.0
    b[2] = 4.0
    adb = erfa_ufunc.pdp(a, b)
    vvd(adb, 20, 1e-12, "pdp", "", status)


def test_pfw06():
    gamb, phib, psib, epsa = erfa_ufunc.pfw06(2400000.5, 50123.9999)
    vvd(gamb, -0.2243387670997995690e-5, 1e-16, "pfw06", "gamb", status)
    vvd(phib,  0.4091014602391312808, 1e-12, "pfw06", "phib", status)
    vvd(psib, -0.9501954178013031895e-3, 1e-14, "pfw06", "psib", status)
    vvd(epsa,  0.4091014316587367491, 1e-12, "pfw06", "epsa", status)


def test_plan94():
    pv = np.empty((), erfa_ufunc.dt_pv)
    pv, j = erfa_ufunc.plan94(2400000.5, 1e6, 0)
    vvd(pv['p'][0], 0.0, 0.0, "plan94", "x 1", status)
    vvd(pv['p'][1], 0.0, 0.0, "plan94", "y 1", status)
    vvd(pv['p'][2], 0.0, 0.0, "plan94", "z 1", status)
    vvd(pv['v'][0], 0.0, 0.0, "plan94", "xd 1", status)
    vvd(pv['v'][1], 0.0, 0.0, "plan94", "yd 1", status)
    vvd(pv['v'][2], 0.0, 0.0, "plan94", "zd 1", status)
    viv(j, -1, "plan94", "j 1", status)
    pv, j = erfa_ufunc.plan94(2400000.5, 1e6, 10)
    viv(j, -1, "plan94", "j 2", status)
    pv, j = erfa_ufunc.plan94(2400000.5, -320000, 3)
    vvd(pv['p'][0], 0.9308038666832975759, 1e-11, "plan94", "x 3", status)
    vvd(pv['p'][1], 0.3258319040261346000, 1e-11, "plan94", "y 3", status)
    vvd(pv['p'][2], 0.1422794544481140560, 1e-11, "plan94", "z 3", status)
    vvd(pv['v'][0], -0.6429458958255170006e-2, 1e-11, "plan94", "xd 3", status)
    vvd(pv['v'][1], 0.1468570657704237764e-1, 1e-11, "plan94", "yd 3", status)
    vvd(pv['v'][2], 0.6406996426270981189e-2, 1e-11, "plan94", "zd 3", status)
    viv(j, 1, "plan94", "j 3", status)
    pv, j = erfa_ufunc.plan94(2400000.5, 43999.9, 1)
    vvd(pv['p'][0], 0.2945293959257430832, 1e-11, "plan94", "x 4", status)
    vvd(pv['p'][1], -0.2452204176601049596, 1e-11, "plan94", "y 4", status)
    vvd(pv['p'][2], -0.1615427700571978153, 1e-11, "plan94", "z 4", status)
    vvd(pv['v'][0], 0.1413867871404614441e-1, 1e-11, "plan94", "xd 4", status)
    vvd(pv['v'][1], 0.1946548301104706582e-1, 1e-11, "plan94", "yd 4", status)
    vvd(pv['v'][2], 0.8929809783898904786e-2, 1e-11, "plan94", "zd 4", status)
    viv(j, 0, "plan94", "j 4", status)


def test_pm():
    p = np.empty((3,), float)
    p[0] = 0.3
    p[1] = 1.2
    p[2] = -2.5
    r = erfa_ufunc.pm(p)
    vvd(r, 2.789265136196270604, 1e-12, "pm", "", status)


def test_pmat00():
    rbp = np.empty((3, 3), float)
    rbp = erfa_ufunc.pmat00(2400000.5, 50123.9999)
    vvd(rbp[0][0], 0.9999995505175087260, 1e-12, "pmat00", "11", status)
    vvd(rbp[0][1], 0.8695405883617884705e-3, 1e-14, "pmat00", "12", status)
    vvd(rbp[0][2], 0.3779734722239007105e-3, 1e-14, "pmat00", "13", status)
    vvd(rbp[1][0], -0.8695405990410863719e-3, 1e-14, "pmat00", "21", status)
    vvd(rbp[1][1], 0.9999996219494925900, 1e-12, "pmat00", "22", status)
    vvd(rbp[1][2], -0.1360775820404982209e-6, 1e-14, "pmat00", "23", status)
    vvd(rbp[2][0], -0.3779734476558184991e-3, 1e-14, "pmat00", "31", status)
    vvd(rbp[2][1], -0.1925857585832024058e-6, 1e-14, "pmat00", "32", status)
    vvd(rbp[2][2], 0.9999999285680153377, 1e-12, "pmat00", "33", status)


def test_pmat06():
    rbp = np.empty((3, 3), float)
    rbp = erfa_ufunc.pmat06(2400000.5, 50123.9999)
    vvd(rbp[0][0], 0.9999995505176007047, 1e-12, "pmat06", "11", status)
    vvd(rbp[0][1], 0.8695404617348208406e-3, 1e-14, "pmat06", "12", status)
    vvd(rbp[0][2], 0.3779735201865589104e-3, 1e-14, "pmat06", "13", status)
    vvd(rbp[1][0], -0.8695404723772031414e-3, 1e-14, "pmat06", "21", status)
    vvd(rbp[1][1], 0.9999996219496027161, 1e-12, "pmat06", "22", status)
    vvd(rbp[1][2], -0.1361752497080270143e-6, 1e-14, "pmat06", "23", status)
    vvd(rbp[2][0], -0.3779734957034089490e-3, 1e-14, "pmat06", "31", status)
    vvd(rbp[2][1], -0.1924880847894457113e-6, 1e-14, "pmat06", "32", status)
    vvd(rbp[2][2], 0.9999999285679971958, 1e-12, "pmat06", "33", status)


def test_pmat76():
    rmatp = np.empty((3, 3), float)
    rmatp = erfa_ufunc.pmat76(2400000.5, 50123.9999)
    vvd(rmatp[0][0], 0.9999995504328350733, 1e-12, "pmat76", "11", status)
    vvd(rmatp[0][1], 0.8696632209480960785e-3, 1e-14, "pmat76", "12", status)
    vvd(rmatp[0][2], 0.3779153474959888345e-3, 1e-14, "pmat76", "13", status)
    vvd(rmatp[1][0], -0.8696632209485112192e-3, 1e-14, "pmat76", "21", status)
    vvd(rmatp[1][1], 0.9999996218428560614, 1e-12, "pmat76", "22", status)
    vvd(rmatp[1][2], -0.1643284776111886407e-6, 1e-14, "pmat76", "23", status)
    vvd(rmatp[2][0], -0.3779153474950335077e-3, 1e-14, "pmat76", "31", status)
    vvd(rmatp[2][1], -0.1643306746147366896e-6, 1e-14, "pmat76", "32", status)
    vvd(rmatp[2][2], 0.9999999285899790119, 1e-12, "pmat76", "33", status)


def test_pmp():
    a = np.empty((3,), float)
    b = np.empty((3,), float)
    amb = np.empty((3,), float)
    a[0] = 2.0
    a[1] = 2.0
    a[2] = 3.0
    b[0] = 1.0
    b[1] = 3.0
    b[2] = 4.0
    amb = erfa_ufunc.pmp(a, b)
    vvd(amb[0],  1.0, 1e-12, "pmp", "0", status)
    vvd(amb[1], -1.0, 1e-12, "pmp", "1", status)
    vvd(amb[2], -1.0, 1e-12, "pmp", "2", status)


def test_pmpx():
    pob = np.empty((3,), float)
    pco = np.empty((3,), float)
    rc = 1.234
    dc = 0.789
    pr = 1e-5
    pd = -2e-5
    px = 1e-2
    rv = 10.0
    pmt = 8.75
    pob[0] = 0.9
    pob[1] = 0.4
    pob[2] = 0.1
    pco = erfa_ufunc.pmpx(rc, dc, pr, pd, px, rv, pmt, pob)
    vvd(pco[0], 0.2328137623960308438, 1e-12, "pmpx", "1", status)
    vvd(pco[1], 0.6651097085397855328, 1e-12, "pmpx", "2", status)
    vvd(pco[2], 0.7095257765896359837, 1e-12, "pmpx", "3", status)


def test_pmsafe():
    ra1 = 1.234
    dec1 = 0.789
    pmr1 = 1e-5
    pmd1 = -2e-5
    px1 = 1e-2
    rv1 = 10.0
    ep1a = 2400000.5
    ep1b = 48348.5625
    ep2a = 2400000.5
    ep2b = 51544.5
    ra2, dec2, pmr2, pmd2, px2, rv2, j = erfa_ufunc.pmsafe(ra1, dec1, pmr1, pmd1, px1, rv1, ep1a, ep1b, ep2a, ep2b)
    vvd(ra2, 1.234087484501017061, 1e-12, "pmsafe", "ra2", status)
    vvd(dec2, 0.7888249982450468567, 1e-12, "pmsafe", "dec2", status)
    vvd(pmr2, 0.9996457663586073988e-5, 1e-12, "pmsafe", "pmr2", status)
    vvd(pmd2, -0.2000040085106754565e-4, 1e-16, "pmsafe", "pmd2", status)
    vvd(px2, 0.9999997295356830666e-2, 1e-12, "pmsafe", "px2", status)
    vvd(rv2, 10.38468380293920069, 1e-10, "pmsafe", "rv2", status)
    viv ( j, 0, "pmsafe", "j", status)


def test_pn():
    p = np.empty((3,), float)
    u = np.empty((3,), float)
    p[0] = 0.3
    p[1] = 1.2
    p[2] = -2.5
    r, u = erfa_ufunc.pn(p)
    vvd(r, 2.789265136196270604, 1e-12, "pn", "r", status)
    vvd(u[0], 0.1075552109073112058, 1e-12, "pn", "u1", status)
    vvd(u[1], 0.4302208436292448232, 1e-12, "pn", "u2", status)
    vvd(u[2], -0.8962934242275933816, 1e-12, "pn", "u3", status)


def test_pn00():
    rb = np.empty((3, 3), float)
    rp = np.empty((3, 3), float)
    rbp = np.empty((3, 3), float)
    rn = np.empty((3, 3), float)
    rbpn = np.empty((3, 3), float)
    dpsi = -0.9632552291149335877e-5
    deps = 0.4063197106621141414e-4
    epsa, rb, rp, rbp, rn, rbpn = erfa_ufunc.pn00(2400000.5, 53736.0, dpsi, deps)
    vvd(epsa, 0.4090791789404229916, 1e-12, "pn00", "epsa", status)
    vvd(rb[0][0], 0.9999999999999942498, 1e-12, "pn00", "rb11", status)
    vvd(rb[0][1], -0.7078279744199196626e-7, 1e-18, "pn00", "rb12", status)
    vvd(rb[0][2], 0.8056217146976134152e-7, 1e-18, "pn00", "rb13", status)
    vvd(rb[1][0], 0.7078279477857337206e-7, 1e-18, "pn00", "rb21", status)
    vvd(rb[1][1], 0.9999999999999969484, 1e-12, "pn00", "rb22", status)
    vvd(rb[1][2], 0.3306041454222136517e-7, 1e-18, "pn00", "rb23", status)
    vvd(rb[2][0], -0.8056217380986972157e-7, 1e-18, "pn00", "rb31", status)
    vvd(rb[2][1], -0.3306040883980552500e-7, 1e-18, "pn00", "rb32", status)
    vvd(rb[2][2], 0.9999999999999962084, 1e-12, "pn00", "rb33", status)
    vvd(rp[0][0], 0.9999989300532289018, 1e-12, "pn00", "rp11", status)
    vvd(rp[0][1], -0.1341647226791824349e-2, 1e-14, "pn00", "rp12", status)
    vvd(rp[0][2], -0.5829880927190296547e-3, 1e-14, "pn00", "rp13", status)
    vvd(rp[1][0], 0.1341647231069759008e-2, 1e-14, "pn00", "rp21", status)
    vvd(rp[1][1], 0.9999990999908750433, 1e-12, "pn00", "rp22", status)
    vvd(rp[1][2], -0.3837444441583715468e-6, 1e-14, "pn00", "rp23", status)
    vvd(rp[2][0], 0.5829880828740957684e-3, 1e-14, "pn00", "rp31", status)
    vvd(rp[2][1], -0.3984203267708834759e-6, 1e-14, "pn00", "rp32", status)
    vvd(rp[2][2], 0.9999998300623538046, 1e-12, "pn00", "rp33", status)
    vvd(rbp[0][0], 0.9999989300052243993, 1e-12, "pn00", "rbp11", status)
    vvd(rbp[0][1], -0.1341717990239703727e-2, 1e-14, "pn00", "rbp12", status)
    vvd(rbp[0][2], -0.5829075749891684053e-3, 1e-14, "pn00", "rbp13", status)
    vvd(rbp[1][0], 0.1341718013831739992e-2, 1e-14, "pn00", "rbp21", status)
    vvd(rbp[1][1], 0.9999990998959191343, 1e-12, "pn00", "rbp22", status)
    vvd(rbp[1][2], -0.3505759733565421170e-6, 1e-14, "pn00", "rbp23", status)
    vvd(rbp[2][0], 0.5829075206857717883e-3, 1e-14, "pn00", "rbp31", status)
    vvd(rbp[2][1], -0.4315219955198608970e-6, 1e-14, "pn00", "rbp32", status)
    vvd(rbp[2][2], 0.9999998301093036269, 1e-12, "pn00", "rbp33", status)
    vvd(rn[0][0], 0.9999999999536069682, 1e-12, "pn00", "rn11", status)
    vvd(rn[0][1], 0.8837746144872140812e-5, 1e-16, "pn00", "rn12", status)
    vvd(rn[0][2], 0.3831488838252590008e-5, 1e-16, "pn00", "rn13", status)
    vvd(rn[1][0], -0.8837590456633197506e-5, 1e-16, "pn00", "rn21", status)
    vvd(rn[1][1], 0.9999999991354692733, 1e-12, "pn00", "rn22", status)
    vvd(rn[1][2], -0.4063198798559573702e-4, 1e-16, "pn00", "rn23", status)
    vvd(rn[2][0], -0.3831847930135328368e-5, 1e-16, "pn00", "rn31", status)
    vvd(rn[2][1], 0.4063195412258150427e-4, 1e-16, "pn00", "rn32", status)
    vvd(rn[2][2], 0.9999999991671806225, 1e-12, "pn00", "rn33", status)
    vvd(rbpn[0][0], 0.9999989440499982806, 1e-12, "pn00", "rbpn11", status)
    vvd(rbpn[0][1], -0.1332880253640848301e-2, 1e-14, "pn00", "rbpn12", status)
    vvd(rbpn[0][2], -0.5790760898731087295e-3, 1e-14, "pn00", "rbpn13", status)
    vvd(rbpn[1][0], 0.1332856746979948745e-2, 1e-14, "pn00", "rbpn21", status)
    vvd(rbpn[1][1], 0.9999991109064768883, 1e-12, "pn00", "rbpn22", status)
    vvd(rbpn[1][2], -0.4097740555723063806e-4, 1e-14, "pn00", "rbpn23", status)
    vvd(rbpn[2][0], 0.5791301929950205000e-3, 1e-14, "pn00", "rbpn31", status)
    vvd(rbpn[2][1], 0.4020553681373702931e-4, 1e-14, "pn00", "rbpn32", status)
    vvd(rbpn[2][2], 0.9999998314958529887, 1e-12, "pn00", "rbpn33", status)


def test_pn00a():
    rb = np.empty((3, 3), float)
    rp = np.empty((3, 3), float)
    rbp = np.empty((3, 3), float)
    rn = np.empty((3, 3), float)
    rbpn = np.empty((3, 3), float)
    dpsi, deps, epsa, rb, rp, rbp, rn, rbpn = erfa_ufunc.pn00a(2400000.5, 53736.0)
    vvd(dpsi, -0.9630909107115518431e-5, 1e-12, "pn00a", "dpsi", status)
    vvd(deps,  0.4063239174001678710e-4, 1e-12, "pn00a", "deps", status)
    vvd(epsa,  0.4090791789404229916, 1e-12, "pn00a", "epsa", status)
    vvd(rb[0][0], 0.9999999999999942498, 1e-12, "pn00a", "rb11", status)
    vvd(rb[0][1], -0.7078279744199196626e-7, 1e-16, "pn00a", "rb12", status)
    vvd(rb[0][2], 0.8056217146976134152e-7, 1e-16, "pn00a", "rb13", status)
    vvd(rb[1][0], 0.7078279477857337206e-7, 1e-16, "pn00a", "rb21", status)
    vvd(rb[1][1], 0.9999999999999969484, 1e-12, "pn00a", "rb22", status)
    vvd(rb[1][2], 0.3306041454222136517e-7, 1e-16, "pn00a", "rb23", status)
    vvd(rb[2][0], -0.8056217380986972157e-7, 1e-16, "pn00a", "rb31", status)
    vvd(rb[2][1], -0.3306040883980552500e-7, 1e-16, "pn00a", "rb32", status)
    vvd(rb[2][2], 0.9999999999999962084, 1e-12, "pn00a", "rb33", status)
    vvd(rp[0][0], 0.9999989300532289018, 1e-12, "pn00a", "rp11", status)
    vvd(rp[0][1], -0.1341647226791824349e-2, 1e-14, "pn00a", "rp12", status)
    vvd(rp[0][2], -0.5829880927190296547e-3, 1e-14, "pn00a", "rp13", status)
    vvd(rp[1][0], 0.1341647231069759008e-2, 1e-14, "pn00a", "rp21", status)
    vvd(rp[1][1], 0.9999990999908750433, 1e-12, "pn00a", "rp22", status)
    vvd(rp[1][2], -0.3837444441583715468e-6, 1e-14, "pn00a", "rp23", status)
    vvd(rp[2][0], 0.5829880828740957684e-3, 1e-14, "pn00a", "rp31", status)
    vvd(rp[2][1], -0.3984203267708834759e-6, 1e-14, "pn00a", "rp32", status)
    vvd(rp[2][2], 0.9999998300623538046, 1e-12, "pn00a", "rp33", status)
    vvd(rbp[0][0], 0.9999989300052243993, 1e-12, "pn00a", "rbp11", status)
    vvd(rbp[0][1], -0.1341717990239703727e-2, 1e-14, "pn00a", "rbp12", status)
    vvd(rbp[0][2], -0.5829075749891684053e-3, 1e-14, "pn00a", "rbp13", status)
    vvd(rbp[1][0], 0.1341718013831739992e-2, 1e-14, "pn00a", "rbp21", status)
    vvd(rbp[1][1], 0.9999990998959191343, 1e-12, "pn00a", "rbp22", status)
    vvd(rbp[1][2], -0.3505759733565421170e-6, 1e-14, "pn00a", "rbp23", status)
    vvd(rbp[2][0], 0.5829075206857717883e-3, 1e-14, "pn00a", "rbp31", status)
    vvd(rbp[2][1], -0.4315219955198608970e-6, 1e-14, "pn00a", "rbp32", status)
    vvd(rbp[2][2], 0.9999998301093036269, 1e-12, "pn00a", "rbp33", status)
    vvd(rn[0][0], 0.9999999999536227949, 1e-12, "pn00a", "rn11", status)
    vvd(rn[0][1], 0.8836238544090873336e-5, 1e-14, "pn00a", "rn12", status)
    vvd(rn[0][2], 0.3830835237722400669e-5, 1e-14, "pn00a", "rn13", status)
    vvd(rn[1][0], -0.8836082880798569274e-5, 1e-14, "pn00a", "rn21", status)
    vvd(rn[1][1], 0.9999999991354655028, 1e-12, "pn00a", "rn22", status)
    vvd(rn[1][2], -0.4063240865362499850e-4, 1e-14, "pn00a", "rn23", status)
    vvd(rn[2][0], -0.3831194272065995866e-5, 1e-14, "pn00a", "rn31", status)
    vvd(rn[2][1], 0.4063237480216291775e-4, 1e-14, "pn00a", "rn32", status)
    vvd(rn[2][2], 0.9999999991671660338, 1e-12, "pn00a", "rn33", status)
    vvd(rbpn[0][0], 0.9999989440476103435, 1e-12, "pn00a", "rbpn11", status)
    vvd(rbpn[0][1], -0.1332881761240011763e-2, 1e-14, "pn00a", "rbpn12", status)
    vvd(rbpn[0][2], -0.5790767434730085751e-3, 1e-14, "pn00a", "rbpn13", status)
    vvd(rbpn[1][0], 0.1332858254308954658e-2, 1e-14, "pn00a", "rbpn21", status)
    vvd(rbpn[1][1], 0.9999991109044505577, 1e-12, "pn00a", "rbpn22", status)
    vvd(rbpn[1][2], -0.4097782710396580452e-4, 1e-14, "pn00a", "rbpn23", status)
    vvd(rbpn[2][0], 0.5791308472168152904e-3, 1e-14, "pn00a", "rbpn31", status)
    vvd(rbpn[2][1], 0.4020595661591500259e-4, 1e-14, "pn00a", "rbpn32", status)
    vvd(rbpn[2][2], 0.9999998314954572304, 1e-12, "pn00a", "rbpn33", status)


def test_pn00b():
    rb = np.empty((3, 3), float)
    rp = np.empty((3, 3), float)
    rbp = np.empty((3, 3), float)
    rn = np.empty((3, 3), float)
    rbpn = np.empty((3, 3), float)
    dpsi, deps, epsa, rb, rp, rbp, rn, rbpn = erfa_ufunc.pn00b(2400000.5, 53736.0)
    vvd(dpsi, -0.9632552291148362783e-5, 1e-12, "pn00b", "dpsi", status)
    vvd(deps,  0.4063197106621159367e-4, 1e-12, "pn00b", "deps", status)
    vvd(epsa,  0.4090791789404229916, 1e-12, "pn00b", "epsa", status)
    vvd(rb[0][0], 0.9999999999999942498, 1e-12, "pn00b", "rb11", status)
    vvd(rb[0][1], -0.7078279744199196626e-7, 1e-16, "pn00b", "rb12", status)
    vvd(rb[0][2], 0.8056217146976134152e-7, 1e-16, "pn00b", "rb13", status)
    vvd(rb[1][0], 0.7078279477857337206e-7, 1e-16, "pn00b", "rb21", status)
    vvd(rb[1][1], 0.9999999999999969484, 1e-12, "pn00b", "rb22", status)
    vvd(rb[1][2], 0.3306041454222136517e-7, 1e-16, "pn00b", "rb23", status)
    vvd(rb[2][0], -0.8056217380986972157e-7, 1e-16, "pn00b", "rb31", status)
    vvd(rb[2][1], -0.3306040883980552500e-7, 1e-16, "pn00b", "rb32", status)
    vvd(rb[2][2], 0.9999999999999962084, 1e-12, "pn00b", "rb33", status)
    vvd(rp[0][0], 0.9999989300532289018, 1e-12, "pn00b", "rp11", status)
    vvd(rp[0][1], -0.1341647226791824349e-2, 1e-14, "pn00b", "rp12", status)
    vvd(rp[0][2], -0.5829880927190296547e-3, 1e-14, "pn00b", "rp13", status)
    vvd(rp[1][0], 0.1341647231069759008e-2, 1e-14, "pn00b", "rp21", status)
    vvd(rp[1][1], 0.9999990999908750433, 1e-12, "pn00b", "rp22", status)
    vvd(rp[1][2], -0.3837444441583715468e-6, 1e-14, "pn00b", "rp23", status)
    vvd(rp[2][0], 0.5829880828740957684e-3, 1e-14, "pn00b", "rp31", status)
    vvd(rp[2][1], -0.3984203267708834759e-6, 1e-14, "pn00b", "rp32", status)
    vvd(rp[2][2], 0.9999998300623538046, 1e-12, "pn00b", "rp33", status)
    vvd(rbp[0][0], 0.9999989300052243993, 1e-12, "pn00b", "rbp11", status)
    vvd(rbp[0][1], -0.1341717990239703727e-2, 1e-14, "pn00b", "rbp12", status)
    vvd(rbp[0][2], -0.5829075749891684053e-3, 1e-14, "pn00b", "rbp13", status)
    vvd(rbp[1][0], 0.1341718013831739992e-2, 1e-14, "pn00b", "rbp21", status)
    vvd(rbp[1][1], 0.9999990998959191343, 1e-12, "pn00b", "rbp22", status)
    vvd(rbp[1][2], -0.3505759733565421170e-6, 1e-14, "pn00b", "rbp23", status)
    vvd(rbp[2][0], 0.5829075206857717883e-3, 1e-14, "pn00b", "rbp31", status)
    vvd(rbp[2][1], -0.4315219955198608970e-6, 1e-14, "pn00b", "rbp32", status)
    vvd(rbp[2][2], 0.9999998301093036269, 1e-12, "pn00b", "rbp33", status)
    vvd(rn[0][0], 0.9999999999536069682, 1e-12, "pn00b", "rn11", status)
    vvd(rn[0][1], 0.8837746144871248011e-5, 1e-14, "pn00b", "rn12", status)
    vvd(rn[0][2], 0.3831488838252202945e-5, 1e-14, "pn00b", "rn13", status)
    vvd(rn[1][0], -0.8837590456632304720e-5, 1e-14, "pn00b", "rn21", status)
    vvd(rn[1][1], 0.9999999991354692733, 1e-12, "pn00b", "rn22", status)
    vvd(rn[1][2], -0.4063198798559591654e-4, 1e-14, "pn00b", "rn23", status)
    vvd(rn[2][0], -0.3831847930134941271e-5, 1e-14, "pn00b", "rn31", status)
    vvd(rn[2][1], 0.4063195412258168380e-4, 1e-14, "pn00b", "rn32", status)
    vvd(rn[2][2], 0.9999999991671806225, 1e-12, "pn00b", "rn33", status)
    vvd(rbpn[0][0], 0.9999989440499982806, 1e-12, "pn00b", "rbpn11", status)
    vvd(rbpn[0][1], -0.1332880253640849194e-2, 1e-14, "pn00b", "rbpn12", status)
    vvd(rbpn[0][2], -0.5790760898731091166e-3, 1e-14, "pn00b", "rbpn13", status)
    vvd(rbpn[1][0], 0.1332856746979949638e-2, 1e-14, "pn00b", "rbpn21", status)
    vvd(rbpn[1][1], 0.9999991109064768883, 1e-12, "pn00b", "rbpn22", status)
    vvd(rbpn[1][2], -0.4097740555723081811e-4, 1e-14, "pn00b", "rbpn23", status)
    vvd(rbpn[2][0], 0.5791301929950208873e-3, 1e-14, "pn00b", "rbpn31", status)
    vvd(rbpn[2][1], 0.4020553681373720832e-4, 1e-14, "pn00b", "rbpn32", status)
    vvd(rbpn[2][2], 0.9999998314958529887, 1e-12, "pn00b", "rbpn33", status)


def test_pn06():
    rb = np.empty((3, 3), float)
    rp = np.empty((3, 3), float)
    rbp = np.empty((3, 3), float)
    rn = np.empty((3, 3), float)
    rbpn = np.empty((3, 3), float)
    dpsi = -0.9632552291149335877e-5
    deps = 0.4063197106621141414e-4
    epsa, rb, rp, rbp, rn, rbpn = erfa_ufunc.pn06(2400000.5, 53736.0, dpsi, deps)
    vvd(epsa, 0.4090789763356509926, 1e-12, "pn06", "epsa", status)
    vvd(rb[0][0], 0.9999999999999942497, 1e-12, "pn06", "rb11", status)
    vvd(rb[0][1], -0.7078368960971557145e-7, 1e-14, "pn06", "rb12", status)
    vvd(rb[0][2], 0.8056213977613185606e-7, 1e-14, "pn06", "rb13", status)
    vvd(rb[1][0], 0.7078368694637674333e-7, 1e-14, "pn06", "rb21", status)
    vvd(rb[1][1], 0.9999999999999969484, 1e-12, "pn06", "rb22", status)
    vvd(rb[1][2], 0.3305943742989134124e-7, 1e-14, "pn06", "rb23", status)
    vvd(rb[2][0], -0.8056214211620056792e-7, 1e-14, "pn06", "rb31", status)
    vvd(rb[2][1], -0.3305943172740586950e-7, 1e-14, "pn06", "rb32", status)
    vvd(rb[2][2], 0.9999999999999962084, 1e-12, "pn06", "rb33", status)
    vvd(rp[0][0], 0.9999989300536854831, 1e-12, "pn06", "rp11", status)
    vvd(rp[0][1], -0.1341646886204443795e-2, 1e-14, "pn06", "rp12", status)
    vvd(rp[0][2], -0.5829880933488627759e-3, 1e-14, "pn06", "rp13", status)
    vvd(rp[1][0], 0.1341646890569782183e-2, 1e-14, "pn06", "rp21", status)
    vvd(rp[1][1], 0.9999990999913319321, 1e-12, "pn06", "rp22", status)
    vvd(rp[1][2], -0.3835944216374477457e-6, 1e-14, "pn06", "rp23", status)
    vvd(rp[2][0], 0.5829880833027867368e-3, 1e-14, "pn06", "rp31", status)
    vvd(rp[2][1], -0.3985701514686976112e-6, 1e-14, "pn06", "rp32", status)
    vvd(rp[2][2], 0.9999998300623534950, 1e-12, "pn06", "rp33", status)
    vvd(rbp[0][0], 0.9999989300056797893, 1e-12, "pn06", "rbp11", status)
    vvd(rbp[0][1], -0.1341717650545059598e-2, 1e-14, "pn06", "rbp12", status)
    vvd(rbp[0][2], -0.5829075756493728856e-3, 1e-14, "pn06", "rbp13", status)
    vvd(rbp[1][0], 0.1341717674223918101e-2, 1e-14, "pn06", "rbp21", status)
    vvd(rbp[1][1], 0.9999990998963748448, 1e-12, "pn06", "rbp22", status)
    vvd(rbp[1][2], -0.3504269280170069029e-6, 1e-14, "pn06", "rbp23", status)
    vvd(rbp[2][0], 0.5829075211461454599e-3, 1e-14, "pn06", "rbp31", status)
    vvd(rbp[2][1], -0.4316708436255949093e-6, 1e-14, "pn06", "rbp32", status)
    vvd(rbp[2][2], 0.9999998301093032943, 1e-12, "pn06", "rbp33", status)
    vvd(rn[0][0], 0.9999999999536069682, 1e-12, "pn06", "rn11", status)
    vvd(rn[0][1], 0.8837746921149881914e-5, 1e-14, "pn06", "rn12", status)
    vvd(rn[0][2], 0.3831487047682968703e-5, 1e-14, "pn06", "rn13", status)
    vvd(rn[1][0], -0.8837591232983692340e-5, 1e-14, "pn06", "rn21", status)
    vvd(rn[1][1], 0.9999999991354692664, 1e-12, "pn06", "rn22", status)
    vvd(rn[1][2], -0.4063198798558931215e-4, 1e-14, "pn06", "rn23", status)
    vvd(rn[2][0], -0.3831846139597250235e-5, 1e-14, "pn06", "rn31", status)
    vvd(rn[2][1], 0.4063195412258792914e-4, 1e-14, "pn06", "rn32", status)
    vvd(rn[2][2], 0.9999999991671806293, 1e-12, "pn06", "rn33", status)
    vvd(rbpn[0][0], 0.9999989440504506688, 1e-12, "pn06", "rbpn11", status)
    vvd(rbpn[0][1], -0.1332879913170492655e-2, 1e-14, "pn06", "rbpn12", status)
    vvd(rbpn[0][2], -0.5790760923225655753e-3, 1e-14, "pn06", "rbpn13", status)
    vvd(rbpn[1][0], 0.1332856406595754748e-2, 1e-14, "pn06", "rbpn21", status)
    vvd(rbpn[1][1], 0.9999991109069366795, 1e-12, "pn06", "rbpn22", status)
    vvd(rbpn[1][2], -0.4097725651142641812e-4, 1e-14, "pn06", "rbpn23", status)
    vvd(rbpn[2][0], 0.5791301952321296716e-3, 1e-14, "pn06", "rbpn31", status)
    vvd(rbpn[2][1], 0.4020538796195230577e-4, 1e-14, "pn06", "rbpn32", status)
    vvd(rbpn[2][2], 0.9999998314958576778, 1e-12, "pn06", "rbpn33", status)


def test_pn06a():
    rb = np.empty((3, 3), float)
    rp = np.empty((3, 3), float)
    rbp = np.empty((3, 3), float)
    rn = np.empty((3, 3), float)
    rbpn = np.empty((3, 3), float)
    dpsi, deps, epsa, rb, rp, rbp, rn, rbpn = erfa_ufunc.pn06a(2400000.5, 53736.0)
    vvd(dpsi, -0.9630912025820308797e-5, 1e-12, "pn06a", "dpsi", status)
    vvd(deps,  0.4063238496887249798e-4, 1e-12, "pn06a", "deps", status)
    vvd(epsa,  0.4090789763356509926, 1e-12, "pn06a", "epsa", status)
    vvd(rb[0][0], 0.9999999999999942497, 1e-12, "pn06a", "rb11", status)
    vvd(rb[0][1], -0.7078368960971557145e-7, 1e-14, "pn06a", "rb12", status)
    vvd(rb[0][2], 0.8056213977613185606e-7, 1e-14, "pn06a", "rb13", status)
    vvd(rb[1][0], 0.7078368694637674333e-7, 1e-14, "pn06a", "rb21", status)
    vvd(rb[1][1], 0.9999999999999969484, 1e-12, "pn06a", "rb22", status)
    vvd(rb[1][2], 0.3305943742989134124e-7, 1e-14, "pn06a", "rb23", status)
    vvd(rb[2][0], -0.8056214211620056792e-7, 1e-14, "pn06a", "rb31", status)
    vvd(rb[2][1], -0.3305943172740586950e-7, 1e-14, "pn06a", "rb32", status)
    vvd(rb[2][2], 0.9999999999999962084, 1e-12, "pn06a", "rb33", status)
    vvd(rp[0][0], 0.9999989300536854831, 1e-12, "pn06a", "rp11", status)
    vvd(rp[0][1], -0.1341646886204443795e-2, 1e-14, "pn06a", "rp12", status)
    vvd(rp[0][2], -0.5829880933488627759e-3, 1e-14, "pn06a", "rp13", status)
    vvd(rp[1][0], 0.1341646890569782183e-2, 1e-14, "pn06a", "rp21", status)
    vvd(rp[1][1], 0.9999990999913319321, 1e-12, "pn06a", "rp22", status)
    vvd(rp[1][2], -0.3835944216374477457e-6, 1e-14, "pn06a", "rp23", status)
    vvd(rp[2][0], 0.5829880833027867368e-3, 1e-14, "pn06a", "rp31", status)
    vvd(rp[2][1], -0.3985701514686976112e-6, 1e-14, "pn06a", "rp32", status)
    vvd(rp[2][2], 0.9999998300623534950, 1e-12, "pn06a", "rp33", status)
    vvd(rbp[0][0], 0.9999989300056797893, 1e-12, "pn06a", "rbp11", status)
    vvd(rbp[0][1], -0.1341717650545059598e-2, 1e-14, "pn06a", "rbp12", status)
    vvd(rbp[0][2], -0.5829075756493728856e-3, 1e-14, "pn06a", "rbp13", status)
    vvd(rbp[1][0], 0.1341717674223918101e-2, 1e-14, "pn06a", "rbp21", status)
    vvd(rbp[1][1], 0.9999990998963748448, 1e-12, "pn06a", "rbp22", status)
    vvd(rbp[1][2], -0.3504269280170069029e-6, 1e-14, "pn06a", "rbp23", status)
    vvd(rbp[2][0], 0.5829075211461454599e-3, 1e-14, "pn06a", "rbp31", status)
    vvd(rbp[2][1], -0.4316708436255949093e-6, 1e-14, "pn06a", "rbp32", status)
    vvd(rbp[2][2], 0.9999998301093032943, 1e-12, "pn06a", "rbp33", status)
    vvd(rn[0][0], 0.9999999999536227668, 1e-12, "pn06a", "rn11", status)
    vvd(rn[0][1], 0.8836241998111535233e-5, 1e-14, "pn06a", "rn12", status)
    vvd(rn[0][2], 0.3830834608415287707e-5, 1e-14, "pn06a", "rn13", status)
    vvd(rn[1][0], -0.8836086334870740138e-5, 1e-14, "pn06a", "rn21", status)
    vvd(rn[1][1], 0.9999999991354657474, 1e-12, "pn06a", "rn22", status)
    vvd(rn[1][2], -0.4063240188248455065e-4, 1e-14, "pn06a", "rn23", status)
    vvd(rn[2][0], -0.3831193642839398128e-5, 1e-14, "pn06a", "rn31", status)
    vvd(rn[2][1], 0.4063236803101479770e-4, 1e-14, "pn06a", "rn32", status)
    vvd(rn[2][2], 0.9999999991671663114, 1e-12, "pn06a", "rn33", status)
    vvd(rbpn[0][0], 0.9999989440480669738, 1e-12, "pn06a", "rbpn11", status)
    vvd(rbpn[0][1], -0.1332881418091915973e-2, 1e-14, "pn06a", "rbpn12", status)
    vvd(rbpn[0][2], -0.5790767447612042565e-3, 1e-14, "pn06a", "rbpn13", status)
    vvd(rbpn[1][0], 0.1332857911250989133e-2, 1e-14, "pn06a", "rbpn21", status)
    vvd(rbpn[1][1], 0.9999991109049141908, 1e-12, "pn06a", "rbpn22", status)
    vvd(rbpn[1][2], -0.4097767128546784878e-4, 1e-14, "pn06a", "rbpn23", status)
    vvd(rbpn[2][0], 0.5791308482835292617e-3, 1e-14, "pn06a", "rbpn31", status)
    vvd(rbpn[2][1], 0.4020580099454020310e-4, 1e-14, "pn06a", "rbpn32", status)
    vvd(rbpn[2][2], 0.9999998314954628695, 1e-12, "pn06a", "rbpn33", status)


def test_pnm00a():
    rbpn = np.empty((3, 3), float)
    rbpn = erfa_ufunc.pnm00a(2400000.5, 50123.9999)
    vvd(rbpn[0][0], 0.9999995832793134257, 1e-12, "pnm00a", "11", status)
    vvd(rbpn[0][1], 0.8372384254137809439e-3, 1e-14, "pnm00a", "12", status)
    vvd(rbpn[0][2], 0.3639684306407150645e-3, 1e-14, "pnm00a", "13", status)
    vvd(rbpn[1][0], -0.8372535226570394543e-3, 1e-14, "pnm00a", "21", status)
    vvd(rbpn[1][1], 0.9999996486491582471, 1e-12, "pnm00a", "22", status)
    vvd(rbpn[1][2], 0.4132915262664072381e-4, 1e-14, "pnm00a", "23", status)
    vvd(rbpn[2][0], -0.3639337004054317729e-3, 1e-14, "pnm00a", "31", status)
    vvd(rbpn[2][1], -0.4163386925461775873e-4, 1e-14, "pnm00a", "32", status)
    vvd(rbpn[2][2], 0.9999999329094390695, 1e-12, "pnm00a", "33", status)


def test_pnm00b():
    rbpn = np.empty((3, 3), float)
    rbpn = erfa_ufunc.pnm00b(2400000.5, 50123.9999)
    vvd(rbpn[0][0], 0.9999995832776208280, 1e-12, "pnm00b", "11", status)
    vvd(rbpn[0][1], 0.8372401264429654837e-3, 1e-14, "pnm00b", "12", status)
    vvd(rbpn[0][2], 0.3639691681450271771e-3, 1e-14, "pnm00b", "13", status)
    vvd(rbpn[1][0], -0.8372552234147137424e-3, 1e-14, "pnm00b", "21", status)
    vvd(rbpn[1][1], 0.9999996486477686123, 1e-12, "pnm00b", "22", status)
    vvd(rbpn[1][2], 0.4132832190946052890e-4, 1e-14, "pnm00b", "23", status)
    vvd(rbpn[2][0], -0.3639344385341866407e-3, 1e-14, "pnm00b", "31", status)
    vvd(rbpn[2][1], -0.4163303977421522785e-4, 1e-14, "pnm00b", "32", status)
    vvd(rbpn[2][2], 0.9999999329092049734, 1e-12, "pnm00b", "33", status)


def test_pnm06a():
    rbpn = np.empty((3, 3), float)
    rbpn = erfa_ufunc.pnm06a(2400000.5, 50123.9999)
    vvd(rbpn[0][0], 0.9999995832794205484, 1e-12, "pnm06a", "11", status)
    vvd(rbpn[0][1], 0.8372382772630962111e-3, 1e-14, "pnm06a", "12", status)
    vvd(rbpn[0][2], 0.3639684771140623099e-3, 1e-14, "pnm06a", "13", status)
    vvd(rbpn[1][0], -0.8372533744743683605e-3, 1e-14, "pnm06a", "21", status)
    vvd(rbpn[1][1], 0.9999996486492861646, 1e-12, "pnm06a", "22", status)
    vvd(rbpn[1][2], 0.4132905944611019498e-4, 1e-14, "pnm06a", "23", status)
    vvd(rbpn[2][0], -0.3639337469629464969e-3, 1e-14, "pnm06a", "31", status)
    vvd(rbpn[2][1], -0.4163377605910663999e-4, 1e-14, "pnm06a", "32", status)
    vvd(rbpn[2][2], 0.9999999329094260057, 1e-12, "pnm06a", "33", status)


def test_pnm80():
    rmatpn = np.empty((3, 3), float)
    rmatpn = erfa_ufunc.pnm80(2400000.5, 50123.9999)
    vvd(rmatpn[0][0], 0.9999995831934611169, 1e-12, "pnm80", "11", status)
    vvd(rmatpn[0][1], 0.8373654045728124011e-3, 1e-14, "pnm80", "12", status)
    vvd(rmatpn[0][2], 0.3639121916933106191e-3, 1e-14, "pnm80", "13", status)
    vvd(rmatpn[1][0], -0.8373804896118301316e-3, 1e-14, "pnm80", "21", status)
    vvd(rmatpn[1][1], 0.9999996485439674092, 1e-12, "pnm80", "22", status)
    vvd(rmatpn[1][2], 0.4130202510421549752e-4, 1e-14, "pnm80", "23", status)
    vvd(rmatpn[2][0], -0.3638774789072144473e-3, 1e-14, "pnm80", "31", status)
    vvd(rmatpn[2][1], -0.4160674085851722359e-4, 1e-14, "pnm80", "32", status)
    vvd(rmatpn[2][2], 0.9999999329310274805, 1e-12, "pnm80", "33", status)


def test_pom00():
    rpom = np.empty((3, 3), float)
    xp = 2.55060238e-7
    yp = 1.860359247e-6
    sp = -0.1367174580728891460e-10
    rpom = erfa_ufunc.pom00(xp, yp, sp)
    vvd(rpom[0][0], 0.9999999999999674721, 1e-12, "pom00", "11", status)
    vvd(rpom[0][1], -0.1367174580728846989e-10, 1e-16, "pom00", "12", status)
    vvd(rpom[0][2], 0.2550602379999972345e-6, 1e-16, "pom00", "13", status)
    vvd(rpom[1][0], 0.1414624947957029801e-10, 1e-16, "pom00", "21", status)
    vvd(rpom[1][1], 0.9999999999982695317, 1e-12, "pom00", "22", status)
    vvd(rpom[1][2], -0.1860359246998866389e-5, 1e-16, "pom00", "23", status)
    vvd(rpom[2][0], -0.2550602379741215021e-6, 1e-16, "pom00", "31", status)
    vvd(rpom[2][1], 0.1860359247002414021e-5, 1e-16, "pom00", "32", status)
    vvd(rpom[2][2], 0.9999999999982370039, 1e-12, "pom00", "33", status)


def test_ppp():
    a = np.empty((3,), float)
    b = np.empty((3,), float)
    apb = np.empty((3,), float)
    a[0] = 2.0
    a[1] = 2.0
    a[2] = 3.0
    b[0] = 1.0
    b[1] = 3.0
    b[2] = 4.0
    apb = erfa_ufunc.ppp(a, b)
    vvd(apb[0], 3.0, 1e-12, "ppp", "0", status)
    vvd(apb[1], 5.0, 1e-12, "ppp", "1", status)
    vvd(apb[2], 7.0, 1e-12, "ppp", "2", status)


def test_ppsp():
    a = np.empty((3,), float)
    b = np.empty((3,), float)
    apsb = np.empty((3,), float)
    a[0] = 2.0
    a[1] = 2.0
    a[2] = 3.0
    s = 5.0
    b[0] = 1.0
    b[1] = 3.0
    b[2] = 4.0
    apsb = erfa_ufunc.ppsp(a, s, b)
    vvd(apsb[0], 7.0, 1e-12, "ppsp", "0", status)
    vvd(apsb[1], 17.0, 1e-12, "ppsp", "1", status)
    vvd(apsb[2], 23.0, 1e-12, "ppsp", "2", status)


def test_pr00():
    dpsipr, depspr = erfa_ufunc.pr00(2400000.5, 53736)
    vvd(dpsipr, -0.8716465172668347629e-7, 1e-22, "pr00", "dpsipr", status)
    vvd(depspr, -0.7342018386722813087e-8, 1e-22, "pr00", "depspr", status)


def test_prec76():
    ep01 = 2400000.5
    ep02 = 33282.0
    ep11 = 2400000.5
    ep12 = 51544.0
    zeta, z, theta = erfa_ufunc.prec76(ep01, ep02, ep11, ep12)
    vvd(zeta,  0.5588961642000161243e-2, 1e-12, "prec76", "zeta",  status)
    vvd(z,     0.5589922365870680624e-2, 1e-12, "prec76", "z",     status)
    vvd(theta, 0.4858945471687296760e-2, 1e-12, "prec76", "theta", status)


def test_pv2p():
    pv = np.empty((), erfa_ufunc.dt_pv)
    p = np.empty((3,), float)
    pv['p'][0] = 0.3
    pv['p'][1] = 1.2
    pv['p'][2] = -2.5
    pv['v'][0] = -0.5
    pv['v'][1] = 3.1
    pv['v'][2] = 0.9
    p = erfa_ufunc.pv2p(pv)
    vvd(p[0],  0.3, 0.0, "pv2p", "1", status)
    vvd(p[1],  1.2, 0.0, "pv2p", "2", status)
    vvd(p[2], -2.5, 0.0, "pv2p", "3", status)


def test_pv2s():
    pv = np.empty((), erfa_ufunc.dt_pv)
    pv['p'][0] = -0.4514964673880165
    pv['p'][1] = 0.03093394277342585
    pv['p'][2] = 0.05594668105108779
    pv['v'][0] = 1.292270850663260e-5
    pv['v'][1] = 2.652814182060692e-6
    pv['v'][2] = 2.568431853930293e-6
    theta, phi, r, td, pd, rd = erfa_ufunc.pv2s(pv)
    vvd(theta, 3.073185307179586515, 1e-12, "pv2s", "theta", status)
    vvd(phi, 0.1229999999999999992, 1e-12, "pv2s", "phi", status)
    vvd(r, 0.4559999999999999757, 1e-12, "pv2s", "r", status)
    vvd(td, -0.7800000000000000364e-5, 1e-16, "pv2s", "td", status)
    vvd(pd, 0.9010000000000001639e-5, 1e-16, "pv2s", "pd", status)
    vvd(rd, -0.1229999999999999832e-4, 1e-16, "pv2s", "rd", status)


def test_pvdpv():
    a = np.empty((), erfa_ufunc.dt_pv)
    b = np.empty((), erfa_ufunc.dt_pv)
    adb = np.empty((), erfa_ufunc.dt_pvdpv)
    a['p'][0] = 2.0
    a['p'][1] = 2.0
    a['p'][2] = 3.0
    a['v'][0] = 6.0
    a['v'][1] = 0.0
    a['v'][2] = 4.0
    b['p'][0] = 1.0
    b['p'][1] = 3.0
    b['p'][2] = 4.0
    b['v'][0] = 0.0
    b['v'][1] = 2.0
    b['v'][2] = 8.0
    adb = erfa_ufunc.pvdpv(a, b)
    vvd(adb[0], 20.0, 1e-12, "pvdpv", "1", status)
    vvd(adb[1], 50.0, 1e-12, "pvdpv", "2", status)


def test_pvm():
    pv = np.empty((), erfa_ufunc.dt_pv)
    pv['p'][0] = 0.3
    pv['p'][1] = 1.2
    pv['p'][2] = -2.5
    pv['v'][0] = 0.45
    pv['v'][1] = -0.25
    pv['v'][2] = 1.1
    r, s = erfa_ufunc.pvm(pv)
    vvd(r, 2.789265136196270604, 1e-12, "pvm", "r", status)
    vvd(s, 1.214495780149111922, 1e-12, "pvm", "s", status)


def test_pvmpv():
    a = np.empty((), erfa_ufunc.dt_pv)
    b = np.empty((), erfa_ufunc.dt_pv)
    amb = np.empty((), erfa_ufunc.dt_pv)
    a['p'][0] = 2.0
    a['p'][1] = 2.0
    a['p'][2] = 3.0
    a['v'][0] = 5.0
    a['v'][1] = 6.0
    a['v'][2] = 3.0
    b['p'][0] = 1.0
    b['p'][1] = 3.0
    b['p'][2] = 4.0
    b['v'][0] = 3.0
    b['v'][1] = 2.0
    b['v'][2] = 1.0
    amb = erfa_ufunc.pvmpv(a, b)
    vvd(amb[0][0],  1.0, 1e-12, "pvmpv", "11", status)
    vvd(amb[0][1], -1.0, 1e-12, "pvmpv", "21", status)
    vvd(amb[0][2], -1.0, 1e-12, "pvmpv", "31", status)
    vvd(amb[1][0],  2.0, 1e-12, "pvmpv", "12", status)
    vvd(amb[1][1],  4.0, 1e-12, "pvmpv", "22", status)
    vvd(amb[1][2],  2.0, 1e-12, "pvmpv", "32", status)


def test_pvppv():
    a = np.empty((), erfa_ufunc.dt_pv)
    b = np.empty((), erfa_ufunc.dt_pv)
    apb = np.empty((), erfa_ufunc.dt_pv)
    a['p'][0] = 2.0
    a['p'][1] = 2.0
    a['p'][2] = 3.0
    a['v'][0] = 5.0
    a['v'][1] = 6.0
    a['v'][2] = 3.0
    b['p'][0] = 1.0
    b['p'][1] = 3.0
    b['p'][2] = 4.0
    b['v'][0] = 3.0
    b['v'][1] = 2.0
    b['v'][2] = 1.0
    apb = erfa_ufunc.pvppv(a, b)
    vvd(apb[0][0], 3.0, 1e-12, "pvppv", "p1", status)
    vvd(apb[0][1], 5.0, 1e-12, "pvppv", "p2", status)
    vvd(apb[0][2], 7.0, 1e-12, "pvppv", "p3", status)
    vvd(apb[1][0], 8.0, 1e-12, "pvppv", "v1", status)
    vvd(apb[1][1], 8.0, 1e-12, "pvppv", "v2", status)
    vvd(apb[1][2], 4.0, 1e-12, "pvppv", "v3", status)


def test_pvstar():
    pv = np.empty((), erfa_ufunc.dt_pv)
    pv['p'][0] = 126668.5912743160601
    pv['p'][1] = 2136.792716839935195
    pv['p'][2] = -245251.2339876830091
    pv['v'][0] = -0.4051854035740712739e-2
    pv['v'][1] = -0.6253919754866173866e-2
    pv['v'][2] = 0.1189353719774107189e-1
    ra, dec, pmr, pmd, px, rv, j = erfa_ufunc.pvstar(pv)
    vvd(ra, 0.1686756e-1, 1e-12, "pvstar", "ra", status)
    vvd(dec, -1.093989828, 1e-12, "pvstar", "dec", status)
    vvd(pmr, -0.1783235160000472788e-4, 1e-16, "pvstar", "pmr", status)
    vvd(pmd, 0.2336024047000619347e-5, 1e-16, "pvstar", "pmd", status)
    vvd(px, 0.74723, 1e-12, "pvstar", "px", status)
    vvd(rv, -21.60000010107306010, 1e-11, "pvstar", "rv", status)
    viv(j, 0, "pvstar", "j", status)


def test_pvtob():
    pv = np.empty((), erfa_ufunc.dt_pv)
    elong = 2.0
    phi = 0.5
    hm = 3000.0
    xp = 1e-6
    yp = -0.5e-6
    sp = 1e-8
    theta = 5.0
    pv = erfa_ufunc.pvtob(elong, phi, hm, xp, yp, sp, theta)
    vvd(pv['p'][0], 4225081.367071159207, 1e-5, "pvtob", "p(1)", status)
    vvd(pv['p'][1], 3681943.215856198144, 1e-5, "pvtob", "p(2)", status)
    vvd(pv['p'][2], 3041149.399241260785, 1e-5, "pvtob", "p(3)", status)
    vvd(pv['v'][0], -268.4915389365998787, 1e-9, "pvtob", "v(1)", status)
    vvd(pv['v'][1], 308.0977983288903123, 1e-9, "pvtob", "v(2)", status)
    vvd(pv['v'][2], 0, 0, "pvtob", "v(3)", status)


def test_pvu():
    pv = np.empty((), erfa_ufunc.dt_pv)
    upv = np.empty((), erfa_ufunc.dt_pv)
    pv['p'][0] = 126668.5912743160734
    pv['p'][1] = 2136.792716839935565
    pv['p'][2] = -245251.2339876830229
    pv['v'][0] = -0.4051854035740713039e-2
    pv['v'][1] = -0.6253919754866175788e-2
    pv['v'][2] = 0.1189353719774107615e-1
    upv = erfa_ufunc.pvu(2920.0, pv)
    vvd(upv['p'][0], 126656.7598605317105, 1e-6, "pvu", "p1", status)
    vvd(upv['p'][1], 2118.531271155726332, 1e-8, "pvu", "p2", status)
    vvd(upv['p'][2], -245216.5048590656190, 1e-6, "pvu", "p3", status)
    vvd(upv['v'][0], -0.4051854035740713039e-2, 1e-12, "pvu", "v1", status)
    vvd(upv['v'][1], -0.6253919754866175788e-2, 1e-12, "pvu", "v2", status)
    vvd(upv['v'][2], 0.1189353719774107615e-1, 1e-12, "pvu", "v3", status)


def test_pvup():
    pv = np.empty((), erfa_ufunc.dt_pv)
    p = np.empty((3,), float)
    pv['p'][0] = 126668.5912743160734
    pv['p'][1] = 2136.792716839935565
    pv['p'][2] = -245251.2339876830229
    pv['v'][0] = -0.4051854035740713039e-2
    pv['v'][1] = -0.6253919754866175788e-2
    pv['v'][2] = 0.1189353719774107615e-1
    p = erfa_ufunc.pvup(2920.0, pv)
    vvd(p[0],  126656.7598605317105,   1e-6, "pvup", "1", status)
    vvd(p[1],    2118.531271155726332, 1e-8, "pvup", "2", status)
    vvd(p[2], -245216.5048590656190,   1e-6, "pvup", "3", status)


def test_pvxpv():
    a = np.empty((), erfa_ufunc.dt_pv)
    b = np.empty((), erfa_ufunc.dt_pv)
    axb = np.empty((), erfa_ufunc.dt_pv)
    a['p'][0] = 2.0
    a['p'][1] = 2.0
    a['p'][2] = 3.0
    a['v'][0] = 6.0
    a['v'][1] = 0.0
    a['v'][2] = 4.0
    b['p'][0] = 1.0
    b['p'][1] = 3.0
    b['p'][2] = 4.0
    b['v'][0] = 0.0
    b['v'][1] = 2.0
    b['v'][2] = 8.0
    axb = erfa_ufunc.pvxpv(a, b)
    vvd(axb[0][0],  -1.0, 1e-12, "pvxpv", "p1", status)
    vvd(axb[0][1],  -5.0, 1e-12, "pvxpv", "p2", status)
    vvd(axb[0][2],   4.0, 1e-12, "pvxpv", "p3", status)
    vvd(axb[1][0],  -2.0, 1e-12, "pvxpv", "v1", status)
    vvd(axb[1][1], -36.0, 1e-12, "pvxpv", "v2", status)
    vvd(axb[1][2],  22.0, 1e-12, "pvxpv", "v3", status)


def test_pxp():
    a = np.empty((3,), float)
    b = np.empty((3,), float)
    axb = np.empty((3,), float)
    a[0] = 2.0
    a[1] = 2.0
    a[2] = 3.0
    b[0] = 1.0
    b[1] = 3.0
    b[2] = 4.0
    axb = erfa_ufunc.pxp(a, b)
    vvd(axb[0], -1.0, 1e-12, "pxp", "1", status)
    vvd(axb[1], -5.0, 1e-12, "pxp", "2", status)
    vvd(axb[2],  4.0, 1e-12, "pxp", "3", status)


def test_refco():
    phpa = 800.0
    tc = 10.0
    rh = 0.9
    wl = 0.4
    refa, refb = erfa_ufunc.refco(phpa, tc, rh, wl)
    vvd(refa, 0.2264949956241415009e-3, 1e-15, "refco", "refa", status)
    vvd(refb, -0.2598658261729343970e-6, 1e-18, "refco", "refb", status)


def test_rm2v():
    r = np.empty((3, 3), float)
    w = np.empty((3,), float)
    r[0][0] = 0.00
    r[0][1] = -0.80
    r[0][2] = -0.60
    r[1][0] = 0.80
    r[1][1] = -0.36
    r[1][2] = 0.48
    r[2][0] = 0.60
    r[2][1] = 0.48
    r[2][2] = -0.64
    w = erfa_ufunc.rm2v(r)
    vvd(w[0],  0.0,                  1e-12, "rm2v", "1", status)
    vvd(w[1],  1.413716694115406957, 1e-12, "rm2v", "2", status)
    vvd(w[2], -1.884955592153875943, 1e-12, "rm2v", "3", status)


def test_rv2m():
    w = np.empty((3,), float)
    r = np.empty((3, 3), float)
    w[0] = 0.0
    w[1] = 1.41371669
    w[2] = -1.88495559
    r = erfa_ufunc.rv2m(w)
    vvd(r[0][0], -0.7071067782221119905, 1e-14, "rv2m", "11", status)
    vvd(r[0][1], -0.5656854276809129651, 1e-14, "rv2m", "12", status)
    vvd(r[0][2], -0.4242640700104211225, 1e-14, "rv2m", "13", status)
    vvd(r[1][0],  0.5656854276809129651, 1e-14, "rv2m", "21", status)
    vvd(r[1][1], -0.0925483394532274246, 1e-14, "rv2m", "22", status)
    vvd(r[1][2], -0.8194112531408833269, 1e-14, "rv2m", "23", status)
    vvd(r[2][0],  0.4242640700104211225, 1e-14, "rv2m", "31", status)
    vvd(r[2][1], -0.8194112531408833269, 1e-14, "rv2m", "32", status)
    vvd(r[2][2],  0.3854415612311154341, 1e-14, "rv2m", "33", status)


def test_rx():
    r = np.empty((3, 3), float)
    phi = 0.3456789
    r[0][0] = 2.0
    r[0][1] = 3.0
    r[0][2] = 2.0
    r[1][0] = 3.0
    r[1][1] = 2.0
    r[1][2] = 3.0
    r[2][0] = 3.0
    r[2][1] = 4.0
    r[2][2] = 5.0
    r = erfa_ufunc.rx(phi, r)
    vvd(r[0][0], 2.0, 0.0, "rx", "11", status)
    vvd(r[0][1], 3.0, 0.0, "rx", "12", status)
    vvd(r[0][2], 2.0, 0.0, "rx", "13", status)
    vvd(r[1][0], 3.839043388235612460, 1e-12, "rx", "21", status)
    vvd(r[1][1], 3.237033249594111899, 1e-12, "rx", "22", status)
    vvd(r[1][2], 4.516714379005982719, 1e-12, "rx", "23", status)
    vvd(r[2][0], 1.806030415924501684, 1e-12, "rx", "31", status)
    vvd(r[2][1], 3.085711545336372503, 1e-12, "rx", "32", status)
    vvd(r[2][2], 3.687721683977873065, 1e-12, "rx", "33", status)


def test_rxp():
    r = np.empty((3, 3), float)
    p = np.empty((3,), float)
    rp = np.empty((3,), float)
    r[0][0] = 2.0
    r[0][1] = 3.0
    r[0][2] = 2.0
    r[1][0] = 3.0
    r[1][1] = 2.0
    r[1][2] = 3.0
    r[2][0] = 3.0
    r[2][1] = 4.0
    r[2][2] = 5.0
    p[0] = 0.2
    p[1] = 1.5
    p[2] = 0.1
    rp = erfa_ufunc.rxp(r, p)
    vvd(rp[0], 5.1, 1e-12, "rxp", "1", status)
    vvd(rp[1], 3.9, 1e-12, "rxp", "2", status)
    vvd(rp[2], 7.1, 1e-12, "rxp", "3", status)


def test_rxpv():
    r = np.empty((3, 3), float)
    pv = np.empty((), erfa_ufunc.dt_pv)
    rpv = np.empty((), erfa_ufunc.dt_pv)
    r[0][0] = 2.0
    r[0][1] = 3.0
    r[0][2] = 2.0
    r[1][0] = 3.0
    r[1][1] = 2.0
    r[1][2] = 3.0
    r[2][0] = 3.0
    r[2][1] = 4.0
    r[2][2] = 5.0
    pv['p'][0] = 0.2
    pv['p'][1] = 1.5
    pv['p'][2] = 0.1
    pv['v'][0] = 1.5
    pv['v'][1] = 0.2
    pv['v'][2] = 0.1
    rpv = erfa_ufunc.rxpv(r, pv)
    vvd(rpv['p'][0], 5.1, 1e-12, "rxpv", "11", status)
    vvd(rpv['v'][0], 3.8, 1e-12, "rxpv", "12", status)
    vvd(rpv['p'][1], 3.9, 1e-12, "rxpv", "21", status)
    vvd(rpv['v'][1], 5.2, 1e-12, "rxpv", "22", status)
    vvd(rpv['p'][2], 7.1, 1e-12, "rxpv", "31", status)
    vvd(rpv['v'][2], 5.8, 1e-12, "rxpv", "32", status)


def test_rxr():
    a = np.empty((3, 3), float)
    b = np.empty((3, 3), float)
    atb = np.empty((3, 3), float)
    a[0][0] = 2.0
    a[0][1] = 3.0
    a[0][2] = 2.0
    a[1][0] = 3.0
    a[1][1] = 2.0
    a[1][2] = 3.0
    a[2][0] = 3.0
    a[2][1] = 4.0
    a[2][2] = 5.0
    b[0][0] = 1.0
    b[0][1] = 2.0
    b[0][2] = 2.0
    b[1][0] = 4.0
    b[1][1] = 1.0
    b[1][2] = 1.0
    b[2][0] = 3.0
    b[2][1] = 0.0
    b[2][2] = 1.0
    atb = erfa_ufunc.rxr(a, b)
    vvd(atb[0][0], 20.0, 1e-12, "rxr", "11", status)
    vvd(atb[0][1],  7.0, 1e-12, "rxr", "12", status)
    vvd(atb[0][2],  9.0, 1e-12, "rxr", "13", status)
    vvd(atb[1][0], 20.0, 1e-12, "rxr", "21", status)
    vvd(atb[1][1],  8.0, 1e-12, "rxr", "22", status)
    vvd(atb[1][2], 11.0, 1e-12, "rxr", "23", status)
    vvd(atb[2][0], 34.0, 1e-12, "rxr", "31", status)
    vvd(atb[2][1], 10.0, 1e-12, "rxr", "32", status)
    vvd(atb[2][2], 15.0, 1e-12, "rxr", "33", status)


def test_ry():
    r = np.empty((3, 3), float)
    theta = 0.3456789
    r[0][0] = 2.0
    r[0][1] = 3.0
    r[0][2] = 2.0
    r[1][0] = 3.0
    r[1][1] = 2.0
    r[1][2] = 3.0
    r[2][0] = 3.0
    r[2][1] = 4.0
    r[2][2] = 5.0
    r = erfa_ufunc.ry(theta, r)
    vvd(r[0][0], 0.8651847818978159930, 1e-12, "ry", "11", status)
    vvd(r[0][1], 1.467194920539316554, 1e-12, "ry", "12", status)
    vvd(r[0][2], 0.1875137911274457342, 1e-12, "ry", "13", status)
    vvd(r[1][0], 3, 1e-12, "ry", "21", status)
    vvd(r[1][1], 2, 1e-12, "ry", "22", status)
    vvd(r[1][2], 3, 1e-12, "ry", "23", status)
    vvd(r[2][0], 3.500207892850427330, 1e-12, "ry", "31", status)
    vvd(r[2][1], 4.779889022262298150, 1e-12, "ry", "32", status)
    vvd(r[2][2], 5.381899160903798712, 1e-12, "ry", "33", status)


def test_rz():
    r = np.empty((3, 3), float)
    psi = 0.3456789
    r[0][0] = 2.0
    r[0][1] = 3.0
    r[0][2] = 2.0
    r[1][0] = 3.0
    r[1][1] = 2.0
    r[1][2] = 3.0
    r[2][0] = 3.0
    r[2][1] = 4.0
    r[2][2] = 5.0
    r = erfa_ufunc.rz(psi, r)
    vvd(r[0][0], 2.898197754208926769, 1e-12, "rz", "11", status)
    vvd(r[0][1], 3.500207892850427330, 1e-12, "rz", "12", status)
    vvd(r[0][2], 2.898197754208926769, 1e-12, "rz", "13", status)
    vvd(r[1][0], 2.144865911309686813, 1e-12, "rz", "21", status)
    vvd(r[1][1], 0.865184781897815993, 1e-12, "rz", "22", status)
    vvd(r[1][2], 2.144865911309686813, 1e-12, "rz", "23", status)
    vvd(r[2][0], 3.0, 1e-12, "rz", "31", status)
    vvd(r[2][1], 4.0, 1e-12, "rz", "32", status)
    vvd(r[2][2], 5.0, 1e-12, "rz", "33", status)


def test_s00():
    x = 0.5791308486706011000e-3
    y = 0.4020579816732961219e-4
    s = erfa_ufunc.s00(2400000.5, 53736.0, x, y)
    vvd(s, -0.1220036263270905693e-7, 1e-18, "s00", "", status)


def test_s00a():
    s = erfa_ufunc.s00a(2400000.5, 52541.0)
    vvd(s, -0.1340684448919163584e-7, 1e-18, "s00a", "", status)


def test_s00b():
    s = erfa_ufunc.s00b(2400000.5, 52541.0)
    vvd(s, -0.1340695782951026584e-7, 1e-18, "s00b", "", status)


def test_s06():
    x = 0.5791308486706011000e-3
    y = 0.4020579816732961219e-4
    s = erfa_ufunc.s06(2400000.5, 53736.0, x, y)
    vvd(s, -0.1220032213076463117e-7, 1e-18, "s06", "", status)


def test_s06a():
    s = erfa_ufunc.s06a(2400000.5, 52541.0)
    vvd(s, -0.1340680437291812383e-7, 1e-18, "s06a", "", status)


def test_s2c():
    c = np.empty((3,), float)
    c = erfa_ufunc.s2c(3.0123, -0.999)
    vvd(c[0], -0.5366267667260523906, 1e-12, "s2c", "1", status)
    vvd(c[1],  0.0697711109765145365, 1e-12, "s2c", "2", status)
    vvd(c[2], -0.8409302618566214041, 1e-12, "s2c", "3", status)


def test_s2p():
    p = np.empty((3,), float)
    p = erfa_ufunc.s2p(-3.21, 0.123, 0.456)
    vvd(p[0], -0.4514964673880165228, 1e-12, "s2p", "x", status)
    vvd(p[1],  0.0309339427734258688, 1e-12, "s2p", "y", status)
    vvd(p[2],  0.0559466810510877933, 1e-12, "s2p", "z", status)


def test_s2pv():
    pv = np.empty((), erfa_ufunc.dt_pv)
    pv = erfa_ufunc.s2pv(-3.21, 0.123, 0.456, -7.8e-6, 9.01e-6, -1.23e-5)
    vvd(pv['p'][0], -0.4514964673880165228, 1e-12, "s2pv", "x", status)
    vvd(pv['p'][1],  0.0309339427734258688, 1e-12, "s2pv", "y", status)
    vvd(pv['p'][2],  0.0559466810510877933, 1e-12, "s2pv", "z", status)
    vvd(pv['v'][0],  0.1292270850663260170e-4, 1e-16, "s2pv", "vx", status)
    vvd(pv['v'][1],  0.2652814182060691422e-5, 1e-16, "s2pv", "vy", status)
    vvd(pv['v'][2],  0.2568431853930292259e-5, 1e-16, "s2pv", "vz", status)


def test_s2xpv():
    pv = np.empty((), erfa_ufunc.dt_pv)
    spv = np.empty((), erfa_ufunc.dt_pv)
    s1 = 2.0
    s2 = 3.0
    pv['p'][0] = 0.3
    pv['p'][1] = 1.2
    pv['p'][2] = -2.5
    pv['v'][0] = 0.5
    pv['v'][1] = 2.3
    pv['v'][2] = -0.4
    spv = erfa_ufunc.s2xpv(s1, s2, pv)
    vvd(spv['p'][0],  0.6, 1e-12, "s2xpv", "p1", status)
    vvd(spv['p'][1],  2.4, 1e-12, "s2xpv", "p2", status)
    vvd(spv['p'][2], -5.0, 1e-12, "s2xpv", "p3", status)
    vvd(spv['v'][0],  1.5, 1e-12, "s2xpv", "v1", status)
    vvd(spv['v'][1],  6.9, 1e-12, "s2xpv", "v2", status)
    vvd(spv['v'][2], -1.2, 1e-12, "s2xpv", "v3", status)


def test_sepp():
    a = np.empty((3,), float)
    b = np.empty((3,), float)
    a[0] = 1.0
    a[1] = 0.1
    a[2] = 0.2
    b[0] = -3.0
    b[1] = 1e-3
    b[2] = 0.2
    s = erfa_ufunc.sepp(a, b)
    vvd(s, 2.860391919024660768, 1e-12, "sepp", "", status)


def test_seps():
    al = 1.0
    ap = 0.1
    bl = 0.2
    bp = -3.0
    s = erfa_ufunc.seps(al, ap, bl, bp)
    vvd(s, 2.346722016996998842, 1e-14, "seps", "", status)


def test_sp00():
    vvd(erfa_ufunc.sp00(2400000.5, 52541.0), -0.6216698469981019309e-11, 1e-12, "sp00", "", status)


def test_starpm():
    ra1 =  0.01686756
    dec1 = -1.093989828
    pmr1 = -1.78323516e-5
    pmd1 = 2.336024047e-6
    px1 =  0.74723
    rv1 = -21.6
    ra2, dec2, pmr2, pmd2, px2, rv2, j = erfa_ufunc.starpm(ra1, dec1, pmr1, pmd1, px1, rv1, 2400000.5, 50083.0, 2400000.5, 53736.0)
    vvd(ra2, 0.01668919069414256149, 1e-13, "starpm", "ra", status)
    vvd(dec2, -1.093966454217127897, 1e-13, "starpm", "dec", status)
    vvd(pmr2, -0.1783662682153176524e-4, 1e-17, "starpm", "pmr", status)
    vvd(pmd2, 0.2338092915983989595e-5, 1e-17, "starpm", "pmd", status)
    vvd(px2, 0.7473533835317719243, 1e-13, "starpm", "px", status)
    vvd(rv2, -21.59905170476417175, 1e-11, "starpm", "rv", status)
    viv(j, 0, "starpm", "j", status)


def test_starpv():
    pv = np.empty((), erfa_ufunc.dt_pv)
    ra =  0.01686756
    dec = -1.093989828
    pmr = -1.78323516e-5
    pmd = 2.336024047e-6
    px =  0.74723
    rv = -21.6
    pv, j = erfa_ufunc.starpv(ra, dec, pmr, pmd, px, rv)
    vvd(pv['p'][0], 126668.5912743160601, 1e-10, "starpv", "11", status)
    vvd(pv['p'][1], 2136.792716839935195, 1e-12, "starpv", "12", status)
    vvd(pv['p'][2], -245251.2339876830091, 1e-10, "starpv", "13", status)
    vvd(pv['v'][0], -0.4051854008955659551e-2, 1e-13, "starpv", "21", status)
    vvd(pv['v'][1], -0.6253919754414777970e-2, 1e-15, "starpv", "22", status)
    vvd(pv['v'][2], 0.1189353714588109341e-1, 1e-13, "starpv", "23", status)
    viv(j, 0, "starpv", "j", status)


def test_sxp():
    p = np.empty((3,), float)
    sp = np.empty((3,), float)
    s = 2.0
    p[0] = 0.3
    p[1] = 1.2
    p[2] = -2.5
    sp = erfa_ufunc.sxp(s, p)
    vvd(sp[0],  0.6, 0.0, "sxp", "1", status)
    vvd(sp[1],  2.4, 0.0, "sxp", "2", status)
    vvd(sp[2], -5.0, 0.0, "sxp", "3", status)


def test_sxpv():
    pv = np.empty((), erfa_ufunc.dt_pv)
    spv = np.empty((), erfa_ufunc.dt_pv)
    s = 2.0
    pv['p'][0] = 0.3
    pv['p'][1] = 1.2
    pv['p'][2] = -2.5
    pv['v'][0] = 0.5
    pv['v'][1] = 3.2
    pv['v'][2] = -0.7
    spv = erfa_ufunc.sxpv(s, pv)
    vvd(spv['p'][0],  0.6, 0.0, "sxpv", "p1", status)
    vvd(spv['p'][1],  2.4, 0.0, "sxpv", "p2", status)
    vvd(spv['p'][2], -5.0, 0.0, "sxpv", "p3", status)
    vvd(spv['v'][0],  1.0, 0.0, "sxpv", "v1", status)
    vvd(spv['v'][1],  6.4, 0.0, "sxpv", "v2", status)
    vvd(spv['v'][2], -1.4, 0.0, "sxpv", "v3", status)


def test_taitt():
    t1, t2, j = erfa_ufunc.taitt(2453750.5, 0.892482639)
    vvd(t1, 2453750.5, 1e-6, "taitt", "t1", status)
    vvd(t2, 0.892855139, 1e-12, "taitt", "t2", status)
    viv(j, 0, "taitt", "j", status)


def test_taiut1():
    u1, u2, j = erfa_ufunc.taiut1(2453750.5, 0.892482639, -32.6659)
    vvd(u1, 2453750.5, 1e-6, "taiut1", "u1", status)
    vvd(u2, 0.8921045614537037037, 1e-12, "taiut1", "u2", status)
    viv(j, 0, "taiut1", "j", status)


def test_taiutc():
    u1, u2, j = erfa_ufunc.taiutc(2453750.5, 0.892482639)
    vvd(u1, 2453750.5, 1e-6, "taiutc", "u1", status)
    vvd(u2, 0.8921006945555555556, 1e-12, "taiutc", "u2", status)
    viv(j, 0, "taiutc", "j", status)


def test_tcbtdb():
    b1, b2, j = erfa_ufunc.tcbtdb(2453750.5, 0.893019599)
    vvd(b1, 2453750.5, 1e-6, "tcbtdb", "b1", status)
    vvd(b2, 0.8928551362746343397, 1e-12, "tcbtdb", "b2", status)
    viv(j, 0, "tcbtdb", "j", status)


def test_tcgtt():
    t1, t2, j = erfa_ufunc.tcgtt(2453750.5, 0.892862531)
    vvd(t1, 2453750.5, 1e-6, "tcgtt", "t1", status)
    vvd(t2, 0.8928551387488816828, 1e-12, "tcgtt", "t2", status)
    viv(j, 0, "tcgtt", "j", status)


def test_tdbtcb():
    b1, b2, j = erfa_ufunc.tdbtcb(2453750.5, 0.892855137)
    vvd( b1, 2453750.5, 1e-6, "tdbtcb", "b1", status)
    vvd( b2, 0.8930195997253656716, 1e-12, "tdbtcb", "b2", status)
    viv(j, 0, "tdbtcb", "j", status)


def test_tdbtt():
    t1, t2, j = erfa_ufunc.tdbtt(2453750.5, 0.892855137, -0.000201)
    vvd(t1, 2453750.5, 1e-6, "tdbtt", "t1", status)
    vvd(t2, 0.8928551393263888889, 1e-12, "tdbtt", "t2", status)
    viv(j, 0, "tdbtt", "j", status)


def test_tf2a():
    a, j = erfa_ufunc.tf2a('+', 4, 58, 20.2)
    vvd(a, 1.301739278189537429, 1e-12, "tf2a", "a", status)
    viv(j, 0, "tf2a", "j", status)


def test_tf2d():
    d, j = erfa_ufunc.tf2d(' ', 23, 55, 10.9)
    vvd(d, 0.9966539351851851852, 1e-12, "tf2d", "d", status)
    viv(j, 0, "tf2d", "j", status)


def test_tpors():
    xi = -0.03
    eta = 0.07
    ra = 1.3
    dec = 1.5
    az1, bz1, az2, bz2, n = erfa_ufunc.tpors(xi, eta, ra, dec)
    vvd(az1, 1.736621577783208748, 1e-13, "tpors", "az1", status)
    vvd(bz1, 1.436736561844090323, 1e-13, "tpors", "bz1", status)
    vvd(az2, 4.004971075806584490, 1e-13, "tpors", "az2", status)
    vvd(bz2, 1.565084088476417917, 1e-13, "tpors", "bz2", status)
    viv(n, 2, "tpors", "n", status)


def test_tporv():
    v = np.empty((3,), float)
    vz1 = np.empty((3,), float)
    vz2 = np.empty((3,), float)
    xi = -0.03
    eta = 0.07
    ra = 1.3
    dec = 1.5
    v = erfa_ufunc.s2c(ra, dec)
    vz1, vz2, n = erfa_ufunc.tporv(xi, eta, v)
    vvd(vz1[0], -0.02206252822366888610, 1e-15, "tporv", "x1", status)
    vvd(vz1[1], 0.1318251060359645016, 1e-14, "tporv", "y1", status)
    vvd(vz1[2], 0.9910274397144543895, 1e-14, "tporv", "z1", status)
    vvd(vz2[0], -0.003712211763801968173, 1e-16, "tporv", "x2", status)
    vvd(vz2[1], -0.004341519956299836813, 1e-16, "tporv", "y2", status)
    vvd(vz2[2], 0.9999836852110587012, 1e-14, "tporv", "z2", status)
    viv(n, 2, "tporv", "n", status)


def test_tpsts():
    xi = -0.03
    eta = 0.07
    raz = 2.3
    decz = 1.5
    ra, dec = erfa_ufunc.tpsts(xi, eta, raz, decz)
    vvd(ra, 0.7596127167359629775, 1e-14, "tpsts", "ra", status)
    vvd(dec, 1.540864645109263028, 1e-13, "tpsts", "dec", status)


def test_tpstv():
    vz = np.empty((3,), float)
    v = np.empty((3,), float)
    xi = -0.03
    eta = 0.07
    raz = 2.3
    decz = 1.5
    vz = erfa_ufunc.s2c(raz, decz)
    v = erfa_ufunc.tpstv(xi, eta, vz)
    vvd(v[0], 0.02170030454907376677, 1e-15, "tpstv", "x", status)
    vvd(v[1], 0.02060909590535367447, 1e-15, "tpstv", "y", status)
    vvd(v[2], 0.9995520806583523804, 1e-14, "tpstv", "z", status)


def test_tpxes():
    ra = 1.3
    dec = 1.55
    raz = 2.3
    decz = 1.5
    xi, eta, j = erfa_ufunc.tpxes(ra, dec, raz, decz)
    vvd(xi, -0.01753200983236980595, 1e-15, "tpxes", "xi", status)
    vvd(eta, 0.05962940005778712891, 1e-15, "tpxes", "eta", status)
    viv(j, 0, "tpxes", "j", status)


def test_tpxev():
    v = np.empty((3,), float)
    vz = np.empty((3,), float)
    ra = 1.3
    dec = 1.55
    raz = 2.3
    decz = 1.5
    v = erfa_ufunc.s2c(ra, dec)
    vz = erfa_ufunc.s2c(raz, decz)
    xi, eta, j = erfa_ufunc.tpxev(v, vz)
    vvd(xi, -0.01753200983236980595, 1e-15, "tpxev", "xi", status)
    vvd(eta, 0.05962940005778712891, 1e-15, "tpxev", "eta", status)
    viv(j, 0, "tpxev", "j", status)


def test_tr():
    r = np.empty((3, 3), float)
    rt = np.empty((3, 3), float)
    r[0][0] = 2.0
    r[0][1] = 3.0
    r[0][2] = 2.0
    r[1][0] = 3.0
    r[1][1] = 2.0
    r[1][2] = 3.0
    r[2][0] = 3.0
    r[2][1] = 4.0
    r[2][2] = 5.0
    rt = erfa_ufunc.tr(r)
    vvd(rt[0][0], 2.0, 0.0, "tr", "11", status)
    vvd(rt[0][1], 3.0, 0.0, "tr", "12", status)
    vvd(rt[0][2], 3.0, 0.0, "tr", "13", status)
    vvd(rt[1][0], 3.0, 0.0, "tr", "21", status)
    vvd(rt[1][1], 2.0, 0.0, "tr", "22", status)
    vvd(rt[1][2], 4.0, 0.0, "tr", "23", status)
    vvd(rt[2][0], 2.0, 0.0, "tr", "31", status)
    vvd(rt[2][1], 3.0, 0.0, "tr", "32", status)
    vvd(rt[2][2], 5.0, 0.0, "tr", "33", status)


def test_trxp():
    r = np.empty((3, 3), float)
    p = np.empty((3,), float)
    trp = np.empty((3,), float)
    r[0][0] = 2.0
    r[0][1] = 3.0
    r[0][2] = 2.0
    r[1][0] = 3.0
    r[1][1] = 2.0
    r[1][2] = 3.0
    r[2][0] = 3.0
    r[2][1] = 4.0
    r[2][2] = 5.0
    p[0] = 0.2
    p[1] = 1.5
    p[2] = 0.1
    trp = erfa_ufunc.trxp(r, p)
    vvd(trp[0], 5.2, 1e-12, "trxp", "1", status)
    vvd(trp[1], 4.0, 1e-12, "trxp", "2", status)
    vvd(trp[2], 5.4, 1e-12, "trxp", "3", status)


def test_trxpv():
    r = np.empty((3, 3), float)
    pv = np.empty((), erfa_ufunc.dt_pv)
    trpv = np.empty((), erfa_ufunc.dt_pv)
    r[0][0] = 2.0
    r[0][1] = 3.0
    r[0][2] = 2.0
    r[1][0] = 3.0
    r[1][1] = 2.0
    r[1][2] = 3.0
    r[2][0] = 3.0
    r[2][1] = 4.0
    r[2][2] = 5.0
    pv['p'][0] = 0.2
    pv['p'][1] = 1.5
    pv['p'][2] = 0.1
    pv['v'][0] = 1.5
    pv['v'][1] = 0.2
    pv['v'][2] = 0.1
    trpv = erfa_ufunc.trxpv(r, pv)
    vvd(trpv['p'][0], 5.2, 1e-12, "trxpv", "p1", status)
    vvd(trpv['p'][1], 4.0, 1e-12, "trxpv", "p1", status)
    vvd(trpv['p'][2], 5.4, 1e-12, "trxpv", "p1", status)
    vvd(trpv['v'][0], 3.9, 1e-12, "trxpv", "v1", status)
    vvd(trpv['v'][1], 5.3, 1e-12, "trxpv", "v2", status)
    vvd(trpv['v'][2], 4.1, 1e-12, "trxpv", "v3", status)


def test_tttai():
    a1, a2, j = erfa_ufunc.tttai(2453750.5, 0.892482639)
    vvd(a1, 2453750.5, 1e-6, "tttai", "a1", status)
    vvd(a2, 0.892110139, 1e-12, "tttai", "a2", status)
    viv(j, 0, "tttai", "j", status)


def test_tttcg():
    g1, g2, j = erfa_ufunc.tttcg(2453750.5, 0.892482639)
    vvd( g1, 2453750.5, 1e-6, "tttcg", "g1", status)
    vvd( g2, 0.8924900312508587113, 1e-12, "tttcg", "g2", status)
    viv(j, 0, "tttcg", "j", status)


def test_tttdb():
    b1, b2, j = erfa_ufunc.tttdb(2453750.5, 0.892855139, -0.000201)
    vvd(b1, 2453750.5, 1e-6, "tttdb", "b1", status)
    vvd(b2, 0.8928551366736111111, 1e-12, "tttdb", "b2", status)
    viv(j, 0, "tttdb", "j", status)


def test_ttut1():
    u1, u2, j = erfa_ufunc.ttut1(2453750.5, 0.892855139, 64.8499)
    vvd(u1, 2453750.5, 1e-6, "ttut1", "u1", status)
    vvd(u2, 0.8921045614537037037, 1e-12, "ttut1", "u2", status)
    viv(j, 0, "ttut1", "j", status)


def test_ut1tai():
    a1, a2, j = erfa_ufunc.ut1tai(2453750.5, 0.892104561, -32.6659)
    vvd(a1, 2453750.5, 1e-6, "ut1tai", "a1", status)
    vvd(a2, 0.8924826385462962963, 1e-12, "ut1tai", "a2", status)
    viv(j, 0, "ut1tai", "j", status)


def test_ut1tt():
    t1, t2, j = erfa_ufunc.ut1tt(2453750.5, 0.892104561, 64.8499)
    vvd(t1, 2453750.5, 1e-6, "ut1tt", "t1", status)
    vvd(t2, 0.8928551385462962963, 1e-12, "ut1tt", "t2", status)
    viv(j, 0, "ut1tt", "j", status)


def test_ut1utc():
    u1, u2, j = erfa_ufunc.ut1utc(2453750.5, 0.892104561, 0.3341)
    vvd(u1, 2453750.5, 1e-6, "ut1utc", "u1", status)
    vvd(u2, 0.8921006941018518519, 1e-12, "ut1utc", "u2", status)
    viv(j, 0, "ut1utc", "j", status)


def test_utctai():
    u1, u2, j = erfa_ufunc.utctai(2453750.5, 0.892100694)
    vvd(u1, 2453750.5, 1e-6, "utctai", "u1", status)
    vvd(u2, 0.8924826384444444444, 1e-12, "utctai", "u2", status)
    viv(j, 0, "utctai", "j", status)


def test_utcut1():
    u1, u2, j = erfa_ufunc.utcut1(2453750.5, 0.892100694, 0.3341)
    vvd(u1, 2453750.5, 1e-6, "utcut1", "u1", status)
    vvd(u2, 0.8921045608981481481, 1e-12, "utcut1", "u2", status)
    viv(j, 0, "utcut1", "j", status)


def test_xy06():
    x, y = erfa_ufunc.xy06(2400000.5, 53736.0)
    vvd(x, 0.5791308486706010975e-3, 1e-15, "xy06", "x", status)
    vvd(y, 0.4020579816732958141e-4, 1e-16, "xy06", "y", status)


def test_xys00a():
    x, y, s = erfa_ufunc.xys00a(2400000.5, 53736.0)
    vvd(x,  0.5791308472168152904e-3, 1e-14, "xys00a", "x", status)
    vvd(y,  0.4020595661591500259e-4, 1e-15, "xys00a", "y", status)
    vvd(s, -0.1220040848471549623e-7, 1e-18, "xys00a", "s", status)


def test_xys00b():
    x, y, s = erfa_ufunc.xys00b(2400000.5, 53736.0)
    vvd(x,  0.5791301929950208873e-3, 1e-14, "xys00b", "x", status)
    vvd(y,  0.4020553681373720832e-4, 1e-15, "xys00b", "y", status)
    vvd(s, -0.1220027377285083189e-7, 1e-18, "xys00b", "s", status)


def test_xys06a():
    x, y, s = erfa_ufunc.xys06a(2400000.5, 53736.0)
    vvd(x,  0.5791308482835292617e-3, 1e-14, "xys06a", "x", status)
    vvd(y,  0.4020580099454020310e-4, 1e-15, "xys06a", "y", status)
    vvd(s, -0.1220032294164579896e-7, 1e-18, "xys06a", "s", status)


@pytest.mark.xfail(reason='do not yet support no-input ufuncs')
def test_zp():
    p = np.empty((3,), float)
    p[0] = 0.3
    p[1] = 1.2
    p[2] = -2.5
    p = erfa_ufunc.zp()
    vvd(p[0], 0.0, 0.0, "zp", "1", status)
    vvd(p[1], 0.0, 0.0, "zp", "2", status)
    vvd(p[2], 0.0, 0.0, "zp", "3", status)


@pytest.mark.xfail(np.__version__ < '1.21', reason='needs numpy >= 1.21')
def test_zpv():
    pv = np.empty((), erfa_ufunc.dt_pv)
    pv['p'][0] = 0.3
    pv['p'][1] = 1.2
    pv['p'][2] = -2.5
    pv['v'][0] = -0.5
    pv['v'][1] = 3.1
    pv['v'][2] = 0.9
    pv = erfa_ufunc.zpv()
    vvd(pv['p'][0], 0.0, 0.0, "zpv", "p1", status)
    vvd(pv['p'][1], 0.0, 0.0, "zpv", "p2", status)
    vvd(pv['p'][2], 0.0, 0.0, "zpv", "p3", status)
    vvd(pv['v'][0], 0.0, 0.0, "zpv", "v1", status)
    vvd(pv['v'][1], 0.0, 0.0, "zpv", "v2", status)
    vvd(pv['v'][2], 0.0, 0.0, "zpv", "v3", status)


@pytest.mark.xfail(reason='do not yet support no-input ufuncs')
def test_zr():
    r = np.empty((3, 3), float)
    r[0][0] = 2.0
    r[1][0] = 3.0
    r[2][0] = 2.0
    r[0][1] = 3.0
    r[1][1] = 2.0
    r[2][1] = 3.0
    r[0][2] = 3.0
    r[1][2] = 4.0
    r[2][2] = 5.0
    r = erfa_ufunc.zr()
    vvd(r[0][0], 0.0, 0.0, "zr", "00", status)
    vvd(r[1][0], 0.0, 0.0, "zr", "01", status)
    vvd(r[2][0], 0.0, 0.0, "zr", "02", status)
    vvd(r[0][1], 0.0, 0.0, "zr", "10", status)
    vvd(r[1][1], 0.0, 0.0, "zr", "11", status)
    vvd(r[2][1], 0.0, 0.0, "zr", "12", status)
    vvd(r[0][2], 0.0, 0.0, "zr", "20", status)
    vvd(r[1][2], 0.0, 0.0, "zr", "21", status)
    vvd(r[2][2], 0.0, 0.0, "zr", "22", status)


def test_status():
    assert status == 0, "Sanity check failed!"
