% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/autoplot.R
\name{automatic_plotting}
\alias{automatic_plotting}
\title{Tailoring plots to particular data types}
\description{
There are three functions to make plotting particular data types easier:
\code{autoplot()}, \code{autolayer()} and \code{fortify()}. These are S3 generics for which
other packages can write methods to display classes of data. The three
functions are complementary and allow different levels of customisation.
Below we'll explore implementing this series of methods to automate plotting
of some class.

Let's suppose we are writing a packages that has a class called 'my_heatmap',
that wraps a matrix and we'd like users to easily plot this heatmap.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{my_heatmap <- function(...) \{
  m <- matrix(...)
  class(m) <- c("my_heatmap", class(m))
  m
\}

my_data <- my_heatmap(volcano)
}\if{html}{\out{</div>}}
}
\section{Automatic data shaping}{
One of the things we have to do is ensure that the data is shaped in the long
format so that it is compatible with ggplot2. This is the job of the
\code{fortify()} function. Because 'my_heatmap' wraps a matrix, we can let the
fortify method 'melt' the matrix to a long format. If your data is already
based on a long-format \verb{<data.frame>}, you can skip implementing a
\code{fortify()} method.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fortify.my_heatmap <- function(model, ...) \{
  data.frame(
    row = as.vector(row(model)),
    col = as.vector(col(model)),
    value = as.vector(model)
  )
\}

fortify(my_data)
}\if{html}{\out{</div>}}

When you have implemented the \code{fortify()} method, it should be easier to
construct a plot with the data than with the matrix.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{ggplot(my_data, aes(x = col, y = row, fill = value)) +
  geom_raster()
}\if{html}{\out{</div>}}
}

\section{Automatic layers}{
A next step in automating plotting of your data type is to write an
\code{autolayer()} method. These are typically wrappers around geoms or stats
that automatically set aesthetics or other parameters. If you haven't
implemented a \code{fortify()} method for your data type, you might have to
reshape the data in \code{autolayer()}.

If you require multiple layers to display your data type, you can use an
\code{autolayer()} method that constructs a list of layers, which can be added
to a plot.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{autolayer.my_heatmap <- function(object, ...) \{
  geom_raster(
    mapping = aes(x = col, y = row, fill = value),
    data = object,
    ...,
    inherit.aes = FALSE
  )
\}

ggplot() + autolayer(my_data)
}\if{html}{\out{</div>}}

As a quick tip: if you define a mapping in \code{autolayer()}, you might want
to set \code{inherit.aes = FALSE} to not have aesthetics set in other layers
interfere with your layer.
}

\section{Automatic plots}{
The last step in automating plotting is to write an \code{autoplot()} method
for your data type. The expectation is that these return a complete plot.
In the example below, we're exploiting the \code{autolayer()} method that we
have already written to make a complete plot.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{autoplot.my_heatmap <- function(object, ..., option = "magma") \{
  ggplot() +
    autolayer(my_data) +
    scale_fill_viridis_c(option = option) +
    theme_void()
\}

autoplot(my_data)
}\if{html}{\out{</div>}}

If you don't have a wish to implement a base R plotting method, you
can set the plot method for your class to the autoplot method.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{plot.my_heatmap <- autoplot.my_heatmap
plot(my_data)
}\if{html}{\out{</div>}}
}

\seealso{
Other plotting automation topics: 
\code{\link{autolayer}()},
\code{\link{autoplot}()},
\code{\link{fortify}()}
}
\concept{plotting automation topics}
