% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calcTOAIrradModel.R
\docType{methods}
\name{calcTOAIrradModel}
\alias{calcTOAIrradModel}
\alias{calcTOAIrradModel,Satellite-method}
\alias{calcTOAIrradModel,data.frame-method}
\title{Compute top of atmosphere solar irradiance for sensor bands using LUTs}
\usage{
\S4method{calcTOAIrradModel}{Satellite}(x, model = "MNewKur",
  normalize = TRUE, esd)

\S4method{calcTOAIrradModel}{data.frame}(x, model = "MNewKur",
  normalize = TRUE, esd)
}
\arguments{
\item{x}{A Satellite object or the relative spectral response function for 
the respective band as \code{data.frame} (see details for structure).}

\item{model}{Tabulated solar radiation model to be used (one of 
MCebKur_MChKur, MNewKur,  MthKur, MoldKur, MODWherli_WMO, NN, see reference
on tabulated solar irradiance below).}

\item{normalize}{Logical; if \code{TRUE}, ESun is normalized to mean 
earth-sun distance.}

\item{esd}{Earth-sun distance (AU, can be estimated using 
\code{\link{calcEarthSunDist}}). If x is a Satellite object and esd is not 
supplied and necessary for normalization, it is tried to take it from the 
metadata, otherwise it is estimated by the day of the year using 
\code{\link{calcEarthSunDist}}.}
}
\value{
If x is a Satellite object, a Satellite object with ESun information 
added to the metadata; if x is a \code{data.frame}, a vector containing ESun
for the respective band(s).
}
\description{
Compute mean extraterrestrial solar irradiance (ESun) using tabulated mean
solar spectral data and the band specific relative spectral response (rsr) 
functions.
}
\details{
Computation of ESun is taken from Updike and Comp (2011).

Tabulated values for mean earth-sun distance are taken from the 
data sources mentioned in the references. 

If results should not be normalized to a mean earth-sun distance, the 
actual earth-sun distance is approximated by the day of the year using
\code{\link{calcEarthSunDist}}.

Relative spectral response values have to be supplied as a \code{data.frame}
which has at least the following three columns: (i) a column "Band" for the 
sensor band number (i.e. 1, 2, etc.), (ii) a column "WAVELENGTH" for the 
WAVELENGTH data in full nm steps, and (iii) a column "RSR" for the response 
information [0...1].
}
\examples{
path <- system.file("extdata", package = "satellite")
files <- list.files(path, pattern = glob2rx("LC08*.TIF"), full.names = TRUE)
sat <- satellite(files)
sat <- calcTOAIrradModel(sat)
getSatESUN(sat)

lut <- lutInfo()
calcTOAIrradModel(lut$L8_RSR, model = "MNewKur", normalize = FALSE, 
  esd = calcEarthSunDist("2015-01-01"))

}
\references{
Updike T, Comp C (2011) Radiometric use of WorldView-2 imagery. 
Technical Note, available online at \url{http://www.pancroma.com/downloads/Radiometric_Use_of_WorldView-2_Imagery.pdf}.

Tabulated relative spectral response functions (nm-1) are taken from taken from the
\href{http://landsat.usgs.gov/instructions.php}{spectral viewer}
of the USGS Landsat FAQ.

Tabulated solar irradiance (W m-2 nm-1) is taken from the 
\href{http://rredc.nrel.gov/solar/spectra/am0/modtran.html}{National Renewable 
Energy Laboratory}.
}
\seealso{
\code{\link{calcTOAIrradTable}} for tabulated solar irradiance
values from the literature or \code{\link{calcTOAIrradRadRef}} for the 
computation of the solar irradiance based on maximum radiation and reflection
values of the dataset.

See \code{\link{calcEarthSunDist}} for calculating the earth-sun
distance based on the day of the year which is called by this function if
ESun should be corrected for actual earth-sun distance.
}
