/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "java/ClassDefinition.h"
#include "util/StringPiece.h"

#include <ostream>

namespace aapt {

bool ClassDefinition::empty() const {
    for (const std::unique_ptr<ClassMember>& member : mMembers) {
        if (!member->empty()) {
            return false;
        }
    }
    return true;
}

void ClassDefinition::writeToStream(const StringPiece& prefix, bool final,
                                    std::ostream* out) const {
    if (mMembers.empty() && !mCreateIfEmpty) {
        return;
    }

    ClassMember::writeToStream(prefix, final, out);

    *out << prefix << "public ";
    if (mQualifier == ClassQualifier::Static) {
        *out << "static ";
    }
    *out << "final class " << mName << " {\n";

    std::string newPrefix = prefix.toString();
    newPrefix.append(kIndent);

    for (const std::unique_ptr<ClassMember>& member : mMembers) {
        member->writeToStream(newPrefix, final, out);
        *out << "\n";
    }

    *out << prefix << "}";
}

constexpr static const char* sWarningHeader =
        "/* AUTO-GENERATED FILE. DO NOT MODIFY.\n"
        " *\n"
        " * This class was automatically generated by the\n"
        " * aapt tool from the resource data it found. It\n"
        " * should not be modified by hand.\n"
        " */\n\n";

bool ClassDefinition::writeJavaFile(const ClassDefinition* def,
                                    const StringPiece& package,
                                    bool final,
                                    std::ostream* out) {
    *out << sWarningHeader << "package " << package << ";\n\n";
    def->writeToStream("", final, out);
    return bool(*out);
}

} // namespace aapt
