/*
 * Copyright (C) - 2017 Francis Deslauriers <francis.deslauriers@efficios.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; version 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include <fcntl.h>
#include <stdio.h>
#include <sys/syscall.h>
#include <unistd.h>

#include "utils.h"

#define MAX_LEN 16
/*
 * The process waits for the creation of a file passed as argument from an
 * external processes to execute a syscall and exiting. This is useful for tests
 * in combinaison with LTTng's PID tracker feature where we can trace the kernel
 * events generated by our test process only.
 */
int main(int argc, char **argv)
{
	int fd, ret;
	char buf[MAX_LEN];
	char *start_file;

	if (argc != 2) {
		fprintf(stderr, "Error: Missing argument\n");
		fprintf(stderr, "USAGE: %s PATH_WAIT_FILE\n", argv[0]);
		ret = -1;
		goto error;
	}

	start_file = argv[1];

	/*
	 * Wait for the start_file to be created by an external process
	 * (typically the test script) before executing the syscalls.
	 */
	ret = wait_on_file(start_file);
	if (ret != 0) {
		goto error;
	}

	/*
	 * Start generating syscalls. We use syscall(2) to prevent libc to change
	 * the underlying syscall. e.g. calling openat(2) instead of open(2).
	 */
	fd = syscall(SYS_openat, AT_FDCWD, "/proc/cpuinfo", O_RDONLY);
	if (fd < 0) {
		perror("open");
		ret = -1;
		goto error;
	}

	ret = syscall(SYS_read, fd, buf, MAX_LEN);
	if (ret < 0) {
		perror("read");
		ret = -1;
		goto error;
	}

	ret = syscall(SYS_close, fd);
	if (ret == -1) {
		perror("close");
		ret = -1;
		goto error;
	}

error:
	return ret;
}
