/*
   libaio Linux async I/O interface
   Copyright 2018 Christoph Hellwig.
   Copyright 2024 Guillem Jover <guillem@hadrons.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */
#include <libaio.h>
#include <errno.h>
#include <stdlib.h>
#include <time.h>
#include <signal.h>
#include "syscall.h"
#include "aio_time.h"
#include "aio_ring.h"

/*
 * FIXME: On 64-bit kernels running 32-bit userland, this syscall misbehaves
 * when passing a sigmask by always returning -EINVAL. This works on 32-bit
 * kernels. The syscall entry point is possible wrong and needs to be fixed
 * in the kernel. For now disable as the code is handling clean 64-bit time_t
 * and the timeout is relative anyway.
 */
#undef __NR_io_pgetevents_time64

#ifdef __NR_io_pgetevents
io_syscall6(int, __io_pgetevents, io_pgetevents, io_context_t, ctx,
		long, min_nr, long, nr, struct io_event *, events,
		struct sys_timespec *, timeout, void *, sigmask);
#endif

#ifdef __NR_io_pgetevents_time64
io_syscall6(int, __io_pgetevents_time64, io_pgetevents_time64,
		io_context_t, ctx,
		long, min_nr, long, nr, struct io_event *, events,
		struct __kernel_timespec *, timeout, void *, sigmask);
#endif

int aio_pgetevents(io_context_t ctx, long min_nr, long nr,
		struct io_event *events, struct __kernel_timespec *timeout,
		sigset_t *sigmask)
{
	int ret;
#if defined(__NR_io_pgetevents) || defined(__NR_io_pgetevents_time64)
	struct io_sigset aio_sigset;
#ifndef __LP64__
	struct io_sigset_compat aio_sigset_compat = { 0 };
#endif
	struct sys_timespec sys_ts;

	if (aio_ring_is_empty(ctx, timeout))
		return 0;

	aio_sigset.ss = (unsigned long)sigmask;
	aio_sigset.ss_len = _NSIG / 8;
#endif

#ifdef __NR_io_pgetevents_time64
	/*
	 * On 32-bit systems that have a time64 variant of the syscall at
	 * compile-time we try to use that, and if missing we fallback to
	 * the 32-bit variant.
	 */
	ret = __io_pgetevents_time64(ctx, min_nr, nr, events, timeout,
			sigmask ? &aio_sigset : NULL);
	if (ret != -ENOSYS)
		return ret;
#endif

#ifdef __NR_io_pgetevents
	if (timeout)
		sys_get_timespec(&sys_ts, timeout);

	ret = __io_pgetevents(ctx, min_nr, nr, events,
			timeout ? &sys_ts : NULL, &aio_sigset);

#ifndef __LP64__
	/*
	 * The compat kernel syscall got introduced with a broken layout for
	 * its sigset argument, expecting it to contain a pointer for the
	 * non-compat pointer size.
	 *
	 * To cope with this on unfixed kernels, in case we are built as a
	 * 32-bit library (which could run on a kernel with compat code) and
	 * when the syscall returns EINVAL due to the kernel not finding the
	 * sigset size member when unpacking the structure, we retry with
	 * the fixed up compat layout, which requires the padding to be
	 * zero-filled, otherwise the 64-bit pointer will contain garbage.
	 */
	if (ret != -EINVAL)
		return ret;

	aio_sigset_compat.ss = (unsigned long)sigmask;
	aio_sigset_compat.ss_len = _NSIG / 8;
	ret = __io_pgetevents(ctx, min_nr, nr, events,
			timeout ? &sys_ts : NULL, &aio_sigset_compat);
#endif
#else
	ret = -ENOSYS;
#endif

	return ret;
}
