///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/basis_on_pointset.h"
namespace rheolef {

// -----------------------------------------------------------------------
// utility: local class:
// uniformize hat_node access for quadrature or Lagrange basis
// => allows code merge for basis_on_pointset
// -----------------------------------------------------------------------
template<class T>
class pointset_on_geo : public std::vector<point_basic<T> > {
  typedef std::vector<point_basic<T> > base;
public:
  pointset_on_geo ();
  pointset_on_geo (const quadrature<T>&  quad, reference_element hat_K);
  pointset_on_geo (const basis_basic<T>& nb,   reference_element hat_K);
  void initialize (const quadrature<T>&  quad, reference_element hat_K);
  void initialize (const basis_basic<T>& nb,   reference_element hat_K);
};
template<class T>
void
pointset_on_geo<T>::initialize (const quadrature<T>& quad, reference_element hat_K)
{
  base::resize (quad.size(hat_K));
  typename base::iterator dest = base::begin();
  for (typename quadrature<T>::const_iterator
      iter = quad.begin(hat_K),
      last = quad.end  (hat_K); iter != last; ++iter, ++dest) {
    *dest = (*iter).x;
  }
}
template<class T>
void
pointset_on_geo<T>::initialize (const basis_basic<T>& nb, reference_element hat_K)
{
  nb.hat_node (hat_K, *this);
}
template<class T>
pointset_on_geo<T>::pointset_on_geo ()
 : base ()
{
}
template<class T>
pointset_on_geo<T>::pointset_on_geo (const quadrature<T>& quad, reference_element hat_K)
 : base (quad.size(hat_K))
{
  initialize(quad, hat_K);
}
template<class T>
pointset_on_geo<T>::pointset_on_geo (const basis_basic<T>& nb, reference_element hat_K)
 : base ()
{
  initialize (nb, hat_K);
}
// ----------------------------------------------------------
// cstors
// ----------------------------------------------------------
template<class T>
basis_on_pointset<T>::basis_on_pointset ()
 : _b(),
   _mode(max_mode),
   _specific_value(false),
   _p_quad(0),
   _nb(),
   _basis_size(),
   _val(),
   _grad_val(),
   _initialized     (reference_element::max_variant, false),
   _grad_initialized(reference_element::max_variant, false),
   _curr_K_variant  (std::numeric_limits<size_type>::max()),
   _curr_q          (std::numeric_limits<size_type>::max())
{
}
template<class T>
basis_on_pointset<T>::basis_on_pointset (const basis_on_pointset<T>& x)
 : _b(x._b),
   _mode(x._mode),
   _specific_value(x._specific_value),
   _p_quad(x._p_quad),
   _nb(x._nb),
   _basis_size(x._basis_size),
   _val(x._val),
   _grad_val(x._grad_val),
   _initialized     (x._initialized),
   _grad_initialized(x._grad_initialized),
   _curr_K_variant  (x._curr_K_variant),
   _curr_q          (x._curr_q)
{
  trace_macro("PHYSICAL COPY***************");
}
template<class T>
basis_on_pointset<T>::basis_on_pointset (const quadrature<T>& quad, const basis_basic<T>& b)
 : _b(b),
   _mode(quad_mode),
   _specific_value(false),
   _p_quad(&quad),
   _nb(),
   _basis_size(),
   _val(),
   _grad_val(),
   _initialized     (reference_element::max_variant, false),
   _grad_initialized(reference_element::max_variant, false),
   _curr_K_variant (std::numeric_limits<size_type>::max()),
   _curr_q         (std::numeric_limits<size_type>::max())
{
}
template<class T>
basis_on_pointset<T>::basis_on_pointset (const basis_basic<T>& nb, const basis_basic<T>& b)
 : _b(b),
   _mode(nodal_mode),
   _specific_value(false),
   _p_quad(0),
   _nb(nb),
   _basis_size(),
   _val(),
   _grad_val(),
   _initialized     (reference_element::max_variant, false),
   _grad_initialized(reference_element::max_variant, false),
   _curr_K_variant (std::numeric_limits<size_type>::max()),
   _curr_q         (std::numeric_limits<size_type>::max())
{
}
template<class T>
void
basis_on_pointset<T>::set (const quadrature<T>& quad, const basis_basic<T>& b)
{
  // data was not defined on this quadrature formulae : reset
  _b  = b;
  _mode = quad_mode;
  _p_quad = &quad;
  std::fill (_initialized.begin(),      _initialized.end(),      false);
  std::fill (_grad_initialized.begin(), _grad_initialized.end(), false);
}
template<class T>
void
basis_on_pointset<T>::set (const basis_basic<T>& nb, const basis_basic<T>& b)
{
  // data was not defined on this quadrature formulae : reset
  _b  = b;
  _mode = nodal_mode;
  _nb = nb;
  std::fill (_initialized.begin(),      _initialized.end(),      false);
  std::fill (_grad_initialized.begin(), _grad_initialized.end(), false);
}
// -----------------------------------------------------------------------
// basis evaluated on lattice of quadrature formulae
// -----------------------------------------------------------------------
template<class T>
void 
basis_on_pointset<T>::_initialize (reference_element hat_K) const
{
  check_macro (!_specific_value, "unsupported eval in specific mode");
  reference_element::variant_type K_variant = hat_K.variant();
  _basis_size[K_variant] = _b.size (hat_K);
  pointset_on_geo<T> hat_node;
  if (_mode == quad_mode) hat_node.initialize (*_p_quad, hat_K);
  else                    hat_node.initialize (_nb,      hat_K);
  _val [K_variant].resize (_basis_size[K_variant]*hat_node.size());
  for (size_type q = 0, nq = hat_node.size(); q < nq; ++q) {
    _b.eval (hat_K, hat_node[q], _val[K_variant].begin() + _basis_size[K_variant]*q);
  }
  _initialized [K_variant] = true;
}
template<class T>
void 
basis_on_pointset<T>::_grad_initialize (reference_element hat_K) const
{
  check_macro (!_specific_value, "unsupported eval in specific mode");
  reference_element::variant_type K_variant = hat_K.variant();
  _basis_size[K_variant] = _b.size (hat_K);
  pointset_on_geo<T> hat_node;
  if (_mode == quad_mode) hat_node.initialize (*_p_quad, hat_K);
  else                    hat_node.initialize (_nb,      hat_K);
  _grad_val[K_variant].resize (_basis_size[K_variant]*hat_node.size());
  for (size_type q = 0, nq = hat_node.size(); q < nq; ++q) {
    _b.grad_eval (hat_K, hat_node[q], _grad_val[K_variant].begin() + _basis_size[K_variant]*q);
  }
  _grad_initialized [K_variant] = true;
}
// ----------------------------------------------------------
// side restriction (for DG)
// ----------------------------------------------------------
template<class T>
void
basis_on_pointset<T>::restrict_on_side (reference_element tilde_L, const side_information_type& sid)
{
  pointset_on_geo<T> hat_node;
  reference_element::variant_type L_variant = tilde_L.variant();
  _basis_size[L_variant] = _b.size (tilde_L);                      // evaluate basis for tilde_L
  if (_mode == quad_mode) hat_node.initialize (*_p_quad, sid.hat); // ...on the side tilde_K, at its quad nodes
  else                    hat_node.initialize (_nb,      sid.hat); // ...or its Lagrange nodes.
       _val[L_variant].resize (_basis_size[L_variant]*hat_node.size());
  _grad_val[L_variant].resize (_basis_size[L_variant]*hat_node.size());
  for (size_type q = 0, nq = hat_node.size(); q < nq; ++q) {
    // transform side hat_node on hat_K into tilde_node on tilde_K, the side of tilde_L:
    // => orient and shift varies for each side 
    // the transformed nodes from hat_K to tilde_K depends upon (loc_isid,orient,shift)
    // TODO: pre-compute for all (loc_isid,orient,shift) combinations (consider lazy version) 
    //       instead of recomputing it each time
    // TODO: split grad_eval and eval initializations
    point_basic<T> tilde_xq = reference_element_face_transformation (tilde_L, sid, hat_node[q]);
    _b.eval      (tilde_L, tilde_xq,      _val[L_variant].begin() + _basis_size[L_variant]*q);
    _b.grad_eval (tilde_L, tilde_xq, _grad_val[L_variant].begin() + _basis_size[L_variant]*q);
  }
       _initialized [L_variant] = true;
  _grad_initialized [L_variant] = true;
}
// ----------------------------------------------------------
// accessors
// ----------------------------------------------------------
template<class T>
typename basis_on_pointset<T>::size_type
basis_on_pointset<T>::size (reference_element hat_K) const
{
  if (_mode == quad_mode) return _p_quad -> size (hat_K);
  else                    return _nb.size (hat_K);
}
// -------------------
// all q-node eval
// -------------------
// basis eval
template<class T>
void
basis_on_pointset<T>::evaluate (reference_element hat_K, size_type q) const
{
  _specific_value = false;
  _curr_K_variant = hat_K.variant();
  _curr_q         = q;
  if (!_initialized [_curr_K_variant]) {
    _initialize (hat_K);
  }
}
// grad basis eval
template<class T>
void
basis_on_pointset<T>::evaluate_grad (reference_element hat_K, size_type q) const
{
  _specific_value = false;
  _curr_K_variant = hat_K.variant();
  _curr_q         = q;
  if (!_grad_initialized [_curr_K_variant]) {
    _grad_initialize (hat_K);
  }
}
// -------------------
// specific node eval
// -------------------
template<class T>
void
basis_on_pointset<T>::evaluate (reference_element hat_K, const point_basic<T>& hat_xq) const
{
  _specific_value = true;
  _curr_K_variant = hat_K.variant();
  _curr_q         = 0;
  _basis_size[_curr_K_variant] = _b.size (hat_K);
  if (_basis_size[_curr_K_variant] != _val[_curr_K_variant].size ()) {
    _val[_curr_K_variant].resize (_basis_size[_curr_K_variant]); 
  }
  _b.eval (hat_K, hat_xq, _val[_curr_K_variant]);
}
template<class T>
void
basis_on_pointset<T>::evaluate_grad (reference_element hat_K, const point_basic<T>& hat_xq) const
{
  _specific_value = true;
  _curr_K_variant = hat_K.variant();
  _curr_q         = 0;
  _basis_size[_curr_K_variant] = _b.size (hat_K);
  if (_basis_size[_curr_K_variant] != _grad_val[_curr_K_variant].size ()) {
    _grad_val[_curr_K_variant].resize (_basis_size[_curr_K_variant]); 
  }
  _b.grad_eval (hat_K, hat_xq, _grad_val[_curr_K_variant]);
}
// ----------------------------------------------------------
// here: could be inlined when code is stabilized
// ----------------------------------------------------------
// -------------------
// basis accessor
// -------------------
template<class T>
typename basis_on_pointset<T>::const_iterator
basis_on_pointset<T>::begin() const
{
  return _val[_curr_K_variant].begin() + _basis_size[_curr_K_variant]*_curr_q;
}
template<class T>
typename basis_on_pointset<T>::const_iterator
basis_on_pointset<T>::end() const
{
  return _val[_curr_K_variant].begin() + _basis_size[_curr_K_variant]*(_curr_q+1);
}
// -------------------
// grad basis accessor
// -------------------
template<class T>
typename basis_on_pointset<T>::const_iterator_grad
basis_on_pointset<T>::begin_grad() const
{
  return _grad_val[_curr_K_variant].begin() + _basis_size[_curr_K_variant]*_curr_q;
}
template<class T>
typename basis_on_pointset<T>::const_iterator_grad
basis_on_pointset<T>::end_grad() const
{
  return _grad_val[_curr_K_variant].begin() + _basis_size[_curr_K_variant]*(_curr_q+1);
}
// -------------------
// random accessors
// -------------------
template<class T>
const T&
basis_on_pointset<T>::value (size_type loc_idof) const
{
  // TODO: parano, clean:
  check_macro (_basis_size[_curr_K_variant]*_curr_q + loc_idof < _val[_curr_K_variant].size(),
	"index out of range");
  return _val[_curr_K_variant] [_basis_size[_curr_K_variant]*_curr_q + loc_idof];
}
template<class T>
const point_basic<T>&
basis_on_pointset<T>::grad_value (size_type loc_idof) const
{
  // TODO: parano, clean:
  check_macro (_basis_size[_curr_K_variant]*_curr_q + loc_idof < _grad_val[_curr_K_variant].size(),
	"index out of range");
  return _grad_val[_curr_K_variant] [_basis_size[_curr_K_variant]*_curr_q + loc_idof];
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T)       	\
template class basis_on_pointset<T>;

_RHEOLEF_instanciation(Float)

} // namespace rheolef
