/*  RipOff - Plugin based CD Ripper
 *  Copyright (C) 2006 Bobby Ryan Newberry
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <vorbis/vorbisenc.h>
#include "lib/RipOffPluginRawInterface.h"
#define READ 1024
#define UI_NAME "Ogg Vorbis Encoder v1.1"
signed char readbuffer[READ*4+44]; /* out of the data segment, not the stack */

struct VorbisPlugin_
{
	gchar *name;
	gchar *label;
	gdouble qualityrating;
	gdouble bitrate;
	gboolean using_bitrate;

	/* vorbis and ogg variables */
  	ogg_stream_state o_stream_state; /* take physical pages, weld into a logical
			  		  stream of packets */
  	ogg_page         o_page; /* one Ogg bitstream page.  Vorbis packets are inside */
  	ogg_packet       o_packet; /* one raw packet of data for decode */
  
  	vorbis_info      v_info; /* struct that stores all the static vorbis bitstream
			            settings */
  	vorbis_comment   v_comment; /* struct that stores all the user comments */

  	vorbis_dsp_state v_dsp_state; /* central working state for the packet->PCM decoder */
  	vorbis_block     v_block;     /* local working space for packet->PCM decode */
	ogg_packet header;
	ogg_packet header_comm;
	ogg_packet header_code;
};
typedef struct VorbisPlugin_ * VorbisPlugin;

struct helper_struct
{
	GtkWidget *quality_rating_hscale;
	GtkWidget *bitrate_hscale;
	GtkWidget *window;
	gboolean *using_bitrate;
};
	

void value_changed(GtkAdjustment *adjustment, gpointer data);
void quality_rating_toggled(	GtkToggleButton *button,
				gpointer data);
void bitrate_toggled(	GtkToggleButton *button,
			gpointer data);
void close_button_clicked(GtkButton *close_button,
			  gpointer data);
gboolean    destroy_handler(	GtkWidget *widget,
                                GdkEvent *event,
                                gpointer user_data);

RipOffPluginRaw ripoff_plugin_raw_new(xmlDocPtr ptr)
{
	VorbisPlugin raw = g_new(struct VorbisPlugin_, 1);
	ripoff_config_file_set_config_file_pointer(ptr);
	raw->name = UI_NAME;
	raw->label = "ripoff_default_VorbisPlugin";
	raw->qualityrating = ripoff_config_file_retrieve_double(raw->label, "qualityrating", 5.0);
	raw->bitrate = ripoff_config_file_retrieve_double(raw->label, "bitrate", 128.0);
	raw->using_bitrate = ripoff_config_file_retrieve_boolean(raw->label, "using_bitrate", FALSE);
	
	/* this function should take care of the stored preferences loading */
	return  (RipOffPluginRaw) raw;
}

gchar *ripoff_plugin_raw_get_name(RipOffPluginRaw raw)
{
	return raw->name;
}

gchar *ripoff_plugin_raw_get_label(RipOffPluginRaw raw)
{
	return raw->label;
}

const gchar *ripoff_plugin_raw_get_extension(RipOffPluginRaw raw)
{
	return "ogg";
}

gboolean ripoff_plugin_raw_has_prefs(RipOffPluginRaw raw)
{
	return TRUE;
}

GtkWidget *ripoff_plugin_raw_prefs(RipOffPluginRaw raw/*, GtkWidget *pref_window*/)
{
	GtkWidget *close_button;
	GtkWidget *quality_radio_button;
	GtkWidget *bitrate_radio_button;
	GtkWidget *window;
	GtkObject *quality_rating_adjustment;
	GtkObject *bitrate_adjustment;
	GtkWidget *quality_rating_hscale;
	GtkWidget *bitrate_hscale;
	GtkWidget *table;
	GtkTooltips *quality_tips;
	GtkTooltips *bitrate_tips;
	VorbisPlugin plugin = (VorbisPlugin) raw;
	struct helper_struct *helper;
	helper = g_new(struct helper_struct, 1);

	table = gtk_table_new(2, 3, FALSE);

	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_size_request (GTK_WIDGET (window), 400, 140);
	gtk_window_set_title (GTK_WINDOW (window), UI_NAME" Preferences");

	/* create quality rating adjustments and scales */
	quality_rating_adjustment = gtk_adjustment_new(	plugin->qualityrating,
							1.0,
							10.0,
							5.0,
							1.0,
							0.0); 

	quality_rating_hscale = gtk_hscale_new(GTK_ADJUSTMENT(quality_rating_adjustment));

	/* create bitrate adjustments and scales */
	bitrate_adjustment = gtk_adjustment_new(		plugin->bitrate,
							48.0,
							320.0,
							128.0,
							1.0,
							0.0); 
	bitrate_hscale = gtk_hscale_new(GTK_ADJUSTMENT(bitrate_adjustment));
	gtk_scale_set_digits(GTK_SCALE(bitrate_hscale), 0);

	/* helper struct setup */
	helper->using_bitrate = &(plugin->using_bitrate);
	helper->quality_rating_hscale = quality_rating_hscale;
	helper->bitrate_hscale = bitrate_hscale;
	helper->window = window;

	/* Creates quality setting radio button and scale bar*/
	quality_tips = gtk_tooltips_new();
	quality_radio_button = gtk_radio_button_new_with_label(NULL, "Vorbis Quality Rating:");
	gtk_tooltips_set_tip(quality_tips, quality_radio_button, "The quality rating allows Vorbis to vary the bitrate (instead of having a static bitrate as is specified with the bitrate setting) in order to achieve an ideal compression-to-quality ratio. 10 is the highest quality and 1 is the lowest. Recommended encoding option.", NULL);


	g_signal_connect(	quality_rating_adjustment,
				"value-changed",
				G_CALLBACK (value_changed),
				&(plugin->qualityrating));

	g_signal_connect(	quality_radio_button,
				"toggled",
				G_CALLBACK (quality_rating_toggled),
				helper);

	/* complete bitrate setups */
	bitrate_radio_button = gtk_radio_button_new_with_label(	gtk_radio_button_get_group(
								GTK_RADIO_BUTTON(quality_radio_button)), "Bitrate:");


	g_signal_connect(	bitrate_adjustment,
				"value-changed",
				G_CALLBACK (value_changed),
				&(plugin->bitrate));

	g_signal_connect(	bitrate_radio_button,
				"toggled",
				G_CALLBACK (bitrate_toggled),
				helper);

	bitrate_tips = gtk_tooltips_new();
	gtk_tooltips_set_tip(bitrate_tips, bitrate_radio_button, "Specifies a static bitrate for the OGG Vorbis file even when varying the bitrate would achieve a better compression-to-quality ration. NOT RECOMMENED.", NULL);


	/* close window button setup */
	close_button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);


	g_signal_connect (		G_OBJECT (close_button), 
					"clicked",
                             		G_CALLBACK (close_button_clicked), 
                              		helper);

	g_signal_connect	 (	G_OBJECT (window), 
				"delete_event",
                             	G_CALLBACK (destroy_handler), 
                                helper);

	if(plugin->using_bitrate)
	{
		gtk_widget_set_sensitive(quality_rating_hscale, FALSE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(bitrate_radio_button), TRUE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(quality_radio_button), FALSE);
	}
	else
	{
		gtk_widget_set_sensitive(bitrate_hscale, FALSE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(quality_radio_button), TRUE);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(bitrate_radio_button), FALSE);
	}

	/* table attachments */
	gtk_table_attach (	GTK_TABLE(table),
				quality_radio_button, 0, 1, 0, 1,
				GTK_FILL | GTK_EXPAND, 
				GTK_FILL | GTK_EXPAND,
				0, 5);

	gtk_table_attach (	GTK_TABLE(table),
				quality_rating_hscale, 1, 2, 0, 1,
				GTK_FILL | GTK_EXPAND,
				GTK_FILL | GTK_EXPAND,
				5, 5);

	gtk_table_attach (	GTK_TABLE(table),
				bitrate_radio_button, 0, 1, 1, 2,
				GTK_FILL | GTK_EXPAND, 
				GTK_FILL | GTK_EXPAND,
				0, 5);

	gtk_table_attach (	GTK_TABLE(table),
				bitrate_hscale, 1, 2, 1, 2,
				GTK_FILL | GTK_EXPAND, 
				GTK_FILL | GTK_EXPAND,
				5, 5);

	gtk_table_attach (	GTK_TABLE(table),
				close_button, 1, 2, 2, 3,
				GTK_SHRINK,
				GTK_SHRINK,
				0, 5);


	gtk_container_add(GTK_CONTAINER(window), table);

	return window;
}

gboolean ripoff_plugin_raw_has_about(RipOffPluginRaw raw)
{
	return TRUE;
}

GtkWidget *ripoff_plugin_raw_about(RipOffPluginRaw raw)
{
	return about_window_draw(	        "Ogg Vorbis Encoder", 
					        UI_NAME, 
						"Ogg Vorbis Encoding Plugin for RipOff",
						"Copyright © "YEAR" Bobby Ryan Newberry",
						WEBSITE);
}

gboolean ripoff_plugin_raw_perform_setup(	RipOffPluginRaw raw,
						long total_bytes_to_encode,
						FILE *output_descriptor,
						RipOffTrack track)
{	
	int ret;
	VorbisPlugin plugin = (VorbisPlugin) raw;
	vorbis_info_init(&(plugin->v_info));

	if(plugin->using_bitrate)
	{
		#ifdef DEBUG
		g_print("VorbisPlugin.c: Using a bitrate\n");
		#endif

		ret = vorbis_encode_init(&(plugin->v_info), 2, 44100, -1, (long) (plugin->bitrate * 1000), -1);
	}
	else
	{
		#ifdef DEBUG
		g_print("VorbisPlugin.c: Using a quality rating\n");
		#endif

		ret = vorbis_encode_init_vbr(&(plugin->v_info), 2, 44100, (plugin->qualityrating) / 10.0);
	}

	if(ret)
	{
		fprintf(stderr, "Simple Vorbis Plugin: Failure to initialize vorbis info\n");
		return FALSE;
	}

	vorbis_comment_init(&(plugin->v_comment));
	vorbis_comment_add_tag(&(plugin->v_comment),"ENCODER", NAME" "VERSION);
	vorbis_comment_add_tag(&(plugin->v_comment), "ARTIST",ripoff_track_get_artist(track));
	vorbis_comment_add_tag(&(plugin->v_comment), "GENRE", ripoff_track_get_genre(track));
	vorbis_comment_add_tag(&(plugin->v_comment), "ALBUM", ripoff_track_get_album_title(track));
	vorbis_comment_add_tag(&(plugin->v_comment), "TITLE", ripoff_track_get_track_title(track));
	vorbis_comment_add_tag(&(plugin->v_comment), "DATE", ripoff_track_get_year(track));
	vorbis_comment_add_tag(&(plugin->v_comment), "TRACKNUMBER", ripoff_track_get_track_num_string(track, TRUE));
	vorbis_comment_add_tag(&(plugin->v_comment), "COMMENT", "Ripped by "NAME" "WEBSITE);

	vorbis_analysis_init(&(plugin->v_dsp_state), &(plugin->v_info));
	vorbis_block_init(&(plugin->v_dsp_state), &(plugin->v_block));

	srand(time(NULL));
	ogg_stream_init(&(plugin->o_stream_state), rand());

	/* This ensures the actual
	 * audio data will start on a new page, as per spec
	 */
	vorbis_analysis_headerout(	&(plugin->v_dsp_state), 
					&(plugin->v_comment), 
					&(plugin->header), 
					&(plugin->header_comm), 
					&(plugin->header_code));

	ogg_stream_packetin(&(plugin->o_stream_state), &(plugin->header)); /* automatically placed in its own
					 		page */
	ogg_stream_packetin(&(plugin->o_stream_state), &(plugin->header_comm));
	ogg_stream_packetin(&(plugin->o_stream_state), &(plugin->header_code));

	while(1)
	{
		int result = ogg_stream_flush(&(plugin->o_stream_state), &(plugin->o_page));
		if(result == 0)
			break;

		fwrite((plugin->o_page).header, 1, (plugin->o_page).header_len, output_descriptor);
		fwrite((plugin->o_page).body, 1, (plugin->o_page).body_len, output_descriptor);
	}

	return TRUE;
}

gboolean ripoff_plugin_raw_encode_buffer(	RipOffPluginRaw raw,
						long total_bytes_to_encode,
						int16_t *audio_data,
						FILE *output_descriptor,
						RipOffTrack track)
{
	int eos = 0;
	long i;
	VorbisPlugin plugin = (VorbisPlugin) raw;

	memcpy(readbuffer, audio_data, CDIO_CD_FRAMESIZE_RAW);

	/* expose the buffer to submit data */
	float **buffer=vorbis_analysis_buffer(&(plugin->v_dsp_state), READ);
      
	/* uninterleave samples */
	for(i=0; i < CDIO_CD_FRAMESIZE_RAW / 4; i++)
	{
		buffer[0][i]=((readbuffer[ i * 4 + 1]<<8)|
			(0x00ff&(int)readbuffer[ i * 4])) / 32768.f;
		buffer[1][i]=((readbuffer[i*4+3]<<8)|
			(0x00ff&(int)readbuffer[i * 4 + 2])) / 32768.f;
	}
    
	/* tell the library how much we actually submitted */
	vorbis_analysis_wrote(&(plugin->v_dsp_state), i);

	/* vorbis does some data preanalysis, then divvies up blocks for
	more involved (potentially parallel) processing.  Get a single
	block for encoding now */

	while(vorbis_analysis_blockout(&(plugin->v_dsp_state), &(plugin->v_block)))
	{
		/* analysis, assume we want to use bitrate management */
		vorbis_analysis(&(plugin->v_block), NULL);
		vorbis_bitrate_addblock(&(plugin->v_block));

		while(vorbis_bitrate_flushpacket(&(plugin->v_dsp_state), &(plugin->o_packet)))
		{
			/* weld the packet into the bitstream */
			ogg_stream_packetin(&(plugin->o_stream_state), &(plugin->o_packet));
	
			/* write out pages (if any) */
			while(!eos)
			{
				int result = ogg_stream_pageout(&(plugin->o_stream_state), &(plugin->o_page));
					
				if(result == 0)
				{
					break;
				}
	  
				fwrite((plugin->o_page).header, 1, (plugin->o_page).header_len, output_descriptor);
	  			fwrite((plugin->o_page).body, 1, (plugin->o_page).body_len, output_descriptor);
	  
	  			/* this could be set above, but for illustrative purposes, I do
	     			it here (to show that vorbis does know where the stream ends) */
	  
	  			if(ogg_page_eos(&(plugin->o_page)))
				{
					eos = 1;
				}
			}
		}
	}

	return TRUE;
}

gboolean ripoff_plugin_raw_perform_cleanup(	RipOffPluginRaw raw,
						long total_bytes_to_encode,
						FILE *output_descriptor,
						RipOffTrack track)
{
	VorbisPlugin plugin = (VorbisPlugin) raw;

	while(1)
	{
		int result = ogg_stream_flush(&(plugin->o_stream_state), &(plugin->o_page));

		if(result == 0)
			break;

		fwrite((plugin->o_page).header, 1, (plugin->o_page).header_len, output_descriptor);
		fwrite((plugin->o_page).body, 1, (plugin->o_page).body_len, output_descriptor);
	}

	vorbis_analysis_wrote(&(plugin->v_dsp_state), 0);
	ogg_stream_clear(&(plugin->o_stream_state));
	vorbis_block_clear(&(plugin->v_block));
	vorbis_dsp_clear(&(plugin->v_dsp_state));
	vorbis_comment_clear(&(plugin->v_comment));
	vorbis_info_clear(&(plugin->v_info));

	return TRUE;
}

void ripoff_plugin_raw_close(RipOffPluginRaw raw)
{
	VorbisPlugin plugin = (VorbisPlugin) raw;
	ripoff_config_file_save_double(plugin->label, "qualityrating", plugin->qualityrating);
	ripoff_config_file_save_double(plugin->label, "bitrate", plugin->bitrate);
	ripoff_config_file_save_boolean(plugin->label, "using_bitrate", plugin->using_bitrate);
}

void value_changed(GtkAdjustment *adjustment, gpointer data)
{
	*((gdouble *)data) = gtk_adjustment_get_value(adjustment);
}

void quality_rating_toggled(	GtkToggleButton *button,
				gpointer data)
{
	struct helper_struct *helper = data;
	
	if(gtk_toggle_button_get_active(button))
	{
		*(helper->using_bitrate)	 = FALSE;
		gtk_widget_set_sensitive(helper->quality_rating_hscale, TRUE);
		gtk_widget_set_sensitive(helper->bitrate_hscale, FALSE);

	}
}

void bitrate_toggled(	GtkToggleButton *button,
			gpointer data)
{
	struct helper_struct *helper = data;
	
	if(gtk_toggle_button_get_active(button))
	{
		*(helper->using_bitrate)	 = TRUE;
		gtk_widget_set_sensitive(helper->bitrate_hscale, TRUE);
		gtk_widget_set_sensitive(helper->quality_rating_hscale, FALSE);
	}
}

gboolean    destroy_handler(	GtkWidget *widget,
                                GdkEvent *event,
                                gpointer data)
{
	gtk_widget_destroy(((struct helper_struct *)data)->window);
	g_free(data);
	return FALSE;
}

void close_button_clicked(GtkButton *close_button,
			  gpointer data)
{
	gtk_widget_destroy(((struct helper_struct *)data)->window);
	g_free(data);
}
