/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file WWidgets.h
    \author Romain BONDUE
    \date 31/03/06
    \warning GtkWidget are allocated and expected to be used, the wrapping
             object will NOT try to free it. */
#ifndef __WWIDGETS_H__
#define __WWIDGETS_H__

#include <string>
#include <vector>
#include <stdexcept>
#include <functional>

extern "C"{
#include <gtk/gtk.h>
}

#include "Parameters.h"
#include "UserData.h" // Only because of IPSetting_e.



namespace nsGUI
{
    template<class T>
    class CConvert : public std::unary_function<std::string, T>
    {
      public :
        std::string operator () (const T&) throw();

    }; // Convert


    template<class T, class C = CConvert<T> >
    class CComboBox
    {
      public :
        CComboBox () throw();
          /** \warning Both parameters must not be null (0). */
        CComboBox (const T*, unsigned) throw();
        const T& GetActive () const throw();
        unsigned GetActiveRow () const throw();
        void SetActive (const T&) throw();
        void SetElem (const T*, unsigned) throw();
        operator ::GtkWidget* () throw();
        operator const ::GtkWidget* () const throw();


      private :
        CComboBox& operator = (const CComboBox&) throw();
        CComboBox (const CComboBox&) throw();

        ::GtkComboBox* const m_pBox;
        unsigned m_TabSize;
        const T* m_Tab;

    }; // CComboBox


    class CModeConvert : public CConvert<nsWireless::Mode_e>
    {
      public :
        std::string operator () (const nsWireless::Mode_e&) throw();

    }; // CModeConvert
    typedef CComboBox<nsWireless::Mode_e, CModeConvert> CModeComboBox;


    typedef CComboBox<unsigned> CChannelComboBox;


    class CEncryptConvert : public CConvert<nsWireless::EncryptType_e>
    {
      public :
        std::string operator () (const nsWireless::EncryptType_e&) throw();

    }; // CEncryptConvert
    typedef CComboBox<nsWireless::EncryptType_e, CEncryptConvert>
                                                            CEncryptComboBox;


    class CAuthConvert : public CConvert<nsWireless::AuthType_e>
    {
      public :
        std::string operator () (const nsWireless::AuthType_e&) throw();

    }; // CAuthConvert
    typedef CComboBox<nsWireless::AuthType_e, CAuthConvert> CAuthComboBox;


    class CKeyEntry
    {
      public :
        CKeyEntry () throw();
        nsWireless::CHexaKey Get () const throw();
        void Set (const nsWireless::CHexaKey&) throw();
        void Clear () throw();
        void ASCIIOnly (bool) throw();
        bool ASCIIOnly () const throw();
        operator ::GtkWidget* () throw();
        operator const ::GtkWidget* () const throw();
        operator ::GtkEntry* () throw();
        operator const ::GtkEntry* () const throw();


      private :
        CKeyEntry& operator = (const CKeyEntry&) throw();
        CKeyEntry (const CKeyEntry&) throw();

        static void NewTextTyped (::GtkEditable*, char*, int, int*,
                                  CKeyEntry*) throw();
        static void ASCIIButtonToggled (::GtkToggleButton*, CKeyEntry*)
                                                                    throw();

        ::GtkEntry* const m_pEntry;
        ::GtkToggleButton* const m_pIsASCIIButton;
        ::GtkBox* const m_pBox;

    }; // CKeyEntry


    template <unsigned N> // N Must be at least 1.
    class CMultiKey
    {
      public :
        CMultiKey () throw();
        operator ::GtkWidget* () throw();
        operator const ::GtkWidget* () const throw();
            // Value between 0 and N - 1, it is *not* checked.
        nsWireless::CHexaKey Get (unsigned) const throw();
        void Set (const nsWireless::CHexaKey&, unsigned) throw();
        void AdditionnalKeyAvailable (bool) throw();
        unsigned GetNbKey () const throw();
        void Clear () throw();
            // Value between 0 and N - 1, it is *not* checked.
        CKeyEntry& GetKeyEntry (unsigned) throw();
            // Value between 0 and N - 1, it is *not* checked.
        const CKeyEntry& GetKeyEntry (unsigned) const throw();
        unsigned GetDefaultKey () const throw();
            // Value between 0 and N - 1, it is *not* checked.
        void SetDefaultKey (unsigned) throw();


      protected :
        void BuildUI () throw();


      private :
        CMultiKey (const CMultiKey&);
        CMultiKey& operator = (const CMultiKey&);

        ::GtkVBox* const m_pBox;
        ::GtkExpander* const m_pKeysExpander;
        ::GtkComboBox* const m_pDefaultKeyBox;
        CKeyEntry m_KeyEntryTab [N];

    }; // CMultiKey


    class CConfigDialog
    {
      public :
        CConfigDialog (const nsWireless::Mode_e*, unsigned,
                       const std::vector<nsWireless::CFreq>&,
                       const nsWireless::EncryptType_e*, unsigned,
                       const nsWireless::AuthType_e*, unsigned)
                                                        throw (std::bad_alloc);
        CConfigDialog (const nsWireless::Mode_e*, unsigned,
                       const nsWireless::EncryptType_e*, unsigned,
                       const nsWireless::AuthType_e*, unsigned)
                                                        throw (std::bad_alloc);
        virtual ~CConfigDialog () throw();

        bool Run (::GtkWindow*) throw();
        void Hide () throw();

        std::string GetConfigName () const throw();
        void SetConfigName (const std::string&) throw();

        std::string GetSSID () const throw();
        void SetSSID (const std::string&) throw();

        nsWireless::Mode_e GetMode () const throw();
        void SetMode (nsWireless::Mode_e) throw();

        unsigned GetChannel () const throw();
        void SetChannel (unsigned) throw();
        void SetAvailableChannel (const std::vector<nsWireless::CFreq>&)
                                                        throw (std::bad_alloc);

        nsWireless::EncryptType_e GetEncrypt () const throw();
        void SetEncrypt (nsWireless::EncryptType_e) throw();

        nsWireless::AuthType_e GetAuth () const throw();
        void SetAuth (nsWireless::AuthType_e) throw();

        unsigned GetNbKey () const throw();

        unsigned GetDefaultKey () const throw();
        void SetDefaultKey (unsigned) throw();

        /** \warning No bound check. */
        nsWireless::CHexaKey GetKey (unsigned) const throw();
        /** \warning No bound check. */
        void SetKey (const nsWireless::CHexaKey&, unsigned) throw();

        nsUserData::IPSetting_e GetIPSetting () const throw();
        void SetIPSetting (nsUserData::IPSetting_e) throw();

        operator ::GtkDialog* () throw();
        operator const ::GtkDialog* () const throw();


      protected :
        void BuildUI () throw();

        CModeComboBox& GetModeBox () throw();
        const CModeComboBox& GetModeBox () const throw();

        const CChannelComboBox& GetChannelBox () const throw();

        CEncryptComboBox& GetEncryptBox () throw();
        const CEncryptComboBox& GetEncryptBox () const throw();

        CAuthComboBox& GetAuthBox () throw();
        const CAuthComboBox& GetAuthBox () const throw();

        ::GtkEntry* GetNameEntry () throw();
        const ::GtkEntry* GetNameEntry () const throw();

        ::GtkEntry* GetSSIDEntry () throw();
        const ::GtkEntry* GetSSIDEntry () const throw();

        ::GtkBox* GetBox () throw();
        const ::GtkBox* GetBox () const throw();

        CMultiKey<4>& GetMultiKey () throw();
        const CMultiKey<4>& GetMultiKey () const throw();

        ::GtkRadioButton* GetIPNoneButton () throw();
        const ::GtkRadioButton* GetIPNoneButton () const throw();

        ::GtkRadioButton* GetIPDhcpButton () throw();
        const ::GtkRadioButton* GetIPDhcpButton () const throw();

        ::GtkRadioButton* GetIPStaticButton () throw();
        const ::GtkRadioButton* GetIPStaticButton () const throw();


      private :
        unsigned* m_ChannelTab;
        unsigned m_ChannelTabSize;

        ::GtkDialog* const m_pDialog;
        ::GtkEntry* const m_pNameEntry;
        ::GtkEntry* const m_pSSIDEntry;
        CModeComboBox m_ModeBox;
        CChannelComboBox m_ChannelBox;
        CEncryptComboBox m_EncryptBox;
        CAuthComboBox m_AuthBox;
        CMultiKey<4> m_MultiKey;

        ::GtkRadioButton* const m_pIPNoneButton;
        ::GtkRadioButton* const m_pIPDhcpButton;
        ::GtkRadioButton* const m_pIPStaticButton;

    }; // CConfigDialog


        // ? TODO ? Disable the channel box with some mode?
    class CSharedConfigDialog : public CConfigDialog
    {
      public :
          /* Auth [0] matches Enc1, Auth [1] matches Enc2 etc...
             TabAuthSize must be equal to the number of EncryptTab passed,
             which must be superior or equal to 1. */
        CSharedConfigDialog (const nsWireless::Mode_e*, unsigned,
                             const std::vector<nsWireless::CFreq>&,
                             const nsWireless::AuthType_e*, unsigned,
                             const nsWireless::EncryptType_e*, unsigned,
                             const nsWireless::EncryptType_e*, unsigned,
                             const nsWireless::EncryptType_e*, unsigned)
                                                        throw (std::bad_alloc);
        CSharedConfigDialog (const nsWireless::Mode_e*, unsigned,
                             const nsWireless::AuthType_e*, unsigned,
                             const nsWireless::EncryptType_e*, unsigned,
                             const nsWireless::EncryptType_e*, unsigned,
                             const nsWireless::EncryptType_e*, unsigned)
                                                        throw (std::bad_alloc);


      protected :
        void ClearKeyEntry () throw();


      private :
        const unsigned m_AuthTabSize;
        const nsWireless::AuthType_e* const m_AuthTab;

        const unsigned m_EncryptTabSize1;
        const nsWireless::EncryptType_e* const m_EncTab1;

        const unsigned m_EncryptTabSize2;
        const nsWireless::EncryptType_e* const m_EncTab2;

        const unsigned m_EncryptTabSize3;
        const nsWireless::EncryptType_e* const m_EncTab3;

        void SetEncryptBox (const nsWireless::EncryptType_e*, unsigned)
                                                                    throw();

        static void EncryptButtonCallBack (::GtkToggleButton*,
                                           CSharedConfigDialog*) throw();
        static void AuthButtonCallBack (::GtkToggleButton*,
                                        CSharedConfigDialog*) throw();

    }; // CSharedConfigDialog

} // namespace nsGUI

#include "private/WWidgets.hxx"

#endif // __WWIDGETS_H__
