use ui;
use core:sync;
use progvis:program;

/**
 * This class represents all data that exists for a program, i.e. that program's view of the world.
 *
 * In contrast to the things in the 'program' package, this is a snapshot of the program's state,
 * and not a picture that is continuously updated as the program executes. There is, however,
 * references back to the program so that the program may be easily manipulated at a later time.
 */
class World on Render {
	// Program we are representing a snapshot of.
	public Program program;

	// Lock for the traversal.
	private Lock traverseLock;

	// All allocations in this world. The order of these is not stable.
	private Allocation[] allocations;

	// Hints available to us.
	public ViewHints[] hints;

	// Errors to track.
	public Bool trackDeadlock;
	public MemoryChecks trackMemory;

	// Are thread allocations sticky?
	public Bool stickyThreads;

	// Create, and associate us with a program.
	init(progvis:program:Program program) {
		init {
			program = program;
			hints = program.hints;
			trackDeadlock = true;
			trackMemory = MemoryChecks:barriers;
			stickyThreads = true;
		}

		clear();
	}

	// Clear the state. This differs from restarting the program and calling update since clear
	// might remove lost objects without references (i.e. memory leaks).
	void clear() {
		program.reset();
		allocations.clear();

		var globals = program.findGlobals();
		if (globals.any) {
			allocations << StickyAllocation(unsafe:RawPtr(), Globals(globals));
		}
	}

	// Update the current snapshot with the current state of the running program.
	ProgramError? update() {
		Lock:Guard z(traverseLock);

		var t = beginTraversal();
		for (thread in program.sortedThreads) {
			traverse(t, thread);
		}
		endTraversal(t);
	}

	// Access the allocations in here.
	Array<Allocation>:Iter begin() { allocations.begin; }
	Array<Allocation>:Iter end() { allocations.end; }

	// Access threads.
	ThreadIter threads() {
		ThreadIter(allocations);
	}

	/**
	 * Special iterator for threads.
	 */
	class ThreadIter on Render {
		init(Allocation[] allocs) {
			init { allocs = allocs; pos = 0; }
		}

		ThreadIter iter() { this; }

		private Allocation[] allocs;
		private Nat pos;

		Thread? next() {
			while (pos < allocs.count) {
				if (t = allocs[pos++].data as Thread)
					return t;
			}
			null;
		}
	}

	// Get a summary of all the *data* in this world. The data is constructed so that the
	// representation is unique for this particular state, and so that all states that are
	// considered equal to this state have the same representation. This means that the summary
	// could be used to re-create the data, even though this is not the intention (i.e. the format
	// is not guaranteed to be stable through versions, nor easy to parse). Note that we are not
	// concerned about links to the running program, nor read/write states in this
	// representation. This is intended for automatically checking for data races or other problems.
	Str summary() {
		// We construct this by finding the globals and the threads and traversing them
		// recursively. Each time we find an object, we give it a number, so that we can record
		// pointers as numbers. It is crucial that the traversal order is stable here. For objects,
		// that is easy, as their representation is generally ordered (with the possible exception
		// of hash tables, which we don't support yet anyway). We do, however, need to be careful
		// about traversing threads in the proper order.

		// Note: This approach means that any data that is no longer reachable from either the
		// globals or a thread will not be present in the summary. This is actually not a problem
		// for us, as unreachable data is not interesting when finding concurrency issues. It could,
		// however, be a problem if we want to use this in other contexts.

		// Note: This approach means that the representation will be different if we start threads
		// in a different order. This will most likely not be too big of a problem in most cases,
		// but for programs that create many threads in a non-deterministic manner, this could
		// increase the search space quite a bit.

		StrBuf out;

		Data->Nat dataId;
		Nat->Nat threadId;

		Thread[] threads;
		Globals? globals;

		Registrator allocMap;

		for (a in allocations) {
			if (t = a.data as Thread) {
				// Ignore dead threads.
				if (t.alive & !t.crashed) {
					threadId.put(t.id, threads.count);
					threads << t;
				}
			} else if (g = a.data as Globals) {
				globals = g;
			}

			a.register(allocMap);
		}

		SummaryData data(allocMap.result, threadId);

		// Start with the globals if we have them.
		if (globals) {
			out << "-{";
			globals.summary(out, data);
			out << "}";
		}

		// Note: We don't need to sort threads by ID. Since we insert threads in a sorted order,
		// they will remain in that order (assuming thread id:s don't wrap around back to zero).
		// threads.sort((a, b) => a.src.threadId < b.src.threadId);

		// Output each thread in turn. We output sequential ID:s, so that we can merge cases where
		// one thread terminates and the other ones keep going. This is not perfect, as the order
		// threads were started in still matter, but not as much.
		for (i, t in threads) {
			out << i << "{";
			t.summary(out, data);
			out << "}";
		}

		out.toS;
	}

	// Find the name for a type.
	Str typeTitle(TypeInfo type) {
		for (h in hints) {
			if (r = h.typeTitle(this, type))
				return r;
		}
		"?";
	}

	// Convenience wrapper.
	Str typeTitle(core:lang:Type type) {
		typeTitle(TypeInfo(type, true, false));
	}

	/**
	 * Data used during object traversal.
	 */
	class Traversal on Render {
		// Owning World object.
		World owner;

		// Keep track of memory issues.
		MemCheck memory;

		// Map from pointers to index in 'allocations'.
		unsafe:RawPtr->Nat allocMap;

		// Map from pointers to data. Some language implementations may "hide" their by-value
		// semantics through references. Furthermore, this is the case for objects on the stack, as
		// they do not have an allocation per se.
		unsafe:RawPtr->Data dataMap;

		// Queue of remaining work during traversal.
		Queue<Allocation> work;

		// Init.
		init(World world) {
			init {
				owner = world;
				memory = create(world.trackMemory);
			}

			// Register all threads with the memory tracker.
			for (t in world.program.sortedThreads) {
				memory.add(t.threadId, t.memory);
			}
		}

		// Add special data for some object.
		void addData(unsafe:RawPtr object, Data data) {
			if (object.any)
				dataMap.put(object, data);
		}

		// Find data that a pointer refers to.
		Data? find(unsafe:RawPtr object) {
			// Heap allocations first.
			Nat id = allocMap.get(object, owner.allocations.count);
			if (id < owner.allocations.count) {
				// Found something. Update the allocation if that has not already been done.
				var alloc = owner.allocations[id];
				if (!alloc.visited) {
					alloc.visited = true;
					work.push(alloc);
				}
				return alloc.data;
			}

			// Check if it is a stack allocation, or some other hidden allocation.
			if (dataMap.has(object))
				return dataMap.get(object);

			// Else, we fail.
			return null;
		}

		// Find data, or create new data if it does not exist.
		Data findOrCreate(unsafe:RawPtr object) {
			if (data = find(object))
				return data;

			var alloc = createAlloc(object);

			Nat id = owner.allocations.count;
			owner.allocations.push(alloc);
			alloc.update(this, id);

			work.push(alloc);

			alloc.data;
		}

		// Create an allocation for an object.
		private Allocation createAlloc(unsafe:RawPtr object) {
			if (type = object.type) {
				TypeInfo info(type, true, false);

				for (h in owner.hints) {
					if (d = h.createData(this, info, object)) {
						return h.createAllocation(object, d);
					}
				}
			}

			return Allocation(object, Unknown());
		}

		// Create data for an object. This object is assumed to be embedded inside some other
		// larger data structure.
		Data create(TypeInfo type) {
			return create(type, unsafe:RawPtr());
		}

		// Create data suitable to show 'object', in the context of a global variable. This is
		// nearly identical to 'create' above, but it lets the ViewHints object treat globals
		// specially if it desires to do so.
		Data create(core:lang:GlobalVar global, TypeInfo type) {
			for (h in owner.hints) {
				if (d = h.createGlobal(this, type, global))
					return d;
			}

			return Unknown();
		}

		// Create data suitable to show 'object'. Creates the data in the context of a stack
		// allocation, and therefore discards references in some cases.
		Data createStack(unsafe:RawPtr object) {
			if (type = object.type) {
				TypeInfo info(type, object.isValue, false);
				if (!info.isValue) {
					StackPointer();
				} else {
					create(info, object);
				}
			} else {
				// Probably a pointer that is null at the moment.
				StackPointer();
			}
		}

		// Helper for createStack, like "create", but assumes a root allocation.
		private Data create(TypeInfo type, unsafe:RawPtr root) {
			for (h in owner.hints) {
				if (d = h.createData(this, type, root)) {
					return d;
				}
			}

			return Unknown();
		}

		// Find a title for a type.
		Str typeTitle(TypeInfo info) {
			owner.typeTitle(info);
		}
	}

private:

	// Begin traversal.
	Traversal beginTraversal() {
		Traversal t(this);

		for (i, a in allocations) {
			a.update(t, i);
			a.visited = false;
		}

		t;
	}

	// End traversal. Returns concurrency errors.
	ProgramError? endTraversal(Traversal t) {
		pop(t);

		// Find dead elements, and potentially traverse them now.
		for (a in allocations) {
			if (!a.visited) {
				if (!a.remove)
					a.traverse(t);
			}
		}

		pop(t);

		// Clean up remaining non-visited objects.
		Nat to = 0;
		for (Nat i = 0; i < allocations.count; i++) {
			Allocation here = allocations[i];
			if (to < i)
				allocations[to] = here;
			if (here.visited)
				to++;
		}
		while (to < allocations.count) {
			allocations.pop();
		}

		if (e = t.memory.error)
			return e;

		if (trackDeadlock)
			if (e = findDeadlock())
				return e;

		null;
	}

	// Traverse a thread.
	void traverse(Traversal t, progvis:program:ProgThread thread) {
		// Note: We don't call "traverse" here, we only put all threads on the work queue. That way,
		// we are sure that all threads will have had a chance to add data to the Traversal before
		// any other thread is traversed.

		var obj = unsafe:RawPtr(thread);
		Nat id = t.allocMap.get(obj, allocations.count);
		if (id >= allocations.count) {
			var data = Thread(thread);

			var alloc = if (stickyThreads) {
				StickyAllocation(obj, data);
			} else {
				Allocation(obj, data);
			};

			allocations.push(alloc);
			alloc.update(t, id);
		}

		// Traverse it.
		allocations[id].traverse(t);
	}

	// Empty the traversal queue.
	void pop(Traversal t) {
		while (t.work.any()) {
			var top = t.work.top();
			t.work.pop();
			top.traverse(t);
		}
	}

	// Find a deadlock.
	ProgramError? findDeadlock() {
		Bool anyThreads = false;
		Bool allSleeping = true;

		for (a in allocations) {
			if (t = a.data as Thread) {
				if (t.alive) {
					anyThreads = true;
					allSleeping &= t.sleeping;
				}
			}
		}

		if (anyThreads & allSleeping) {
			trackDeadlock = false;
			return DeadlockError();
		}

		null;
	}

}
