#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2016, Cristian García <cristian99garcia@gmail.com>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 3 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

from gettext import gettext as _


class Element:
    H, He, Li, Be, B, C, N, O, F, Ne, Na, Mg, Al, Si, P = list(range(1, 16))
    S, Cl, Ar, K, Ca, Sc, Ti, V, Cr, Mn, Fe, Co, Ni, Cu, Zn = list(range(16, 31))
    Ga, Ge, As, Se, Br, Kr, Rb, Sr, Y, Zr, Nb, Mo, Tc, Ru = list(range(31, 45))
    Rh, Pd, Ag, Cd, In, Sn, Sb, Te, I, Xe, Cs, Ba, La, Ce = list(range(45, 59))
    Pr, Nd, Pm, Sm, Eu, Gd, Tb, Dy, Ho, Er, Tm, Yb, Lu, Hf = list(range(59, 73))
    Ta, W, Re, Os, Ir, Pt, Au, Hg, Tl, Pb, Bi, Po, At, Rn = list(range(73, 87))
    Fr, Ra, Ac, Th, Pa, U, Np, Pu, Am, Cm, Bk, Cf, Es, Fm = list(range(87, 101))
    Md, No, Lr, Rf, Db, Sg, Bh, Hs, Mt, Ds, Rg, Cn, Uut = list(range(101, 114))
    Fl, Uup, Lv, Uus, Uuo = list(range(114, 119))


class Category:
    UNKNOWN               = 0
    DIATOMIC_NONMETAL     = 1
    ALKALI_METAL          = 2
    ALKALINE_EARTH_METAL  = 3
    TRANSITION_METAL      = 4
    LANTHANIDE            = 5
    ACTINIDE              = 6
    POST_TRANSITION_METAL = 7
    METALLOID             = 8
    NONMETAL              = 9
    NOBLE_GAS             = 10
    SYNTHETIC             = 11


class Phase:
    GAS = 0
    LIQUID = 1
    SOLID = 2


# Source: https://github.com/Bowserinator/Periodic-Table-JSON/blob/master/PeriodicTableJSON.json
ELEMENTS_DATA = {
    Element.H: {
        "name": _("Hydrogen"),
        "atomic-mass": 1.0079,
        "boil": 20.271,
        "category": Category.NONMETAL,
        "density": 0.08988,
        "discovered-by": "Henry Cavendish",
        "melt": 13.99,
        "molar-heat": 28.836,
        "named-by": "Antoine Lavoisier",
        "number": 1,
        "period": 1,
        "phase": Phase.GAS,
        "source": "https://en.wikipedia.org/wiki/Hydrogen",
        "summary": "Hydrogen is a chemical element with chemical symbol H and atomic number 1. With an atomic weight of 1.00794 u, hydrogen is the lightest element on the periodic table. Its monatomic form (H) is the most abundant chemical substance in the Universe, constituting roughly 75% of all baryonic mass.",
        "symbol": "H",
        "x": 1,
        "y": 1
    },
    Element.He: {
        "name": _("Helium"),
        "atomic-mass": 4.0026022,
        "boil": 4.222,
        "category": Category.NOBLE_GAS,
        "density": 0.1786,
        "discovered-by": "Pierre Janssen",
        "melt": 0.95,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 2,
        "period": 1,
        "phase": Phase.GAS,
        "source": "https://en.wikipedia.org/wiki/Helium",
        "summary": "Helium is a chemical element with symbol He and atomic number 2. It is a colorless, odorless, tasteless, non-toxic, inert, monatomic gas that heads the noble gas group in the periodic table. Its boiling and melting points are the lowest among all the elements.",
        "symbol": "He",
        "x": 18,
        "y": 1
    },
    Element.Li: {
        "name": _("Lithium"),
        "atomic-mass": 6.94,
        "boil": 1603,
        "category": Category.ALKALI_METAL,
        "density": 0.534,
        "discovered-by": "Johan August Arfwedson",
        "melt": 453.65,
        "molar-heat": 24.86,
        "named-by": _("Unknown"),
        "number": 3,
        "period": 2,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Lithium",
        "summary": "Lithium (from Greek:\u03bb\u03af\u03b8\u03bf\u03c2 lithos, \"stone\") is a chemical element with the symbol Li and atomic number 3. It is a soft, silver-white metal belonging to the alkali metal group of chemical elements. Under standard conditions it is the lightest metal and the least dense solid element.",
        "symbol": "Li",
        "x": 1,
        "y": 2
    },
    Element.Be: {
        "name": _("Beryllium"),
        "atomic-mass": 9.01218315,
        "boil": 2742,
        "category": Category.ALKALINE_EARTH_METAL,
        "density": 1.85,
        "discovered-by": "Louis Nicolas Vauquelin",
        "melt": 1560,
        "molar-heat": 16.443,
        "named-by": _("Unknown"),
        "number": 4,
        "period": 2,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Beryllium",
        "summary": "Beryllium is a chemical element with symbol Be and atomic number 4. It is created through stellar nucleosynthesis and is a relatively rare element in the universe. It is a divalent element which occurs naturally only in combination with other elements in minerals.",
        "symbol": "Be",
        "x": 2,
        "y": 2
    },
    Element.B: {
        "name": _("Boron"),
        "atomic-mass": 10.81,
        "boil": 4200,
        "category": Category.METALLOID,
        "density": 2.08,
        "discovered-by": "Joseph Louis Gay-Lussac",
        "melt": 2349,
        "molar-heat": 11.087,
        "named-by": _("Unknown"),
        "number": 5,
        "period": 2,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Boron",
        "summary": "Boron is a metalloid chemical element with symbol B and atomic number 5. Produced entirely by cosmic ray spallation and supernovae and not by stellar nucleosynthesis, it is a low-abundance element in both the Solar system and the Earth's crust. Boron is concentrated on Earth by the water-solubility of its more common naturally occurring compounds, the borate minerals.",
        "symbol": "B",
        "x": 13,
        "y": 2
    },
    Element.C: {
        "name": _("Carbon"),
        "atomic-mass": 12.011,
        "boil": None,
        "category": Category.NONMETAL,
        "density": 1.821,
        "discovered-by": "Ancient Egypt",
        "melt": None,
        "molar-heat": 8.517,
        "named-by": _("Unknown"),
        "number": 6,
        "period": 2,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Carbon",
        "summary": "Carbon (from Latin:carbo \"coal\") is a chemical element with symbol C and atomic number 6. On the periodic table, it is the first (row 2) of six elements in column (group) 14, which have in common the composition of their outer electron shell. It is nonmetallic and tetravalent\u2014making four electrons available to form covalent chemical bonds.",
        "symbol": "C",
        "x": 14,
        "y": 2
    },
    Element.N: {
        "name": _("Nitrogen"),
        "atomic-mass": 14.007,
        "boil": 77.355,
        "category": Category.NONMETAL,
        "density": 1.251,
        "discovered-by": "Daniel Rutherford",
        "melt": 63.15,
        "molar-heat": None,
        "named-by": "Jean-Antoine Chaptal",
        "number": 7,
        "period": 2,
        "phase": Phase.GAS,
        "source": "https://en.wikipedia.org/wiki/Nitrogen",
        "summary": "Nitrogen is a chemical element with symbol N and atomic number 7. It is the lightest pnictogen and at room temperature, it is a transparent, odorless diatomic gas. Nitrogen is a common element in the universe, estimated at about seventh in total abundance in the Milky Way and the Solar System.",
        "symbol": "N",
        "x": 15,
        "y": 2
    },
    Element.O: {
        "name": _("Oxygen"),
        "atomic-mass": 15.999,
        "boil": 90.188,
        "category": Category.NONMETAL,
        "density": 1.429,
        "discovered-by": "Carl Wilhelm Scheele",
        "melt": 54.36,
        "molar-heat": None,
        "named-by": "Antoine Lavoisier",
        "number": 8,
        "period": 2,
        "phase": Phase.GAS,
        "source": "https://en.wikipedia.org/wiki/Oxygen",
        "summary": "Oxygen is a chemical element with symbol O and atomic number 8. It is a member of the chalcogen group on the periodic table and is a highly reactive nonmetal and oxidizing agent that readily forms compounds (notably oxides) with most elements. By mass, oxygen is the third-most abundant element in the universe, after hydrogen and helium.",
        "symbol": "O",
        "x": 16,
        "y": 2
    },
    Element.F: {
        "name": _("Fluorine"),
        "atomic-mass": 18.9984031636,
        "boil": 85.03,
        "category": Category.DIATOMIC_NONMETAL,
        "density": 1.696,
        "discovered-by": "Andr\u00e9-Marie Amp\u00e8re",
        "melt": 53.48,
        "molar-heat": None,
        "named-by": "Humphry Davy",
        "number": 9,
        "period": 2,
        "phase": Phase.GAS,
        "source": "https://en.wikipedia.org/wiki/Fluorine",
        "summary": "Fluorine is a chemical element with symbol F and atomic number 9. It is the lightest halogen and exists as a highly toxic pale yellow diatomic gas at standard conditions. As the most electronegative element, it is extremely reactive:almost all other elements, including some noble gases, form compounds with fluorine.",
        "symbol": "F",
        "x": 17,
        "y": 2
    },
    Element.Ne: {
        "appearance": "colorless gas exhibiting an orange-red glow when placed in a high voltage electric field",
        "name": _("Neon"),
        "atomic-mass": 20.17976,
        "boil": 27.104,
        "category": Category.NOBLE_GAS,
        "density": 0.9002,
        "discovered-by": "Morris Travers",
        "melt": 24.56,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 10,
        "period": 2,
        "phase": Phase.GAS,
        "source": "https://en.wikipedia.org/wiki/Neon",
        "summary": "Neon is a chemical element with symbol Ne and atomic number 10. It is in group 18 (noble gases) of the periodic table. Neon is a colorless, odorless, inert monatomic gas under standard conditions, with about two-thirds the density of air.",
        "symbol": "Ne",
        "x": 18,
        "y": 2
    },
    Element.Na: {
        "appearance": "silvery white metallic",
        "name": _("Sodium"),
        "atomic-mass": 22.989769282,
        "boil": 1156.09,
        "category": Category.ALKALI_METAL,
        "density": 0.968,
        "discovered-by": "Humphry Davy",
        "melt": 370.944,
        "molar-heat": 28.23,
        "named-by": _("Unknown"),
        "number": 11,
        "period": 3,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Sodium",
        "summary": "Sodium /\u02c8so\u028adi\u0259m/ is a chemical element with symbol Na (from Ancient Greek \u039d\u03ac\u03c4\u03c1\u03b9\u03bf) and atomic number 11. It is a soft, silver-white, highly reactive metal. In the Periodic table it is in column 1 (alkali metals), and shares with the other six elements in that column that it has a single electron in its outer shell, which it readily donates, creating a positively charged atom - a cation.",
        "symbol": "Na",
        "x": 1,
        "y": 3
    },
    Element.Mg: {
        "appearance": "shiny grey solid",
        "name": _("Magnesium"),
        "atomic-mass": 24.305,
        "boil": 1363,
        "category": Category.ALKALINE_EARTH_METAL,
        "density": 1.738,
        "discovered-by": "Joseph Black",
        "melt": 923,
        "molar-heat": 24.869,
        "named-by": _("Unknown"),
        "number": 12,
        "period": 3,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Magnesium",
        "summary": "Magnesium is a chemical element with symbol Mg and atomic number 12. It is a shiny gray solid which bears a close physical resemblance to the other five elements in the second column (Group 2, or alkaline earth metals) of the periodic table:they each have the same electron configuration in their outer electron shell producing a similar crystal structure. Magnesium is the ninth most abundant element in the universe.",
        "symbol": "Mg",
        "x": 2,
        "y": 3
    },
    Element.Al: {
        "appearance": "silvery gray metallic",
        "name": _("Aluminium"),
        "atomic-mass": 26.98153857,
        "boil": 2743,
        "category": Category.POST_TRANSITION_METAL,
        "density": 2.7,
        "discovered-by": None,
        "melt": 933.47,
        "molar-heat": 24.2,
        "named-by": "Humphry Davy",
        "number": 13,
        "period": 3,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Aluminium",
        "summary": "Aluminium (or aluminum; see different endings) is a chemical element in the boron group with symbol Al and atomic number 13. It is a silvery-white, soft, nonmagnetic, ductile metal. Aluminium is the third most abundant element (after oxygen and silicon), and the most abundant metal, in the Earth's crust.",
        "symbol": "Al",
        "x": 13,
        "y": 3
    },
    Element.Si: {
        "appearance": "crystalline, reflective with bluish-tinged faces",
        "name": _("Silicon"),
        "atomic-mass": 28.085,
        "boil": 3538,
        "category": Category.METALLOID,
        "density": 2.329,
        "discovered-by": "J\u00f6ns Jacob Berzelius",
        "melt": 1687,
        "molar-heat": 19.789,
        "named-by": "Thomas Thomson (chemist)",
        "number": 14,
        "period": 3,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Silicon",
        "summary": "Silicon is a chemical element with symbol Si and atomic number 14. It is a tetravalent metalloid, more reactive than germanium, the metalloid directly below it in the table. Controversy about silicon's character dates to its discovery.",
        "symbol": "Si",
        "x": 14,
        "y": 3
    },
    Element.P: {
        "appearance": "colourless, waxy white, yellow, scarlet, red, violet, black",
        "name": _("Phosphorus"),
        "atomic-mass": 30.9737619985,
        "boil": None,
        "category": Category.NONMETAL,
        "density": None,
        "discovered-by": "Hennig Brand",
        "melt": None,
        "molar-heat": 23.824,
        "named-by": _("Unknown"),
        "number": 15,
        "period": 3,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Phosphorus",
        "summary": "Phosphorus is a chemical element with symbol P and atomic number 15. As an element, phosphorus exists in two major forms\u2014white phosphorus and red phosphorus\u2014but due to its high reactivity, phosphorus is never found as a free element on Earth. Instead phosphorus-containing minerals are almost always present in their maximally oxidised state, as inorganic phosphate rocks.",
        "symbol": "P",
        "x": 15,
        "y": 3
    },
    Element.S: {
        "appearance": "lemon yellow sintered microcrystals",
        "name": _("Sulfur"),
        "atomic-mass": 32.06,
        "boil": 717.8,
        "category": Category.NONMETAL,
        "density": 2.07,
        "discovered-by": "Ancient china",
        "melt": 388.36,
        "molar-heat": 22.75,
        "named-by": _("Unknown"),
        "number": 16,
        "period": 3,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Sulfur",
        "summary": "Sulfur or sulphur (see spelling differences) is a chemical element with symbol S and atomic number 16. It is an abundant, multivalent non-metal. Under normal conditions, sulfur atoms form cyclic octatomic molecules with chemical formula S8.",
        "symbol": "S",
        "x": 16,
        "y": 3
    },
    Element.Cl: {
        "appearance": "pale yellow-green gas",
        "name": _("Chlorine"),
        "atomic-mass": 35.45,
        "boil": 239.11,
        "category": Category.DIATOMIC_NONMETAL,
        "density": 3.2,
        "discovered-by": "Carl Wilhelm Scheele",
        "melt": 171.6,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 17,
        "period": 3,
        "phase": Phase.GAS,
        "source": "https://en.wikipedia.org/wiki/Chlorine",
        "summary": "Chlorine is a chemical element with symbol Cl and atomic number 17. It also has a relative atomic mass of 35.5. Chlorine is in the halogen group (17) and is the second lightest halogen following fluorine.",
        "symbol": "Cl",
        "x": 17,
        "y": 3
    },
    Element.Ar: {
        "appearance": "colorless gas exhibiting a lilac/violet glow when placed in a high voltage electric field",
        "name": _("Argon"),
        "atomic-mass": 39.9481,
        "boil": 87.302,
        "category": Category.NOBLE_GAS,
        "density": 1.784,
        "discovered-by": "Lord Rayleigh",
        "melt": 83.81,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 18,
        "period": 3,
        "phase": Phase.GAS,
        "source": "https://en.wikipedia.org/wiki/Argon",
        "summary": "Argon is a chemical element with symbol Ar and atomic number 18. It is in group 18 of the periodic table and is a noble gas. Argon is the third most common gas in the Earth's atmosphere, at 0.934% (9,340 ppmv), making it over twice as abundant as the next most common atmospheric gas, water vapor (which averages about 4000 ppmv, but varies greatly), and 23 times as abundant as the next most common non-condensing atmospheric gas, carbon dioxide (400 ppmv), and more than 500 times as abundant as the next most common noble gas, neon (18 ppmv).",
        "symbol": "Ar",
        "x": 18,
        "y": 3
    },
    Element.K: {
        "appearance": "silvery gray",
        "name": _("Potassium"),
        "atomic-mass": 39.09831,
        "boil": 1032,
        "category": Category.ALKALI_METAL,
        "density": 0.862,
        "discovered-by": "Humphry Davy",
        "melt": 336.7,
        "molar-heat": 29.6,
        "named-by": _("Unknown"),
        "number": 19,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Potassium",
        "summary": "Potassium is a chemical element with symbol K (derived from Neo-Latin, kalium) and atomic number 19. It was first isolated from potash, the ashes of plants, from which its name is derived. In the Periodic table, potassium is one of seven elements in column (group) 1 (alkali metals):they all have a single valence electron in their outer electron shell, which they readily give up to create an atom with a positive charge - a cation, and combine with anions to form salts.",
        "symbol": "K",
        "x": 1,
        "y": 4
    },
    Element.Ca: {
        "name": _("Calcium"),
        "atomic-mass": 40.0784,
        "boil": 1757,
        "category": Category.ALKALINE_EARTH_METAL,
        "density": 1.55,
        "discovered-by": "Humphry Davy",
        "melt": 1115,
        "molar-heat": 25.929,
        "named-by": _("Unknown"),
        "number": 20,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Calcium",
        "summary": "Calcium is a chemical element with symbol Ca and atomic number 20. Calcium is a soft gray alkaline earth metal, fifth-most-abundant element by mass in the Earth's crust. The ion Ca2+ is also the fifth-most-abundant dissolved ion in seawater by both molarity and mass, after sodium, chloride, magnesium, and sulfate.",
        "symbol": "Ca",
        "x": 2,
        "y": 4
    },
    Element.Sc: {
        "appearance": "silvery white",
        "name": _("Scandium"),
        "atomic-mass": 44.9559085,
        "boil": 3109,
        "category": Category.TRANSITION_METAL,
        "density": 2.985,
        "discovered-by": "Lars Fredrik Nilson",
        "melt": 1814,
        "molar-heat": 25.52,
        "named-by": _("Unknown"),
        "number": 21,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Scandium",
        "summary": "Scandium is a chemical element with symbol Sc and atomic number 21. A silvery-white metallic d-block element, it has historically been sometimes classified as a rare earth element, together with yttrium and the lanthanoids. It was discovered in 1879 by spectral analysis of the minerals euxenite and gadolinite from Scandinavia.",
        "symbol": "Sc",
        "x": 3,
        "y": 4
    },
    Element.Ti: {
        "appearance": "silvery grey-white metallic",
        "name": _("Titanium"),
        "atomic-mass": 47.8671,
        "boil": 3560,
        "category": Category.TRANSITION_METAL,
        "density": 4.506,
        "discovered-by": "William Gregor",
        "melt": 1941,
        "molar-heat": 25.06,
        "named-by": "Martin Heinrich Klaproth",
        "number": 22,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Titanium",
        "summary": "Titanium is a chemical element with symbol Ti and atomic number 22. It is a lustrous transition metal with a silver color, low density and high strength. It is highly resistant to corrosion in sea water, aqua regia and chlorine.",
        "symbol": "Ti",
        "x": 4,
        "y": 4
    },
    Element.V: {
        "appearance": "blue-silver-grey metal",
        "name": _("Vanadium"),
        "atomic-mass": 50.94151,
        "boil": 3680,
        "category": Category.TRANSITION_METAL,
        "density": 6.0,
        "discovered-by": "Andr\u00e9s Manuel del R\u00edo",
        "melt": 2183,
        "molar-heat": 24.89,
        "named-by": "Isotopes of vanadium",
        "number": 23,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Vanadium",
        "summary": "Vanadium is a chemical element with symbol V and atomic number 23. It is a hard, silvery grey, ductile and malleable transition metal. The element is found only in chemically combined form in nature, but once isolated artificially, the formation of an oxide layer stabilizes the free metal somewhat against further oxidation.",
        "symbol": "V",
        "x": 5,
        "y": 4
    },
    Element.Cr: {
        "appearance": "silvery metallic",
        "name": _("Chromium"),
        "atomic-mass": 51.99616,
        "boil": 2944,
        "category": Category.TRANSITION_METAL,
        "density": 7.19,
        "discovered-by": "Louis Nicolas Vauquelin",
        "melt": 2180,
        "molar-heat": 23.35,
        "named-by": _("Unknown"),
        "number": 24,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Chromium",
        "summary": "Chromium is a chemical element with symbol Cr and atomic number 24. It is the first element in Group 6. It is a steely-gray, lustrous, hard and brittle metal which takes a high polish, resists tarnishing, and has a high melting point.",
        "symbol": "Cr",
        "x": 6,
        "y": 4
    },
    Element.Mn: {
        "appearance": "silvery metallic",
        "name": _("Manganese"),
        "atomic-mass": 54.9380443,
        "boil": 2334,
        "category": Category.TRANSITION_METAL,
        "density": 7.21,
        "discovered-by": "Torbern Olof Bergman",
        "melt": 1519,
        "molar-heat": 26.32,
        "named-by": _("Unknown"),
        "number": 25,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Manganese",
        "summary": "Manganese is a chemical element with symbol Mn and atomic number 25. It is not found as a free element in nature; it is often found in combination with iron, and in many minerals. Manganese is a metal with important industrial metal alloy uses, particularly in stainless steels.",
        "symbol": "Mn",
        "x": 7,
        "y": 4
    },
    Element.Fe: {
        "appearance": "lustrous metallic with a grayish tinge",
        "name": _("Iron"),
        "atomic-mass": 55.8452,
        "boil": 3134,
        "category": Category.TRANSITION_METAL,
        "density": 7.874,
        "discovered-by": "5000 BC",
        "melt": 1811,
        "molar-heat": 25.1,
        "named-by": _("Unknown"),
        "number": 26,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Iron",
        "summary": "Iron is a chemical element with symbol Fe (from Latin:ferrum) and atomic number 26. It is a metal in the first transition series. It is by mass the most common element on Earth, forming much of Earth's outer and inner core.",
        "symbol": "Fe",
        "x": 8,
        "y": 4
    },
    Element.Co: {
        "appearance": "hard lustrous gray metal",
        "name": _("Cobalt"),
        "atomic-mass": 58.9331944,
        "boil": 3200,
        "category": Category.TRANSITION_METAL,
        "density": 8.9,
        "discovered-by": "Georg Brandt",
        "melt": 1768,
        "molar-heat": 24.81,
        "named-by": _("Unknown"),
        "number": 27,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Cobalt",
        "summary": "Cobalt is a chemical element with symbol Co and atomic number 27. Like nickel, cobalt in the Earth's crust is found only in chemically combined form, save for small deposits found in alloys of natural meteoric iron. The free element, produced by reductive smelting, is a hard, lustrous, silver-gray metal.",
        "symbol": "Co",
        "x": 9,
        "y": 4
    },
    Element.Ni: {
        "appearance": "lustrous, metallic, and silver with a gold tinge",
        "name": _("Nickel"),
        "atomic-mass": 58.69344,
        "boil": 3003,
        "category": Category.TRANSITION_METAL,
        "density": 8.908,
        "discovered-by": "Axel Fredrik Cronstedt",
        "melt": 1728,
        "molar-heat": 26.07,
        "named-by": _("Unknown"),
        "number": 28,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Nickel",
        "summary": "Nickel is a chemical element with symbol Ni and atomic number 28. It is a silvery-white lustrous metal with a slight golden tinge. Nickel belongs to the transition metals and is hard and ductile.",
        "symbol": "Ni",
        "x": 10,
        "y": 4
    },
    Element.Cu: {
        "appearance": "red-orange metallic luster",
        "name": _("Copper"),
        "atomic-mass": 63.5463,
        "boil": 2835,
        "category": Category.TRANSITION_METAL,
        "density": 8.96,
        "discovered-by": "Middle East",
        "melt": 1357.77,
        "molar-heat": 24.44,
        "named-by": _("Unknown"),
        "number": 29,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Copper",
        "summary": "Copper is a chemical element with symbol Cu (from Latin:cuprum) and atomic number 29. It is a soft, malleable and ductile metal with very high thermal and electrical conductivity. A freshly exed surface of pure copper has a reddish-orange color.",
        "symbol": "Cu",
        "x": 11,
        "y": 4
    },
    Element.Zn: {
        "appearance": "silver-gray",
        "name": _("Zinc"),
        "atomic-mass": 65.382,
        "boil": 1180,
        "category": Category.TRANSITION_METAL,
        "density": 7.14,
        "discovered-by": "India",
        "melt": 692.68,
        "molar-heat": 25.47,
        "named-by": _("Unknown"),
        "number": 30,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Zinc",
        "summary": "Zinc, in commerce also spelter, is a chemical element with symbol Zn and atomic number 30. It is the first element of group 12 of the periodic table. In some respects zinc is chemically similar to magnesium:its ion is of similar size and its only common oxidation state is +2.",
        "symbol": "Zn",
        "x": 12,
        "y": 4
    },
    Element.Ga: {
        "appearance": "silver-white",
        "name": _("Gallium"),
        "atomic-mass": 69.7231,
        "boil": 2673,
        "category": Category.POST_TRANSITION_METAL,
        "density": 5.91,
        "discovered-by": "Lecoq de Boisbaudran",
        "melt": 302.9146,
        "molar-heat": 25.86,
        "named-by": _("Unknown"),
        "number": 31,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Gallium",
        "summary": "Gallium is a chemical element with symbol Ga and atomic number 31. Elemental gallium does not occur in free form in nature, but as the gallium(III) compounds that are in trace amounts in zinc ores and in bauxite. Gallium is a soft, silvery metal, and elemental gallium is a brittle solid at low temperatures, and melts at 29.76 \u00b0C (85.57 \u00b0F) (slightly above room temperature).",
        "symbol": "Ga",
        "x": 13,
        "y": 4
    },
    Element.Ge: {
        "appearance": "grayish-white",
        "name": _("Germanium"),
        "atomic-mass": 72.6308,
        "boil": 3106,
        "category": Category.METALLOID,
        "density": 5.323,
        "discovered-by": "Clemens Winkler",
        "melt": 1211.4,
        "molar-heat": 23.222,
        "named-by": _("Unknown"),
        "number": 32,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Germanium",
        "summary": "Germanium is a chemical element with symbol Ge and atomic number 32. It is a lustrous, hard, grayish-white metalloid in the carbon group, chemically similar to its group neighbors tin and silicon. Purified germanium is a semiconductor, with an appearance most similar to elemental silicon.",
        "symbol": "Ge",
        "x": 14,
        "y": 4
    },
    Element.As: {
        "appearance": "metallic grey",
        "name": _("Arsenic"),
        "atomic-mass": 74.9215956,
        "boil": None,
        "category": Category.METALLOID,
        "density": 5.727,
        "discovered-by": "Bronze Age",
        "melt": None,
        "molar-heat": 24.64,
        "named-by": _("Unknown"),
        "number": 33,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Arsenic",
        "summary": "Arsenic is a chemical element with symbol As and atomic number 33. Arsenic occurs in many minerals, usually in conjunction with sulfur and metals, and also as a pure elemental crystal. Arsenic is a metalloid.",
        "symbol": "As",
        "x": 15,
        "y": 4
    },
    Element.Se: {
        "appearance": "black, red, and gray (not pictured) allotropes",
        "name": _("Selenium"),
        "atomic-mass": 78.9718,
        "boil": 958,
        "category": Category.NONMETAL,
        "density": 4.81,
        "discovered-by": "J\u00f6ns Jakob Berzelius",
        "melt": 494,
        "molar-heat": 25.363,
        "named-by": _("Unknown"),
        "number": 34,
        "period": 4,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Selenium",
        "summary": "Selenium is a chemical element with symbol Se and atomic number 34. It is a nonmetal with properties that are intermediate between those of its periodic table column-adjacent chalcogen elements sulfur and tellurium. It rarely occurs in its elemental state in nature, or as pure ore compounds.",
        "symbol": "Se",
        "x": 16,
        "y": 4
    },
    Element.Br: {
        "name": _("Bromine"),
        "atomic-mass": 79.904,
        "boil": 332.0,
        "category": Category.DIATOMIC_NONMETAL,
        "density": 23.1028,
        "discovered-by": "Antoine J\u00e9r\u00f4me Balard",
        "melt": 265.8,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 35,
        "period": 4,
        "phase": Phase.LIQUID,
        "source": "https://en.wikipedia.org/wiki/Bromine",
        "summary": "Bromine (from Ancient Greek:\u03b2\u03c1\u1ff6\u03bc\u03bf\u03c2, br\u00f3mos, meaning \"stench\") is a chemical element with symbol Br, and atomic number 35. It is a halogen. The element was isolated independently by two chemists, Carl Jacob L\u00f6wig and Antoine Jerome Balard, in 1825\u20131826.",
        "symbol": "Br",
        "x": 17,
        "y": 4
    },
    Element.Kr: {
        "appearance": "colorless gas, exhibiting a whitish glow in a high electric field",
        "name": _("Krypton"),
        "atomic-mass": 83.7982,
        "boil": 119.93,
        "category": Category.NOBLE_GAS,
        "density": 3.749,
        "discovered-by": "William Ramsay",
        "melt": 115.78,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 36,
        "period": 4,
        "phase": Phase.GAS,
        "source": "https://en.wikipedia.org/wiki/Krypton",
        "summary": "Krypton (from Greek:\u03ba\u03c1\u03c5\u03c0\u03c4\u03cc\u03c2 kryptos \"the hidden one\") is a chemical element with symbol Kr and atomic number 36. It is a member of group 18 (noble gases) elements. A colorless, odorless, tasteless noble gas, krypton occurs in trace amounts in the atmosphere, is isolated by fractionally distilling liquefied air, and is often used with other rare gases in fluorescent lamps.",
        "symbol": "Kr",
        "x": 18,
        "y": 4
    },
    Element.Rb: {
        "appearance": "grey white",
        "name": _("Rubidium"),
        "atomic-mass": 85.46783,
        "boil": 961,
        "category": Category.ALKALI_METAL,
        "density": 1.532,
        "discovered-by": "Robert Bunsen",
        "melt": 312.45,
        "molar-heat": 31.06,
        "named-by": _("Unknown"),
        "number": 37,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Rubidium",
        "summary": "Rubidium is a chemical element with symbol Rb and atomic number 37. Rubidium is a soft, silvery-white metallic element of the alkali metal group, with an atomic mass of 85.4678. Elemental rubidium is highly reactive, with properties similar to those of other alkali metals, such as very rapid oxidation in air.",
        "symbol": "Rb",
        "x": 1,
        "y": 5
    },
    Element.Sr: {
        "name": _("Strontium"),
        "atomic-mass": 87.621,
        "boil": 1650,
        "category": Category.ALKALINE_EARTH_METAL,
        "density": 2.64,
        "discovered-by": "William Cruickshank (chemist)",
        "melt": 1050,
        "molar-heat": 26.4,
        "named-by": _("Unknown"),
        "number": 38,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Strontium",
        "summary": "Strontium is a chemical element with symbol Sr and atomic number 38. An alkaline earth metal, strontium is a soft silver-white or yellowish metallic element that is highly reactive chemically. The metal turns yellow when it is exed to air.",
        "symbol": "Sr",
        "x": 2,
        "y": 5
    },
    Element.Y: {
        "appearance": "silvery white",
        "name": _("Yttrium"),
        "atomic-mass": 88.905842,
        "boil": 3203,
        "category": Category.TRANSITION_METAL,
        "density": 4.472,
        "discovered-by": "Johan Gadolin",
        "melt": 1799,
        "molar-heat": 26.53,
        "named-by": _("Unknown"),
        "number": 39,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Yttrium",
        "summary": "Yttrium is a chemical element with symbol Y and atomic number 39. It is a silvery-metallic transition metal chemically similar to the lanthanides and it has often been classified as a \"rare earth element\". Yttrium is almost always found combined with the lanthanides in rare earth minerals and is never found in nature as a free element.",
        "symbol": "Y",
        "x": 3,
        "y": 5
    },
    Element.Zr: {
        "appearance": "silvery white",
        "name": _("Zirconium"),
        "atomic-mass": 91.2242,
        "boil": 4650,
        "category": Category.TRANSITION_METAL,
        "density": 6.52,
        "discovered-by": "Martin Heinrich Klaproth",
        "melt": 2128,
        "molar-heat": 25.36,
        "named-by": _("Unknown"),
        "number": 40,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Zirconium",
        "summary": "Zirconium is a chemical element with symbol Zr and atomic number 40. The name of zirconium is taken from the name of the mineral zircon, the most important source of zirconium. The word zircon comes from the Persian word zargun \u0632\u0631\u06af\u0648\u0646, meaning \"gold-colored\".",
        "symbol": "Zr",
        "x": 4,
        "y": 5
    },
    Element.Nb: {
        "appearance": "gray metallic, bluish when oxidized",
        "name": _("Niobium"),
        "atomic-mass": 92.906372,
        "boil": 5017,
        "category": Category.TRANSITION_METAL,
        "density": 8.57,
        "discovered-by": "Charles Hatchett",
        "melt": 2750,
        "molar-heat": 24.6,
        "named-by": _("Unknown"),
        "number": 41,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Niobium",
        "summary": "Niobium, formerly columbium, is a chemical element with symbol Nb (formerly Cb) and atomic number 41. It is a soft, grey, ductile transition metal, which is often found in the pyrochlore mineral, the main commercial source for niobium, and columbite. The name comes from Greek mythology:Niobe, daughter of Tantalus since it is so similar to tantalum.",
        "symbol": "Nb",
        "x": 5,
        "y": 5
    },
    Element.Mo: {
        "appearance": "gray metallic",
        "name": _("Molybdenum"),
        "atomic-mass": 95.951,
        "boil": 4912,
        "category": Category.TRANSITION_METAL,
        "density": 10.28,
        "discovered-by": "Carl Wilhelm Scheele",
        "melt": 2896,
        "molar-heat": 24.06,
        "named-by": _("Unknown"),
        "number": 42,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Molybdenum",
        "summary": "Molybdenum is a chemical element with symbol Mo and atomic number 42. The name is from Neo-Latin molybdaenum, from Ancient Greek \u039c\u03cc\u03bb\u03c5\u03b2\u03b4\u03bf\u03c2 molybdos, meaning lead, since its ores were confused with lead ores. Molybdenum minerals have been known throughout history, but the element was discovered (in the sense of differentiating it as a new entity from the mineral salts of other metals) in 1778 by Carl Wilhelm Scheele.",
        "symbol": "Mo",
        "x": 6,
        "y": 5
    },
    Element.Tc: {
        "appearance": "shiny gray metal",
        "name": _("Technetium"),
        "atomic-mass": 98,
        "boil": 4538,
        "category": Category.TRANSITION_METAL,
        "density": 11,
        "discovered-by": "Emilio Segr\u00e8",
        "melt": 2430,
        "molar-heat": 24.27,
        "named-by": _("Unknown"),
        "number": 43,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Technetium",
        "summary": "Technetium (/t\u025bk\u02c8ni\u02d0\u0283i\u0259m/) is a chemical element with symbol Tc and atomic number 43. It is the element with the lowest atomic number in the periodic table that has no stable isotopes:every form of it is radioactive. Nearly all technetium is produced synthetically, and only minute amounts are found in nature.",
        "symbol": "Tc",
        "x": 7,
        "y": 5
    },
    Element.Ru: {
        "appearance": "silvery white metallic",
        "name": _("Ruthenium"),
        "atomic-mass": 101.072,
        "boil": 4423,
        "category": Category.TRANSITION_METAL,
        "density": 12.45,
        "discovered-by": "Karl Ernst Claus",
        "melt": 2607,
        "molar-heat": 24.06,
        "named-by": _("Unknown"),
        "number": 44,
        "period": 5,
        "phase": None,
        "source": "https://en.wikipedia.org/wiki/Ruthenium",
        "summary": "Ruthenium is a chemical element with symbol Ru and atomic number 44. It is a rare transition metal belonging to the platinum group of the periodic table. Like the other metals of the platinum group, ruthenium is inert to most other chemicals.",
        "symbol": "Ru",
        "x": 8,
        "y": 5
    },
    Element.Rh: {
        "appearance": "silvery white metallic",
        "name": _("Rhodium"),
        "atomic-mass": 102.905502,
        "boil": 3968,
        "category": Category.TRANSITION_METAL,
        "density": 12.41,
        "discovered-by": "William Hyde Wollaston",
        "melt": 2237,
        "molar-heat": 24.98,
        "named-by": _("Unknown"),
        "number": 45,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Rhodium",
        "summary": "Rhodium is a chemical element with symbol Rh and atomic number 45. It is a rare, silvery-white, hard, and chemically inert transition metal. It is a member of the platinum group.",
        "symbol": "Rh",
        "x": 9,
        "y": 5
    },
    Element.Pd: {
        "appearance": "silvery white",
        "name": _("Palladium"),
        "atomic-mass": 106.421,
        "boil": 3236,
        "category": Category.TRANSITION_METAL,
        "density": 12.023,
        "discovered-by": "William Hyde Wollaston",
        "melt": 1828.05,
        "molar-heat": 25.98,
        "named-by": _("Unknown"),
        "number": 46,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Palladium",
        "summary": "Palladium is a chemical element with symbol Pd and atomic number 46. It is a rare and lustrous silvery-white metal discovered in 1803 by William Hyde Wollaston. He named it after the asteroid Pallas, which was itself named after the epithet of the Greek goddess Athena, acquired by her when she slew Pallas.",
        "symbol": "Pd",
        "x": 10,
        "y": 5
    },
    Element.Ag: {
        "appearance": "lustrous white metal",
        "name": _("Silver"),
        "atomic-mass": 107.86822,
        "boil": 2435,
        "category": Category.TRANSITION_METAL,
        "density": 10.49,
        "discovered-by": "unknown, before 5000 BC",
        "melt": 1234.93,
        "molar-heat": 25.35,
        "named-by": _("Unknown"),
        "number": 47,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Silver",
        "summary": "Silver is a chemical element with symbol Ag (Greek:\u03ac\u03c1\u03b3\u03c5\u03c1\u03bf\u03c2 \u00e1rguros, Latin:argentum, both from the Indo-European root *h\u2082er\u01f5- for \"grey\" or \"shining\") and atomic number 47. A soft, white, lustrous transition metal, it possesses the highest electrical conductivity, thermal conductivity and reflectivity of any metal. The metal occurs naturally in its pure, free form (native silver), as an alloy with gold and other metals, and in minerals such as argentite and chlorargyrite.",
        "symbol": "Ag",
        "x": 11,
        "y": 5
    },
    Element.Cd: {
        "appearance": "silvery bluish-gray metallic",
        "name": _("Cadmium"),
        "atomic-mass": 112.4144,
        "boil": 1040,
        "category": Category.TRANSITION_METAL,
        "density": 8.65,
        "discovered-by": "Karl Samuel Leberecht Hermann",
        "melt": 594.22,
        "molar-heat": 26.02,
        "named-by": "Isotopes of cadmium",
        "number": 48,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Cadmium",
        "summary": "Cadmium is a chemical element with symbol Cd and atomic number 48. This soft, bluish-white metal is chemically similar to the two other stable metals in group 12, zinc and mercury. Like zinc, it prefers oxidation state +2 in most of its compounds and like mercury it shows a low melting point compared to transition metals.",
        "symbol": "Cd",
        "x": 12,
        "y": 5
    },
    Element.In: {
        "appearance": "silvery lustrous gray",
        "name": _("Indium"),
        "atomic-mass": 114.8181,
        "boil": 2345,
        "category": Category.POST_TRANSITION_METAL,
        "density": 7.31,
        "discovered-by": "Ferdinand Reich",
        "melt": 429.7485,
        "molar-heat": 26.74,
        "named-by": _("Unknown"),
        "number": 49,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Indium",
        "summary": "Indium is a chemical element with symbol In and atomic number 49. It is a post-transition metallic element that is rare in Earth's crust. The metal is very soft, malleable and easily fusible, with a melting point higher than sodium, but lower than lithium or tin.",
        "symbol": "In",
        "x": 13,
        "y": 5
    },
    Element.Sn: {
        "appearance": "silvery-white (beta, \u03b2) or gray (alpha, \u03b1)",
        "name": _("Tin"),
        "atomic-mass": 118.7107,
        "boil": 2875,
        "category": Category.POST_TRANSITION_METAL,
        "density": 7.365,
        "discovered-by": "unknown, before 3500 BC",
        "melt": 505.08,
        "molar-heat": 27.112,
        "named-by": _("Unknown"),
        "number": 50,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Tin",
        "summary": "Tin is a chemical element with the symbol Sn (for Latin:stannum) and atomic number 50. It is a main group metal in group 14 of the periodic table. Tin shows a chemical similarity to both neighboring group-14 elements, germanium and lead, and has two possible oxidation states, +2 and the slightly more stable +4.",
        "symbol": "Sn",
        "x": 14,
        "y": 5
    },
    Element.Sb: {
        "appearance": "silvery lustrous gray",
        "name": _("Antimony"),
        "atomic-mass": 121.7601,
        "boil": 1908,
        "category": Category.METALLOID,
        "density": 6.697,
        "discovered-by": "unknown, before 3000 BC",
        "melt": 903.78,
        "molar-heat": 25.23,
        "named-by": _("Unknown"),
        "number": 51,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Antimony",
        "summary": "Antimony is a chemical element with symbol Sb (from Latin:stibium) and atomic number 51. A lustrous gray metalloid, it is found in nature mainly as the sulfide mineral stibnite (Sb2S3). Antimony compounds have been known since ancient times and were used for cosmetics; metallic antimony was also known, but it was erroneously identified as lead upon its discovery.",
        "symbol": "Sb",
        "x": 15,
        "y": 5
    },
    Element.Te: {
        "name": _("Tellurium"),
        "atomic-mass": 127.603,
        "boil": 1261,
        "category": Category.METALLOID,
        "density": 6.24,
        "discovered-by": "Franz-Joseph M\u00fcller von Reichenstein",
        "melt": 722.66,
        "molar-heat": 25.73,
        "named-by": _("Unknown"),
        "number": 52,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Tellurium",
        "summary": "Tellurium is a chemical element with symbol Te and atomic number 52. It is a brittle, mildly toxic, rare, silver-white metalloid. Tellurium is chemically related to selenium and sulfur.",
        "symbol": "Te",
        "x": 16,
        "y": 5
    },
    Element.I: {
        "appearance": "lustrous metallic gray, violet as a gas",
        "name": _("Iodine"),
        "atomic-mass": 126.904473,
        "boil": 457.4,
        "category": Category.DIATOMIC_NONMETAL,
        "density": 4.933,
        "discovered-by": "Bernard Courtois",
        "melt": 386.85,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 53,
        "period": 5,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Iodine",
        "summary": "Iodine is a chemical element with symbol I and atomic number 53. The name is from Greek \u1f30\u03bf\u03b5\u03b9\u03b4\u03ae\u03c2 ioeid\u0113s, meaning violet or purple, due to the color of iodine vapor. Iodine and its compounds are primarily used in nutrition, and industrially in the production of acetic acid and certain polymers.",
        "symbol": "I",
        "x": 17,
        "y": 5
    },
    Element.Xe: {
        "appearance": "colorless gas, exhibiting a blue glow when placed in a high voltage electric field",
        "name": _("Xenon"),
        "atomic-mass": 131.2936,
        "boil": 165.051,
        "category": Category.NOBLE_GAS,
        "density": 5.894,
        "discovered-by": "William Ramsay",
        "melt": 161.4,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 54,
        "period": 5,
        "phase": Phase.GAS,
        "source": "https://en.wikipedia.org/wiki/Xenon",
        "summary": "Xenon is a chemical element with symbol Xe and atomic number 54. It is a colorless, dense, odorless noble gas, that occurs in the Earth's atmosphere in trace amounts. Although generally unreactive, xenon can undergo a few chemical reactions such as the formation of xenon hexafluoroplatinate, the first noble gas compound to be synthesized.",
        "symbol": "Xe",
        "x": 18,
        "y": 5
    },
    Element.Cs: {
        "appearance": "silvery gold",
        "name": _("Cesium"),
        "atomic-mass": 132.905451966,
        "boil": 944,
        "category": Category.ALKALI_METAL,
        "density": 1.93,
        "discovered-by": "Robert Bunsen",
        "melt": 301.7,
        "molar-heat": 32.21,
        "named-by": _("Unknown"),
        "number": 55,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Cesium",
        "summary": "Caesium or cesium is a chemical element with symbol Cs and atomic number 55. It is a soft, silvery-gold alkali metal with a melting point of 28 \u00b0C (82 \u00b0F), which makes it one of only five elemental metals that are LIQUID at or near room temperature. Caesium is an alkali metal and has physical and chemical properties similar to those of rubidium and potassium.",
        "symbol": "Cs",
        "x": 1,
        "y": 6
    },
    Element.Ba: {
        "name": _("Barium"),
        "atomic-mass": 137.3277,
        "boil": 2118,
        "category": Category.ALKALINE_EARTH_METAL,
        "density": 3.51,
        "discovered-by": "Carl Wilhelm Scheele",
        "melt": 1000,
        "molar-heat": 28.07,
        "named-by": _("Unknown"),
        "number": 56,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Barium",
        "summary": "Barium is a chemical element with symbol Ba and atomic number 56. It is the fifth element in Group 2, a soft silvery metallic alkaline earth metal. Because of its high chemical reactivity barium is never found in nature as a free element.",
        "symbol": "Ba",
        "x": 2,
        "y": 6
    },
    Element.La: {
        "appearance": "silvery white",
        "name": _("Lanthanum"),
        "atomic-mass": 138.905477,
        "boil": 3737,
        "category": Category.LANTHANIDE,
        "density": 6.162,
        "discovered-by": "Carl Gustaf Mosander",
        "melt": 1193,
        "molar-heat": 27.11,
        "named-by": _("Unknown"),
        "number": 57,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Lanthanum",
        "summary": "Lanthanum is a soft, ductile, silvery-white metallic chemical element with symbol La and atomic number 57. It tarnishes rapidly when exed to air and is soft enough to be cut with a knife. It gave its name to the lanthanide series, a group of 15 similar elements between lanthanum and lutetium in the periodic table:it is also sometimes considered the first element of the 6th-period transition metals.",
        "symbol": "La",
        "x": 3,
        "y": 9
    },
    Element.Ce: {
        "appearance": "silvery white",
        "name": _("Cerium"),
        "atomic-mass": 140.1161,
        "boil": 3716,
        "category": Category.LANTHANIDE,
        "density": 6.77,
        "discovered-by": "Martin Heinrich Klaproth",
        "melt": 1068,
        "molar-heat": 26.94,
        "named-by": _("Unknown"),
        "number": 58,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Cerium",
        "summary": "Cerium is a chemical element with symbol Ce and atomic number 58. It is a soft, silvery, ductile metal which easily oxidizes in air. Cerium was named after the dwarf planet Ceres (itself named after the Roman goddess of agriculture).",
        "symbol": "Ce",
        "x": 4,
        "y": 9
    },
    Element.Pr: {
        "appearance": "grayish white",
        "name": _("Praseodymium"),
        "atomic-mass": 140.907662,
        "boil": 3403,
        "category": Category.LANTHANIDE,
        "density": 6.77,
        "discovered-by": "Carl Auer von Welsbach",
        "melt": 1208,
        "molar-heat": 27.2,
        "named-by": _("Unknown"),
        "number": 59,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Praseodymium",
        "summary": "Praseodymium is a chemical element with symbol Pr and atomic number 59. Praseodymium is a soft, silvery, malleable and ductile metal in the lanthanide group. It is valued for its magnetic, electrical, chemical, and optical properties.",
        "symbol": "Pr",
        "x": 5,
        "y": 9
    },
    Element.Nd: {
        "appearance": "silvery white",
        "name": _("Neodymium"),
        "atomic-mass": 144.2423,
        "boil": 3347,
        "category": Category.LANTHANIDE,
        "density": 7.01,
        "discovered-by": "Carl Auer von Welsbach",
        "melt": 1297,
        "molar-heat": 27.45,
        "named-by": _("Unknown"),
        "number": 60,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Neodymium",
        "summary": "Neodymium is a chemical element with symbol Nd and atomic number 60. It is a soft silvery metal that tarnishes in air. Neodymium was discovered in 1885 by the Austrian chemist Carl Auer von Welsbach.",
        "symbol": "Nd",
        "x": 6,
        "y": 9
    },
    Element.Pm: {
        "appearance": "metallic",
        "name": _("Promethium"),
        "atomic-mass": 145,
        "boil": 3273,
        "category": Category.LANTHANIDE,
        "density": 7.26,
        "discovered-by": "Chien Shiung Wu",
        "melt": 1315,
        "molar-heat": None,
        "named-by": "Isotopes of promethium",
        "number": 61,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Promethium",
        "summary": "Promethium, originally prometheum, is a chemical element with the symbol Pm and atomic number 61. All of its isotopes are radioactive; it is one of only two such elements that are followed in the periodic table by elements with stable forms, a distinction shared with technetium. Chemically, promethium is a lanthanide, which forms salts when combined with other elements.",
        "symbol": "Pm",
        "x": 7,
        "y": 9
    },
    Element.Sm: {
        "appearance": "silvery white",
        "name": _("Samarium"),
        "atomic-mass": 150.362,
        "boil": 2173,
        "category": Category.LANTHANIDE,
        "density": 7.52,
        "discovered-by": "Lecoq de Boisbaudran",
        "melt": 1345,
        "molar-heat": 29.54,
        "named-by": _("Unknown"),
        "number": 62,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Samarium",
        "summary": "Samarium is a chemical element with symbol Sm and atomic number 62. It is a moderately hard silvery metal that readily oxidizes in air. Being a typical member of the lanthanide series, samarium usually assumes the oxidation state +3.",
        "symbol": "Sm",
        "x": 8,
        "y": 9
    },
    Element.Eu: {
        "name": _("Europium"),
        "atomic-mass": 151.9641,
        "boil": 1802,
        "category": Category.LANTHANIDE,
        "density": 5.264,
        "discovered-by": "Eug\u00e8ne-Anatole Demar\u00e7ay",
        "melt": 1099,
        "molar-heat": 27.66,
        "named-by": _("Unknown"),
        "number": 63,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Europium",
        "summary": "Europium is a chemical element with symbol Eu and atomic number 63. It was isolated in 1901 and is named after the continent of Europe. It is a moderately hard, silvery metal which readily oxidizes in air and water.",
        "symbol": "Eu",
        "x": 9,
        "y": 9
    },
    Element.Gd: {
        "appearance": "silvery white",
        "name": _("Gadolinium"),
        "atomic-mass": 157.253,
        "boil": 3273,
        "category": Category.LANTHANIDE,
        "density": 7.9,
        "discovered-by": "Jean Charles Galissard de Marignac",
        "melt": 1585,
        "molar-heat": 37.03,
        "named-by": _("Unknown"),
        "number": 64,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Gadolinium",
        "summary": "Gadolinium is a chemical element with symbol Gd and atomic number 64. It is a silvery-white, malleable and ductile rare-earth metal. It is found in nature only in combined (salt) form.",
        "symbol": "Gd",
        "x": 10,
        "y": 9
    },
    Element.Tb: {
        "appearance": "silvery white",
        "name": _("Terbium"),
        "atomic-mass": 158.925352,
        "boil": 3396,
        "category": Category.LANTHANIDE,
        "density": 8.23,
        "discovered-by": "Carl Gustaf Mosander",
        "melt": 1629,
        "molar-heat": 28.91,
        "named-by": _("Unknown"),
        "number": 65,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Terbium",
        "summary": "Terbium is a chemical element with symbol Tb and atomic number 65. It is a silvery-white rare earth metal that is malleable, ductile and soft enough to be cut with a knife. Terbium is never found in nature as a free element, but it is contained in many minerals, including cerite, gadolinite, monazite, xenotime and euxenite.",
        "symbol": "Tb",
        "x": 11,
        "y": 9
    },
    Element.Dy: {
        "appearance": "silvery white",
        "name": _("Dysprosium"),
        "atomic-mass": 162.5001,
        "boil": 2840,
        "category": Category.LANTHANIDE,
        "density": 8.54,
        "discovered-by": "Lecoq de Boisbaudran",
        "melt": 1680,
        "molar-heat": 27.7,
        "named-by": _("Unknown"),
        "number": 66,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Dysprosium",
        "summary": "Dysprosium is a chemical element with the symbol Dy and atomic number 66. It is a rare earth element with a metallic silver luster. Dysprosium is never found in nature as a free element, though it is found in various minerals, such as xenotime.",
        "symbol": "Dy",
        "x": 12,
        "y": 9
    },
    Element.Ho: {
        "appearance": "silvery white",
        "name": _("Holmium"),
        "atomic-mass": 164.930332,
        "boil": 2873,
        "category": Category.LANTHANIDE,
        "density": 8.79,
        "discovered-by": "Marc Delafontaine",
        "melt": 1734,
        "molar-heat": 27.15,
        "named-by": _("Unknown"),
        "number": 67,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Holmium",
        "summary": "Holmium is a chemical element with symbol Ho and atomic number 67. Part of the lanthanide series, holmium is a rare earth element. Holmium was discovered by Swedish chemist Per Theodor Cleve.",
        "symbol": "Ho",
        "x": 13,
        "y": 9
    },
    Element.Er: {
        "appearance": "silvery white",
        "name": _("Erbium"),
        "atomic-mass": 167.2593,
        "boil": 3141,
        "category": Category.LANTHANIDE,
        "density": 9.066,
        "discovered-by": "Carl Gustaf Mosander",
        "melt": 1802,
        "molar-heat": 28.12,
        "named-by": _("Unknown"),
        "number": 68,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Erbium",
        "summary": "Erbium is a chemical element in the lanthanide series, with symbol Er and atomic number 68. A silvery-white solid metal when artificially isolated, natural erbium is always found in chemical combination with other elements on Earth. As such, it is a rare earth element which is associated with several other rare elements in the mineral gadolinite from Ytterby in Sweden, where yttrium, ytterbium, and terbium were discovered.",
        "symbol": "Er",
        "x": 14,
        "y": 9
    },
    Element.Tm: {
        "appearance": "silvery gray",
        "name": _("Thulium"),
        "atomic-mass": 168.934222,
        "boil": 2223,
        "category": Category.LANTHANIDE,
        "density": 9.32,
        "discovered-by": "Per Teodor Cleve",
        "melt": 1818,
        "molar-heat": 27.03,
        "named-by": _("Unknown"),
        "number": 69,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Thulium",
        "summary": "Thulium is a chemical element with symbol Tm and atomic number 69. It is the thirteenth and antepenultimate (third-last) element in the lanthanide series. Like the other lanthanides, the most common oxidation state is +3, seen in its oxide, halides and other compounds.",
        "symbol": "Tm",
        "x": 15,
        "y": 9
    },
    Element.Yb: {
        "name": _("Ytterbium"),
        "atomic-mass": 173.0451,
        "boil": 1469,
        "category": Category.LANTHANIDE,
        "density": 6.9,
        "discovered-by": "Jean Charles Galissard de Marignac",
        "melt": 1097,
        "molar-heat": 26.74,
        "named-by": _("Unknown"),
        "number": 70,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Ytterbium",
        "summary": "Ytterbium is a chemical element with symbol Yb and atomic number 70. It is the fourteenth and penultimate element in the lanthanide series, which is the basis of the relative stability of its +2 oxidation state. However, like the other lanthanides, its most common oxidation state is +3, seen in its oxide, halides and other compounds.",
        "symbol": "Yb",
        "x": 16,
        "y": 9
    },
    Element.Lu: {
        "appearance": "silvery white",
        "name": _("Lutetium"),
        "atomic-mass": 174.96681,
        "boil": 3675,
        "category": Category.LANTHANIDE,
        "density": 9.841,
        "discovered-by": "Georges Urbain",
        "melt": 1925,
        "molar-heat": 26.86,
        "named-by": _("Unknown"),
        "number": 71,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Lutetium",
        "summary": "Lutetium is a chemical element with symbol Lu and atomic number 71. It is a silvery white metal, which resists corrosion in dry, but not in moist air. It is considered the first element of the 6th-period transition metals and the last element in the lanthanide series, and is traditionally counted among the rare earths.",
        "symbol": "Lu",
        "x": 17,
        "y": 9
    },
    Element.Hf: {
        "appearance": "steel gray",
        "name": _("Hafnium"),
        "atomic-mass": 178.492,
        "boil": 4876,
        "category": Category.TRANSITION_METAL,
        "density": 13.31,
        "discovered-by": "Dirk Coster",
        "melt": 2506,
        "molar-heat": 25.73,
        "named-by": _("Unknown"),
        "number": 72,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Hafnium",
        "summary": "Hafnium is a chemical element with symbol Hf and atomic number 72. A lustrous, silvery gray, tetravalent transition metal, hafnium chemically resembles zirconium and is found in zirconium minerals. Its existence was predicted by Dmitri Mendeleev in 1869, though it was not identified until 1923, making it the penultimate stable element to be discovered (rhenium was identified two years later).",
        "symbol": "Hf",
        "x": 4,
        "y": 6
    },
    Element.Ta: {
        "appearance": "gray blue",
        "name": _("Tantalum"),
        "atomic-mass": 180.947882,
        "boil": 5731,
        "category": Category.TRANSITION_METAL,
        "density": 16.69,
        "discovered-by": "Anders Gustaf Ekeberg",
        "melt": 3290,
        "molar-heat": 25.36,
        "named-by": _("Unknown"),
        "number": 73,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Tantalum",
        "summary": "Tantalum is a chemical element with symbol Ta and atomic number 73. Previously known as tantalium, its name comes from Tantalus, an antihero from Greek mythology. Tantalum is a rare, hard, blue-gray, lustrous transition metal that is highly corrosion-resistant.",
        "symbol": "Ta",
        "x": 5,
        "y": 6
    },
    Element.W: {
        "appearance": "grayish white, lustrous",
        "name": _("Tungsten"),
        "atomic-mass": 183.841,
        "boil": 6203,
        "category": Category.TRANSITION_METAL,
        "density": 19.25,
        "discovered-by": "Carl Wilhelm Scheele",
        "melt": 3695,
        "molar-heat": 24.27,
        "named-by": _("Unknown"),
        "number": 74,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Tungsten",
        "summary": "Tungsten, also known as wolfram, is a chemical element with symbol W and atomic number 74. The word tungsten comes from the Swedish language tung sten, which directly translates to heavy stone. Its name in Swedish is volfram, however, in order to distinguish it from scheelite, which in Swedish is alternatively named tungsten.",
        "symbol": "W",
        "x": 6,
        "y": 6
    },
    Element.Re: {
        "appearance": "silvery-grayish",
        "name": _("Rhenium"),
        "atomic-mass": 186.2071,
        "boil": 5869,
        "category": Category.TRANSITION_METAL,
        "density": 21.02,
        "discovered-by": "Masataka Ogawa",
        "melt": 3459,
        "molar-heat": 25.48,
        "named-by": "Walter Noddack",
        "number": 75,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Rhenium",
        "summary": "Rhenium is a chemical element with symbol Re and atomic number 75. It is a silvery-white, heavy, third-row transition metal in group 7 of the periodic table. With an estimated average concentration of 1 part per billion (ppb), rhenium is one of the rarest elements in the Earth's crust.",
        "symbol": "Re",
        "x": 7,
        "y": 6
    },
    Element.Os: {
        "appearance": "silvery, blue cast",
        "name": _("Osmium"),
        "atomic-mass": 190.233,
        "boil": 5285,
        "category": Category.TRANSITION_METAL,
        "density": 22.59,
        "discovered-by": "Smithson Tennant",
        "melt": 3306,
        "molar-heat": 24.7,
        "named-by": _("Unknown"),
        "number": 76,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Osmium",
        "summary": "Osmium (from Greek osme (\u1f40\u03c3\u03bc\u03ae) meaning \"smell\") is a chemical element with symbol Os and atomic number 76. It is a hard, brittle, bluish-white transition metal in the platinum group that is found as a trace element in alloys, mostly in platinum ores. Osmium is the densest naturally occurring element, with a density of 7004225900000000000\u266022.59 g/cm3.",
        "symbol": "Os",
        "x": 8,
        "y": 6
    },
    Element.Ir: {
        "appearance": "silvery white",
        "name": _("Iridium"),
        "atomic-mass": 192.2173,
        "boil": 4403,
        "category": Category.TRANSITION_METAL,
        "density": 22.56,
        "discovered-by": "Smithson Tennant",
        "melt": 2719,
        "molar-heat": 25.1,
        "named-by": _("Unknown"),
        "number": 77,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Iridium",
        "summary": "Iridium is a chemical element with symbol Ir and atomic number 77. A very hard, brittle, silvery-white transition metal of the platinum group, iridium is generally credited with being the second densest element (after osmium) based on measured density, although calculations involving the space lattices of the elements show that iridium is denser. It is also the most corrosion-resistant metal, even at temperatures as high as 2000 \u00b0C. Although only certain molten salts and halogens are corrosive to solid iridium, finely divided iridium dust is much more reactive and can be flammable.",
        "symbol": "Ir",
        "x": 9,
        "y": 6
    },
    Element.Pt: {
        "appearance": "silvery white",
        "name": _("Platinum"),
        "atomic-mass": 195.0849,
        "boil": 4098,
        "category": Category.TRANSITION_METAL,
        "density": 21.45,
        "discovered-by": "Antonio de Ulloa",
        "melt": 2041.4,
        "molar-heat": 25.86,
        "named-by": _("Unknown"),
        "number": 78,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Platinum",
        "summary": "Platinum is a chemical element with symbol Pt and atomic number 78. It is a dense, malleable, ductile, highly unreactive, precious, gray-white transition metal. Its name is derived from the Spanish term platina, which is literally translated into \"little silver\".",
        "symbol": "Pt",
        "x": 10,
        "y": 6
    },
    Element.Au: {
        "appearance": "metallic yellow",
        "name": _("Gold"),
        "atomic-mass": 196.9665695,
        "boil": 3243,
        "category": Category.TRANSITION_METAL,
        "density": 19.3,
        "discovered-by": "Middle East",
        "melt": 1337.33,
        "molar-heat": 25.418,
        "named-by": _("Unknown"),
        "number": 79,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Gold",
        "summary": "Gold is a chemical element with symbol Au (from Latin:aurum) and atomic number 79. In its purest form, it is a bright, slightly reddish yellow, dense, soft, malleable and ductile metal. Chemically, gold is a transition metal and a group 11 element.",
        "symbol": "Au",
        "x": 11,
        "y": 6
    },
    Element.Hg: {
        "appearance": "silvery",
        "name": _("Mercury"),
        "atomic-mass": 200.5923,
        "boil": 629.88,
        "category": Category.TRANSITION_METAL,
        "density": 13.534,
        "discovered-by": "unknown, before 2000 BCE",
        "melt": 234.321,
        "molar-heat": 27.983,
        "named-by": _("Unknown"),
        "number": 80,
        "period": 6,
        "phase": Phase.LIQUID,
        "source": "https://en.wikipedia.org/wiki/Mercury_(element)",
        "summary": "Mercury is a chemical element with symbol Hg and atomic number 80. It is commonly known as quicksilver and was formerly named hydrargyrum (/ha\u026a\u02c8dr\u0251\u02d0rd\u0292\u0259r\u0259m/). A heavy, silvery d-block element, mercury is the only metallic element that is LIQUID at standard conditions for temperature and pressure; the only other element that is LIQUID under these conditions is bromine, though metals such as caesium, gallium, and rubidium melt just above room temperature.",
        "symbol": "Hg",
        "x": 12,
        "y": 6
    },
    Element.Tl: {
        "appearance": "silvery white",
        "name": _("Thallium"),
        "atomic-mass": 204.38,
        "boil": 1746,
        "category": Category.POST_TRANSITION_METAL,
        "density": 11.85,
        "discovered-by": "William Crookes",
        "melt": 577,
        "molar-heat": 26.32,
        "named-by": _("Unknown"),
        "number": 81,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Thallium",
        "summary": "Thallium is a chemical element with symbol Tl and atomic number 81. This soft gray post-transition metal is not found free in nature. When isolated, it resembles tin, but discolors when exed to air.",
        "symbol": "Tl",
        "x": 13,
        "y": 6
    },
    Element.Pb: {
        "appearance": "metallic gray",
        "name": _("Lead"),
        "atomic-mass": 207.21,
        "boil": 2022,
        "category": Category.POST_TRANSITION_METAL,
        "density": 11.34,
        "discovered-by": "Middle East",
        "melt": 600.61,
        "molar-heat": 26.65,
        "named-by": _("Unknown"),
        "number": 82,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Lead_(element)",
        "summary": "Lead (/l\u025bd/) is a chemical element in the carbon group with symbol Pb (from Latin:plumbum) and atomic number 82. Lead is a soft, malleable and heavy post-transition metal. Metallic lead has a bluish-white color after being freshly cut, but it soon tarnishes to a dull grayish color when exed to air.",
        "symbol": "Pb",
        "x": 14,
        "y": 6
    },
    Element.Bi: {
        "appearance": "lustrous silver",
        "name": _("Bismuth"),
        "atomic-mass": 208.980401,
        "boil": 1837,
        "category": Category.POST_TRANSITION_METAL,
        "density": 9.78,
        "discovered-by": "Claude Fran\u00e7ois Geoffroy",
        "melt": 544.7,
        "molar-heat": 25.52,
        "named-by": _("Unknown"),
        "number": 83,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Bismuth",
        "summary": "Bismuth is a chemical element with symbol Bi and atomic number 83. Bismuth, a pentavalent post-transition metal, chemically resembles arsenic and antimony. Elemental bismuth may occur naturally, although its sulfide and oxide form important commercial ores.",
        "symbol": "Bi",
        "x": 15,
        "y": 6
    },
    Element.Po: {
        "appearance": "silvery",
        "name": _("Polonium"),
        "atomic-mass": 209,
        "boil": 1235,
        "category": Category.METALLOID,
        "density": 9.196,
        "discovered-by": "Pierre Curie",
        "melt": 527,
        "molar-heat": 26.4,
        "named-by": _("Unknown"),
        "number": 84,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Polonium",
        "summary": "Polonium is a chemical element with symbol Po and atomic number 84, discovered in 1898 by Marie Curie and Pierre Curie. A rare and highly radioactive element with no stable isotopes, polonium is chemically similar to bismuth and tellurium, and it occurs in uranium ores. Applications of polonium are few.",
        "symbol": "Po",
        "x": 16,
        "y": 6
    },
    Element.At: {
        "appearance": "unknown, probably metallic",
        "name": _("Astatine"),
        "atomic-mass": 210,
        "boil": 610,
        "category": Category.DIATOMIC_NONMETAL,
        "density": 26.35,
        "discovered-by": "Dale R. Corson",
        "melt": 575,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 85,
        "period": 6,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Astatine",
        "summary": "Astatine is a very rare radioactive chemical element with the chemical symbol At and atomic number 85. It occurs on Earth as the decay product of various heavier elements. All its isotopes are short-lived; the most stable is astatine-210, with a half-life of 8.1 hours.",
        "symbol": "At",
        "x": 17,
        "y": 6
    },
    Element.Rn: {
        "appearance": "colorless gas, occasionally glows green or red in discharge tubes",
        "name": _("Radon"),
        "atomic-mass": 222,
        "boil": 211.5,
        "category": Category.NOBLE_GAS,
        "density": 9.73,
        "discovered-by": "Friedrich Ernst Dorn",
        "melt": 202,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 86,
        "period": 6,
        "phase": Phase.GAS,
        "source": "https://en.wikipedia.org/wiki/Radon",
        "summary": "Radon is a chemical element with symbol Rn and atomic number 86. It is a radioactive, colorless, odorless, tasteless noble gas, occurring naturally as a decay product of radium. Its most stable isotope, 222Rn, has a half-life of 3.8 days.",
        "symbol": "Rn",
        "x": 18,
        "y": 6
    },
    Element.Fr: {
        "name": _("Francium"),
        "atomic-mass": 223,
        "boil": 950,
        "category": Category.ALKALI_METAL,
        "density": 1.87,
        "discovered-by": "Marguerite Perey",
        "melt": 300,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 87,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Francium",
        "summary": "Francium is a chemical element with symbol Fr and atomic number 87. It used to be known as eka-caesium and actinium K. It is the second-least electronegative element, behind only caesium. Francium is a highly radioactive metal that decays into astatine, radium, and radon.",
        "symbol": "Fr",
        "x": 1,
        "y": 7
    },
    Element.Ra: {
        "appearance": "silvery white metallic",
        "name": _("Radium"),
        "atomic-mass": 226,
        "boil": 2010,
        "category": Category.ALKALINE_EARTH_METAL,
        "density": 5.5,
        "discovered-by": "Pierre Curie",
        "melt": 1233,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 88,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Radium",
        "summary": "Radium is a chemical element with symbol Ra and atomic number 88. It is the sixth element in group 2 of the periodic table, also known as the alkaline earth metals. Pure radium is almost colorless, but it readily combines with nitrogen (rather than oxygen) on exure to air, forming a black surface layer of radium nitride (Ra3N2).",
        "symbol": "Ra",
        "x": 2,
        "y": 7
    },
    Element.Ac: {
        "name": _("Actinium"),
        "atomic-mass": 227,
        "boil": 3500300,
        "category": Category.ACTINIDE,
        "density": 10,
        "discovered-by": "Friedrich Oskar Giesel",
        "melt": 1500,
        "molar-heat": 27.2,
        "named-by": _("Unknown"),
        "number": 89,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Actinium",
        "summary": "Actinium is a radioactive chemical element with symbol Ac (not to be confused with the abbreviation for an acetyl group) and atomic number 89, which was discovered in 1899. It was the first non-primordial radioactive element to be isolated. Polonium, radium and radon were observed before actinium, but they were not isolated until 1902.",
        "symbol": "Ac",
        "x": 3,
        "y": 10
    },
    Element.Th: {
        "appearance": "silvery, often with black tarnish",
        "name": _("Thorium"),
        "atomic-mass": 232.03774,
        "boil": 5061,
        "category": Category.ACTINIDE,
        "density": 11.724,
        "discovered-by": "J\u00f6ns Jakob Berzelius",
        "melt": 2023,
        "molar-heat": 26.23,
        "named-by": _("Unknown"),
        "number": 90,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Thorium",
        "summary": "Thorium is a chemical element with symbol Th and atomic number 90. A radioactive actinide metal, thorium is one of only two significantly radioactive elements that still occur naturally in large quantities as a primordial element (the other being uranium). It was discovered in 1828 by the Norwegian Reverend and amateur mineralogist Morten Thrane Esmark and identified by the Swedish chemist J\u00f6ns Jakob Berzelius, who named it after Thor, the Norse god of thunder.",
        "symbol": "Th",
        "x": 4,
        "y": 10
    },
    Element.Pa: {
        "appearance": "bright, silvery metallic luster",
        "name": _("Protactinium"),
        "atomic-mass": 231.035882,
        "boil": 4300,
        "category": Category.ACTINIDE,
        "density": 15.37,
        "discovered-by": "William Crookes",
        "melt": 1841,
        "molar-heat": None,
        "named-by": "Otto Hahn",
        "number": 91,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Protactinium",
        "summary": "Protactinium is a chemical element with symbol Pa and atomic number 91. It is a dense, silvery-gray metal which readily reacts with oxygen, water vapor and inorganic acids. It forms various chemical compounds where protactinium is usually present in the oxidation state +5, but can also assume +4 and even +2 or +3 states.",
        "symbol": "Pa",
        "x": 5,
        "y": 10
    },
    Element.U: {
        "name": _("Uranium"),
        "atomic-mass": 238.028913,
        "boil": 4404,
        "category": Category.ACTINIDE,
        "density": 19.1,
        "discovered-by": "Martin Heinrich Klaproth",
        "melt": 1405.3,
        "molar-heat": 27.665,
        "named-by": _("Unknown"),
        "number": 92,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Uranium",
        "summary": "Uranium is a chemical element with symbol U and atomic number 92. It is a silvery-white metal in the actinide series of the periodic table. A uranium atom has 92 protons and 92 electrons, of which 6 are valence electrons.",
        "symbol": "U",
        "x": 6,
        "y": 10
    },
    Element.Np: {
        "appearance": "silvery metallic",
        "name": _("Neptunium"),
        "atomic-mass": 237,
        "boil": 4447,
        "category": Category.ACTINIDE,
        "density": 20.45,
        "discovered-by": "Edwin McMillan",
        "melt": 9123,
        "molar-heat": 29.46,
        "named-by": _("Unknown"),
        "number": 93,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Neptunium",
        "summary": "Neptunium is a chemical element with symbol Np and atomic number 93. A radioactive actinide metal, neptunium is the first transuranic element. Its position in the periodic table just after uranium, named after the planet Uranus, led to it being named after Neptune, the next planet beyond Uranus.",
        "symbol": "Np",
        "x": 7,
        "y": 10
    },
    Element.Pu: {
        "appearance": "silvery white, tarnishing to dark gray in air",
        "name": _("Plutonium"),
        "atomic-mass": 244,
        "boil": 3505,
        "category": Category.ACTINIDE,
        "density": 19.816,
        "discovered-by": "Glenn T. Seaborg",
        "melt": 912.5,
        "molar-heat": 35.5,
        "named-by": _("Unknown"),
        "number": 94,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Plutonium",
        "summary": "Plutonium is a transuranic radioactive chemical element with symbol Pu and atomic number 94. It is an actinide metal of silvery-gray appearance that tarnishes when exed to air, and forms a dull coating when oxidized. The element normally exhibits six allotropes and four oxidation states.",
        "symbol": "Pu",
        "x": 8,
        "y": 10
    },
    Element.Am: {
        "appearance": "silvery white",
        "name": _("Americium"),
        "atomic-mass": 243,
        "boil": 2880,
        "category": Category.ACTINIDE,
        "density": 12,
        "discovered-by": "Glenn T. Seaborg",
        "melt": 1449,
        "molar-heat": 62.7,
        "named-by": _("Unknown"),
        "number": 95,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Americium",
        "summary": "Americium is a radioactive transuranic chemical element with symbol Am and atomic number 95. This member of the actinide series is located in the periodic table under the lanthanide element europium, and thus by analogy was named after the Americas. Americium was first produced in 1944 by the group of Glenn T.Seaborg from Berkeley, California, at the metallurgical laboratory of University of Chicago.",
        "symbol": "Am",
        "x": 9,
        "y": 10
    },
    Element.Cm: {
        "appearance": "silvery metallic, glows purple in the dark",
        "name": _("Curium"),
        "atomic-mass": 247,
        "boil": 3383,
        "category": Category.ACTINIDE,
        "density": 13.51,
        "discovered-by": "Glenn T. Seaborg",
        "melt": 1613,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 96,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Curium",
        "summary": "Curium is a transuranic radioactive chemical element with symbol Cm and atomic number 96. This element of the actinide series was named after Marie and Pierre Curie \u2013 both were known for their research on radioactivity. Curium was first intentionally produced and identified in July 1944 by the group of Glenn T. Seaborg at the University of California, Berkeley.",
        "symbol": "Cm",
        "x": 10,
        "y": 10
    },
    Element.Bk: {
        "appearance": "silvery",
        "name": _("Berkelium"),
        "atomic-mass": 247,
        "boil": 2900,
        "category": Category.ACTINIDE,
        "density": 14.78,
        "discovered-by": "Lawrence Berkeley National Laboratory",
        "melt": 1259,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 97,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Berkelium",
        "summary": "Berkelium is a transuranic radioactive chemical element with symbol Bk and atomic number 97. It is a member of the actinide and transuranium element series. It is named after the city of Berkeley, California, the location of the University of California Radiation Laboratory where it was discovered in December 1949.",
        "symbol": "Bk",
        "x": 11,
        "y": 10
    },
    Element.Cf: {
        "appearance": "silvery",
        "name": _("Californium"),
        "atomic-mass": 251,
        "boil": 1743,
        "category": Category.ACTINIDE,
        "density": 15.1,
        "discovered-by": "Lawrence Berkeley National Laboratory",
        "melt": 1173,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 98,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Californium",
        "summary": "Californium is a radioactive metallic chemical element with symbol Cf and atomic number 98. The element was first made in 1950 at the University of California Radiation Laboratory in Berkeley, by bombarding curium with alpha particles (helium-4 ions). It is an actinide element, the sixth transuranium element to be synthesized, and has the second-highest atomic mass of all the elements that have been produced in amounts large enough to see with the unaided eye (after einsteinium).",
        "symbol": "Cf",
        "x": 12,
        "y": 10
    },
    Element.Es: {
        "appearance": "silver-colored",
        "name": _("Einsteinium"),
        "atomic-mass": 252,
        "boil": 1269,
        "category": Category.ACTINIDE,
        "density": 8.84,
        "discovered-by": "Lawrence Berkeley National Laboratory",
        "melt": 1133,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 99,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Einsteinium",
        "summary": "Einsteinium is a synthetic element with symbol Es and atomic number 99. It is the seventh transuranic element, and an actinide. Einsteinium was discovered as a component of the debris of the first hydrogen bomb explosion in 1952, and named after Albert Einstein.",
        "symbol": "Es",
        "x": 13,
        "y": 10
    },
    Element.Fm: {
        "name": _("Fermium"),
        "atomic-mass": 257,
        "boil": None,
        "category": Category.ACTINIDE,
        "density": None,
        "discovered-by": "Lawrence Berkeley National Laboratory",
        "melt": 1800,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 100,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Fermium",
        "summary": "Fermium is a synthetic element with symbol Fm and atomic number 100. It is a member of the actinide series. It is the heaviest element that can be formed by neutron bombardment of lighter elements, and hence the last element that can be prepared in macroscopic quantities, although pure fermium metal has not yet been prepared.",
        "symbol": "Fm",
        "x": 14,
        "y": 10
    },
    Element.Md: {
        "name": _("Mendelevium"),
        "atomic-mass": 258,
        "boil": None,
        "category": Category.ACTINIDE,
        "density": None,
        "discovered-by": "Lawrence Berkeley National Laboratory",
        "melt": 1100,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 101,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Mendelevium",
        "summary": "Mendelevium is a synthetic element with chemical symbol Md (formerly Mv) and atomic number 101. A metallic radioactive transuranic element in the actinide series, it is the first element that currently cannot be produced in macroscopic quantities through neutron bombardment of lighter elements. It is the antepenultimate actinide and the ninth transuranic element.",
        "symbol": "Md",
        "x": 15,
        "y": 10
    },
    Element.No: {
        "name": _("Nobelium"),
        "atomic-mass": 259,
        "boil": None,
        "category": Category.ACTINIDE,
        "density": None,
        "discovered-by": "Joint Institute for Nuclear Research",
        "melt": 1100,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 102,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Nobelium",
        "summary": "Nobelium is a synthetic chemical element with symbol No and atomic number 102. It is named in honor of Alfred Nobel, the inventor of dynamite and benefactor of science. A radioactive metal, it is the tenth transuranic element and is the penultimate member of the actinide series.",
        "symbol": "No",
        "x": 16,
        "y": 10
    },
    Element.Lr: {
        "name": _("Lawrencium"),
        "atomic-mass": 266,
        "boil": None,
        "category": Category.ACTINIDE,
        "density": None,
        "discovered-by": "Lawrence Berkeley National Laboratory",
        "melt": 1900,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 103,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Lawrencium",
        "summary": "Lawrencium is a synthetic chemical element with chemical symbol Lr (formerly Lw) and atomic number 103. It is named in honor of Ernest Lawrence, inventor of the cyclotron, a device that was used to discover many artificial radioactive elements. A radioactive metal, lawrencium is the eleventh transuranic element and is also the final member of the actinide series.",
        "symbol": "Lr",
        "x": 17,
        "y": 10
    },
    Element.Rf: {
        "name": _("Rutherfordium"),
        "atomic-mass": 267,
        "boil": 5800,
        "category": Category.TRANSITION_METAL,
        "density": 23.2,
        "discovered-by": "Joint Institute for Nuclear Research",
        "melt": 2400,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 104,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Rutherfordium",
        "summary": "Rutherfordium is a chemical element with symbol Rf and atomic number 104, named in honor of physicist Ernest Rutherford. It is a synthetic element (an element that can be created in a laboratory but is not found in nature) and radioactive; the most stable known isotope, 267Rf, has a half-life of approximately 1.3 hours. In the periodic table of the elements, it is a d - block element and the second of the fourth - row transition elements.",
        "symbol": "Rf",
        "x": 4,
        "y": 7
    },
    Element.Db: {
        "name": _("Dubnium"),
        "atomic-mass": 268,
        "boil": None,
        "category": Category.TRANSITION_METAL,
        "density": 29.3,
        "discovered-by": "Joint Institute for Nuclear Research",
        "melt": None,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 105,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Dubnium",
        "summary": "Dubnium is a chemical element with symbol Db and atomic number 105. It is named after the town of Dubna in Russia (north of Moscow), where it was first produced. It is a synthetic element (an element that can be created in a laboratory but is not found in nature) and radioactive; the most stable known isotope, dubnium-268, has a half-life of approximately 28 hours.",
        "symbol": "Db",
        "x": 5,
        "y": 7
    },
    Element.Sg: {
        "name": _("Seaborgium"),
        "atomic-mass": 269,
        "boil": None,
        "category": Category.TRANSITION_METAL,
        "density": 35.0,
        "discovered-by": "Lawrence Berkeley National Laboratory",
        "melt": None,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 106,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Seaborgium",
        "summary": "Seaborgium is a synthetic element with symbol Sg and atomic number 106. Its most stable isotope 271Sg has a half-life of 1.9 minutes. A more recently discovered isotope 269Sg has a potentially slightly longer half-life (ca.",
        "symbol": "Sg",
        "x": 6,
        "y": 7
    },
    Element.Bh: {
        "name": _("Bohrium"),
        "atomic-mass": 270,
        "boil": None,
        "category": Category.TRANSITION_METAL,
        "density": 37.1,
        "discovered-by": "Gesellschaft f\u00fcr Schwerionenforschung",
        "melt": None,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 107,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Bohrium",
        "summary": "Bohrium is a chemical element with symbol Bh and atomic number 107. It is named after Danish physicist Niels Bohr. It is a synthetic element (an element that can be created in a laboratory but is not found in nature) and radioactive; the most stable known isotope, 270Bh, has a half-life of approximately 61 seconds.",
        "symbol": "Bh",
        "x": 7,
        "y": 7
    },
    Element.Hs: {
        "name": _("Hassium"),
        "atomic-mass": 269,
        "boil": None,
        "category": Category.TRANSITION_METAL,
        "density": 40.7,
        "discovered-by": "Gesellschaft f\u00fcr Schwerionenforschung",
        "melt": 126,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 108,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Hassium",
        "summary": "Hassium is a chemical element with symbol Hs and atomic number 108, named after the German state of Hesse. It is a synthetic element (an element that can be created in a laboratory but is not found in nature) and radioactive; the most stable known isotope, 269Hs, has a half-life of approximately 9.7 seconds, although an unconfirmed metastable state, 277mHs, may have a longer half-life of about 130 seconds. More than 100 atoms of hassium have been synthesized to date.",
        "symbol": "Hs",
        "x": 8,
        "y": 7
    },
    Element.Mt: {
        "name": _("Meitnerium"),
        "atomic-mass": 278,
        "boil": None,
        "category": Category.TRANSITION_METAL,
        "density": 37.4,
        "discovered-by": "Gesellschaft f\u00fcr Schwerionenforschung",
        "melt": None,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 109,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Meitnerium",
        "summary": "Meitnerium is a chemical element with symbol Mt and atomic number 109. It is an extremely radioactive synthetic element (an element not found in nature that can be created in a laboratory). The most stable known isotope, meitnerium-278, has a half-life of 7.6 seconds.",
        "symbol": "Mt",
        "x": 9,
        "y": 7
    },
    Element.Ds: {
        "name": _("Darmstadtium"),
        "atomic-mass": 281,
        "boil": None,
        "category": Category.TRANSITION_METAL,
        "density": 34.8,
        "discovered-by": "Gesellschaft f\u00fcr Schwerionenforschung",
        "melt": None,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 110,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Darmstadtium",
        "summary": "Darmstadtium is a chemical element with symbol Ds and atomic number 110. It is an extremely radioactive synthetic element. The most stable known isotope, darmstadtium-281, has a half-life of approximately 10 seconds.",
        "symbol": "Ds",
        "x": 10,
        "y": 7
    },
    Element.Rg: {
        "name": _("Roentgenium"),
        "atomic-mass": 282,
        "boil": None,
        "category": Category.TRANSITION_METAL,
        "density": 28.7,
        "discovered-by": "Gesellschaft f\u00fcr Schwerionenforschung",
        "melt": None,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 111,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Roentgenium",
        "summary": "Roentgenium is a chemical element with symbol Rg and atomic number 111. It is an extremely radioactive synthetic element (an element that can be created in a laboratory but is not found in nature); the most stable known isotope, roentgenium-282, has a half-life of 2.1 minutes. Roentgenium was first created in 1994 by the GSI Helmholtz Centre for Heavy Ion Research near Darmstadt, Germany.",
        "symbol": "Rg",
        "x": 11,
        "y": 7
    },
    Element.Cn: {
        "name": _("Copernicium"),
        "atomic-mass": 285,
        "boil": 3570,
        "category": Category.TRANSITION_METAL,
        "density": 23.7,
        "discovered-by": "Gesellschaft f\u00fcr Schwerionenforschung",
        "melt": None,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 112,
        "period": 7,
        "phase": "Boiling point",
        "source": "https://en.wikipedia.org/wiki/Copernicium",
        "summary": "Copernicium is a chemical element with symbol Cn and atomic number 112. It is an extremely radioactive synthetic element that can only be created in a laboratory. The most stable known isotope, copernicium-285, has a half-life of approximately 29 seconds, but it is possible that this copernicium isotope may have a nuclear isomer with a longer half-life, 8.9 min.",
        "symbol": "Cn",
        "x": 12,
        "y": 7
    },
    Element.Uut: {
        "name": _("Nihonium"),
        "atomic-mass": 286,
        "boil": 1430,
        "category": Category.SYNTHETIC,
        "density": 16,
        "discovered-by": "RIKEN",
        "melt": 700,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 113,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Ununtrium",
        "summary": "Nihonium is a chemical element with atomic number 113. It has a symbol Nh. It is a synthetic element (an element that can be created in a laboratory but is not found in nature) and is extremely radioactive; its most stable known isotope, nihonium-286, has a half-life of 20 seconds.",
        "symbol": "Nh",
        "x": 13,
        "y": 7
    },
    Element.Fl: {
        "name": _("Flerovium"),
        "atomic-mass": 289,
        "boil": 420,
        "category": Category.UNKNOWN,
        "density": 14,
        "discovered-by": "Joint Institute for Nuclear Research",
        "melt": 340,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 114,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Flerovium",
        "summary": "Flerovium is a superheavy artificial chemical element with symbol Fl and atomic number 114. It is an extremely radioactive synthetic element. The element is named after the Flerov Laboratory of Nuclear Reactions of the Joint Institute for Nuclear Research in Dubna, Russia, where the element was discovered in 1998.",
        "symbol": "Fl",
        "x": 14,
        "y": 7
    },
    Element.Uup: {
        "name": _("Moscovium"),
        "atomic-mass": 289,
        "boil": 1400,
        "category": Category.SYNTHETIC,
        "density": 13.5,
        "discovered-by": "Joint Institute for Nuclear Research",
        "melt": 670,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 115,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Ununpentium",
        "summary": "Moscovium is the name of a synthetic superheavy element in the periodic table that has the symbol Mc and has the atomic number 115. It is an extremely radioactive element; its most stable known isotope, moscovium-289, has a half-life of only 220 milliseconds. It is also known as eka-bismuth or simply element 115.",
        "symbol": "Mc",
        "x": 15,
        "y": 7
    },
    Element.Lv: {
        "name": _("Livermorium"),
        "atomic-mass": 293,
        "boil": 10351135,
        "category": Category.UNKNOWN,
        "density": 12.9,
        "discovered-by": "Joint Institute for Nuclear Research",
        "melt": 637780,
        "molar-heat": None,
        "named-by": _("Unknown"),
        "number": 116,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Livermorium",
        "summary": "Livermorium is a synthetic superheavy element with symbol Lv and atomic number 116. It is an extremely radioactive element that has only been created in the laboratory and has not been observed in nature. The element is named after the Lawrence Livermore National Laboratory in the United States, which collaborated with the Joint Institute for Nuclear Research in Dubna, Russia to discover livermorium in 2000.",
        "symbol": "Lv",
        "x": 16,
        "y": 7
    },
    Element.Uus: {
        "name": _("Tennessine"),
        "atomic-mass": 294,
        "boil": 883,
        "category": Category.SYNTHETIC,
        "density": 7.17,
        "discovered-by": "Joint Institute for Nuclear Research",
        "melt": 623823,
        "molar-heat": None,
        "named-by": _("Joseph Hamilton"),
        "number": 117,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Ununseptium",
        "summary": "Tennessine is a superheavy artificial chemical element with an atomic number of 117 and a symbol of Ts. Also known as eka-astatine or element 117, it is the second-heaviest known element and penultimate element of the 7th period of the periodic table. As of 2016, fifteen tennessine atoms have been observed:six when it was first synthesized in 2010, seven in 2012, and two in 2014.",
        "symbol": "Ts",
        "x": 17,
        "y": 7
    },
    Element.Uuo: {
        "name": _("Oganesson"),
        "atomic-mass": 294,
        "boil": 35030,
        "category": Category.SYNTHETIC,
        "density": 4.95,
        "discovered-by": "Joint Institute for Nuclear Research",
        "melt": None,
        "molar-heat": None,
        "named-by": _("Yuri Oganessian"),
        "number": 118,
        "period": 7,
        "phase": Phase.SOLID,
        "source": "https://en.wikipedia.org/wiki/Ununoctium",
        "summary": "Ooganesson is IUPAC's name for the transactinide element with the atomic number 118 and element symbol Og. It is also known as eka-radon or element 118, and on the periodic table of the elements it is a p-block element and the last one of the 7th period. Oganesson is currently the only synthetic member of group 18.",
        "symbol": "Og",
        "x": 18,
        "y": 7
    }
}
